package `in`.navanatech.zabaan.publishersdk.providers

import `in`.navanatech.zabaan.publishersdk.data.network.PublisherService
import `in`.navanatech.zabaan.publishersdk.model.ViewNodeJsonAdapter
import `in`.navanatech.zabaan.publishersdk.util.Logger
import com.skydoves.sandwich.coroutines.CoroutinesResponseCallAdapterFactory
import com.squareup.moshi.Moshi
import com.squareup.moshi.addAdapter
import com.squareup.moshi.kotlin.reflect.KotlinJsonAdapterFactory
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.moshi.MoshiConverterFactory
import java.util.concurrent.TimeUnit

/**
 * Network providers
 * Maintains instance of all the network related object required throughout the lifecycle of the app
 *
 */
object NetworkProviders {

    @OptIn(ExperimentalStdlibApi::class)
    private fun provideMoshi(): Moshi {
        return Moshi.Builder()
            .addAdapter(ViewNodeJsonAdapter(Moshi.Builder().build())) // TODO: Cleanup
            .add(KotlinJsonAdapterFactory())
            .build()
    }

    private fun providesLoggingInterceptor(): HttpLoggingInterceptor {
        val loggingInterceptor = HttpLoggingInterceptor()
        if (Logger.LOGGING_ENABLED) {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY)
        } else {
            loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.NONE)
        }
        return loggingInterceptor
    }

    private fun providesBaseHttpClientBuilder(loggingInterceptor: HttpLoggingInterceptor) =
        OkHttpClient.Builder().apply {
            readTimeout(15, TimeUnit.SECONDS)
            connectTimeout(15, TimeUnit.SECONDS)
            addInterceptor(loggingInterceptor)
        }.build()

    private fun createService(baseUrl: String): Retrofit {
        return Retrofit.Builder()
            .client(providesBaseHttpClientBuilder(providesLoggingInterceptor()))
            .baseUrl(baseUrl)
            .addCallAdapterFactory(CoroutinesResponseCallAdapterFactory.create())
            .addConverterFactory(MoshiConverterFactory.create(provideMoshi()).asLenient())
            .build()
    }

    fun provideService(): PublisherService {
        return createService("https://zabaan-cms-backend.navanatech.in/").create(PublisherService::class.java) // TODO: Move this to NDK if need be
    }
}
