package `in`.navanatech.zabaan.publishersdk.ui

import `in`.navanatech.zabaan.publishersdk.R
import `in`.navanatech.zabaan.publishersdk.util.className
import `in`.navanatech.zabaan.publishersdk.util.e
import `in`.navanatech.zabaan.publishersdk.util.rootView
import android.annotation.SuppressLint
import android.app.Activity
import android.view.HapticFeedbackConstants
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import androidx.cardview.widget.CardView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.isVisible

/**
 * Bubble manager
 * Class to inflate bubble view in UI
 *
 * Notes
 * 1. It locally manages the state if bubble is to be shown or not
 * 2. Doesn't keep activity instance to avoid leaks
 * 3. Click on bubble is handled based on time since there aren't any other reliable method.
 *
 * @constructor Create empty Bubble manager
 */
class BubbleManager() {

    // Variable to keep track if bubble is dismissed or not
    private var show: Boolean = true
    private var rootView: ViewGroup? = null

    private var showParams: Pair<Activity, () -> Unit>? = null

    fun show(activity: Activity, onClick: () -> Unit) {
        clear()
        showParams = Pair(activity, onClick)
        if (show) {
            val activityRootView = activity.rootView
            if (activityRootView !is ViewGroup) {
                error("Root view is not View Group")
            }
            rootView = activityRootView
            activity.layoutInflater.inflate(R.layout.bubble_overlay, activityRootView)
            touchEventHandler(activity, onClick)
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun touchEventHandler(activity: Activity, onClick: () -> Unit) {
        e {"Touch event handler ${activity.className}"}
        val bubbleOverlay: ConstraintLayout by lazy { activity.findViewById(R.id.zzz_bubble_overlay) }
        val bubble: CardView by lazy { activity.findViewById(R.id.zzz_bubble) }
        val deleteBox: ImageView by lazy { activity.findViewById(R.id.zzz_delete_box) }
        var dX = 0f
        var dY = 0f

        bubble.setOnTouchListener { view, event -> // TODO: Convert this to drag listener and click listener
            e {"Touch listener"}
            val duration = event.eventTime - event.downTime
            if (event.action == MotionEvent.ACTION_UP && duration < 200) {
                // Click detection
                bubbleOverlay.visibility = View.GONE
                deleteBox.isVisible = false
                onClick()
                bubbleOverlay.isVisible = true
                return@setOnTouchListener true
            }

            // Movement detection
            when (event.action) {
                MotionEvent.ACTION_DOWN -> {
                    dX = view.x - event.rawX
                    dY = view.y - event.rawY
                    deleteBox.isVisible = true
                }
                MotionEvent.ACTION_MOVE -> {
                    view.animate()
                        .x(event.rawX + dX)
                        .y(event.rawY + dY)
                        .setDuration(0)
                        .start()
                }
                MotionEvent.ACTION_UP -> {
                    // Check if the drag is for delete action
                    if (view.y in deleteBox.y..deleteBox.y.plus(deleteBox.height)) {
                        view.performHapticFeedback(HapticFeedbackConstants.LONG_PRESS)
                        bubble.isVisible = false
                        show = false
                        rootView?.let { viewGroup ->
                            viewGroup.removeView(viewGroup.findViewById(R.id.zzz_bubble_overlay))
                        }
                    }

                    // Which half the bubble is being dragged to
                    if (view.x < activity.rootView.width / 2)
                        view.animate().x(0f).y(view.y).setDuration(0).start()
                    else
                        view.animate().x(activity.rootView.width.toFloat() - view.width)
                            .y(view.y)
                            .setDuration(0).start()

                    deleteBox.isVisible = false
                }
            }
            true
        }
    }

    fun clear() {
        rootView?.let { viewGroup ->
            viewGroup.removeView(viewGroup.findViewById(R.id.zzz_bubble_overlay))
        }
        showParams = null
    }

    fun forceShow() {
        if (!show) {
            show = true
            showParams?.let {
                show(it.first, it.second)
            }
        }
    }
}