package `in`.navanatech.zabaan.publishersdk.ui

import `in`.navanatech.zabaan.publishersdk.R
import `in`.navanatech.zabaan.publishersdk.data.PublisherRepository
import `in`.navanatech.zabaan.publishersdk.providers.RepositoryProvider
import android.Manifest
import android.os.Bundle
import android.widget.ImageView
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.core.widget.doOnTextChanged
import androidx.lifecycle.lifecycleScope
import com.chaos.view.PinView
import com.journeyapps.barcodescanner.ScanContract
import com.journeyapps.barcodescanner.ScanIntentResult
import com.journeyapps.barcodescanner.ScanOptions
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * An activity that allows users to scan QR code or enter the pin code manually.
 * This activity is launched when the user clicks the publisher sdk notification.
 */
class CodeScannerActivity : AppCompatActivity() {

    private lateinit var qrCodeImageView: ImageView
    private lateinit var secretCodePinView: PinView

    // TODO: Connect API calls
    private val repo: PublisherRepository = RepositoryProvider.provideRepository()

    private val scannerLauncher =
        registerForActivityResult(ScanContract()) { result: ScanIntentResult ->
            if (result.contents == null) {
                Toast.makeText(this@CodeScannerActivity, "Cancelled", Toast.LENGTH_LONG).show()
            } else {
                Toast.makeText(
                    this@CodeScannerActivity,
                    "Scanned: " + result.contents,
                    Toast.LENGTH_LONG
                ).show()
                sendToken(result.contents)
            }
        }

    private val permissionsLauncher =
        registerForActivityResult(ActivityResultContracts.RequestPermission()) { isGranted ->
            if (isGranted) {
                val scanOptions = ScanOptions().apply {
                    this.setOrientationLocked(false)
                    this.setBeepEnabled(false)
                    this.setDesiredBarcodeFormats(ScanOptions.QR_CODE)
                    this.setPrompt("Scan the QR code that is visible on the Zabaan CMS")
                }
                scannerLauncher.launch(scanOptions)
            } else {
                Toast.makeText(
                    this@CodeScannerActivity,
                    "Please grant camera permission to use QR code functionality",
                    Toast.LENGTH_LONG
                ).show()
            }
        }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_code_scanner)
        findViews()
        setupListeners()
    }

    private fun findViews() {
        qrCodeImageView = findViewById(R.id.zzz_qrCodeImageView)
        secretCodePinView = findViewById(R.id.zzz_secretCodePinView)
    }

    private fun setupListeners() {
        qrCodeImageView.setOnClickListener { checkPermissions() }

        secretCodePinView.doOnTextChanged { text, _, _, _ ->
            if (text?.length == 6)
                sendToken(text.toString())
        }
    }

    private fun checkPermissions() {
        permissionsLauncher.launch(Manifest.permission.CAMERA)
    }

    private fun sendToken(code: String) {
        lifecycleScope.launch {
            BlockingOverlay.show(this@CodeScannerActivity)
            repo.getToken(code) { success ->
                withContext(Dispatchers.Main) {
                    BlockingOverlay.hide(this@CodeScannerActivity)

                    if (success)
                        finish()
                    else
                        Toast.makeText(
                            this@CodeScannerActivity,
                            "Error connecting to CMS.",
                            Toast.LENGTH_SHORT
                        ).show()
                }
            }
        }
    }

}