package `in`.navanatech.zabaan.publishersdk.util

import `in`.navanatech.zabaan.publishersdk.R
import `in`.navanatech.zabaan.publishersdk.ui.CodeScannerActivity
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.content.getSystemService

/**
 * A helper singleton which allows creating, showing and cancelling notifications.
 */
object NotificationHelper {
    private const val CHANNEL_ID = "Zabaan_Publisher_SDK_Channel_Id"
    private const val CHANNEL_NAME = "Zabaan Publisher SDK"
    private const val CHANNEL_DESCRIPTION =
        "Notification channel to show notifications from Zabaan Publisher SDK"
    private const val NOTIFICATION_ID = 1010101010

    // This is less than ideal and we should look into it later
    private var isNotificationVisible = false

    private fun createNotificationChannel(context: Context) {
        // Create the NotificationChannel, but only on API 26+ because
        // the NotificationChannel class is new and not in the support library
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val name = CHANNEL_NAME
            val descriptionText = CHANNEL_DESCRIPTION
            val importance = NotificationManager.IMPORTANCE_DEFAULT
            val channel = NotificationChannel(CHANNEL_ID, name, importance).apply {
                description = descriptionText
            }
            // Register the channel with the system
            val notificationManager: NotificationManager? = context.getSystemService()
            notificationManager?.createNotificationChannel(channel)
        }
    }

    private fun createNotification(context: Context): NotificationCompat.Builder {
        val intent = Intent(context, CodeScannerActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }

        var pendingIntentFlag = PendingIntent.FLAG_UPDATE_CURRENT
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M)
            pendingIntentFlag = pendingIntentFlag or PendingIntent.FLAG_IMMUTABLE

        val pendingIntent = PendingIntent.getActivity(context, 0, intent, pendingIntentFlag)

        return NotificationCompat.Builder(context, CHANNEL_ID)
            .setOngoing(true)
            .setSmallIcon(R.drawable.zabaan_logo)
            .setContentTitle("Connect To Zabaan CMS")
            .setContentText("Tap the notification to connect your app to Zabaan CMS")
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
            .setContentIntent(pendingIntent)
    }

    private fun createErrorNotification(context: Context): NotificationCompat.Builder {
        val intent = Intent(context, CodeScannerActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }

        var pendingIntentFlag = PendingIntent.FLAG_UPDATE_CURRENT
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M)
            pendingIntentFlag = pendingIntentFlag or PendingIntent.FLAG_IMMUTABLE

        return NotificationCompat.Builder(context, CHANNEL_ID)
            .setOngoing(false)
            .setSmallIcon(R.drawable.zabaan_logo)
            .setContentTitle("Zabaan SDK not found")
            .setContentText("To use Zabaan Publisher SDK please ensure Zabaan SDK is added to the app.")
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
    }

    private fun createConnectedNotification(context: Context): NotificationCompat.Builder {
        val intent = Intent(context, CodeScannerActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }

        var pendingIntentFlag = PendingIntent.FLAG_UPDATE_CURRENT
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M)
            pendingIntentFlag = pendingIntentFlag or PendingIntent.FLAG_IMMUTABLE

        return NotificationCompat.Builder(context, CHANNEL_ID)
            .setOngoing(true)
            .setSmallIcon(R.drawable.zabaan_logo)
            .setContentTitle("Connected to CMS")
            .setContentText("You are connected to the project, press the \"Z\" icon to capture the screen")
            .addAction(
                android.R.drawable.ic_menu_camera,
                "Enable screenshot bubble",
                PendingIntent.getBroadcast(
                    context,
                    1,
                    Intent("in.navanatech.zabaan.publisher.ENABLE_BUBBLE"),
                    PendingIntent.FLAG_IMMUTABLE
                )
            )
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
    }

    fun showNotification(context: Context, type: NotificationType) {
        createNotificationChannel(context)
        val builder = when (type) {
            NotificationType.START -> createNotification(context)
            NotificationType.ERROR -> createErrorNotification(context)
            NotificationType.CONNECTED -> createConnectedNotification(context)
        }
        with(NotificationManagerCompat.from(context)) {
            // notificationId is a unique int for each notification that you must define
            notify(NOTIFICATION_ID, builder.build())
        }
        isNotificationVisible = true
    }


    fun updateNotification(context: Context) {
        if (!isNotificationVisible) return

        val builder = createNotification(context)
        builder.setContentTitle("Zabaan CMS Connected")
        builder.setContentText("Browse your app and capture the flows that you would like to configure over the CMS")
        with(NotificationManagerCompat.from(context)) {
            // notificationId is a unique int for each notification that you must define
            notify(NOTIFICATION_ID, builder.build())
        }
    }

    fun cancelNotification(context: Context) {
        NotificationManagerCompat.from(context).cancel(NOTIFICATION_ID)
        isNotificationVisible = false
    }
}

enum class NotificationType {
    START, ERROR, CONNECTED
}
