package com.payu.upiboltcore

import android.content.Context
import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upiboltcore.network.CommonHeaders
import com.payu.upiboltcore.network.UpiCoreApiService
import com.payu.upiboltcore.network.NPCIRepository
import com.payu.upiboltcore.npci.AppUPIRegistrationManager
import com.payu.upiboltcore.npci.DeviceInfoManager
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.interfaces.ManagementService
import com.payu.upiboltcore.interfaces.RegistrationService
import com.payu.upiboltcore.interfaces.TransactionService
import com.payu.upiboltcore.models.AccountInfo

import com.payu.upiboltcore.models.PayUPluginInitParams
import com.payu.upiboltcore.network.AuthTokenRepository
import com.payu.upiboltcore.npci.CLCredentialManager
import com.payu.upiboltcore.npci.ListKeysPayloadManager
import com.payu.upiboltcore.utils.AnalyticsUtils
import com.payu.upiboltcore.utils.PayUSPUtils
import com.payu.upiboltcore.utils.Utils
import org.npci.upi.security.services.CLServices
import org.npci.upi.security.services.ServiceConnectionStatusNotifier
import java.lang.ref.WeakReference
import java.util.UUID


object PayUUPIPlugin {

    private lateinit var clServicesRef: WeakReference<CLServices>
    private val commonHeaders = CommonHeaders()
    private val upiCoreApiService = UpiCoreApiService()
    private val npciRepository = NPCIRepository(commonHeaders, upiCoreApiService)
    private lateinit var credentialManager: CLCredentialManager
    private var activityRef: WeakReference<FragmentActivity>? = null

    private lateinit var registrationService: RegistrationService
    private lateinit var managementService: ManagementService
    private lateinit var transactionService: TransactionService

    fun initPayUSDK(
        activity: FragmentActivity,
        initParams: PayUPluginInitParams,
        authToken: String,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        this.activityRef = WeakReference(activity)
        InternalConfig.sdkInitParams = initParams
        InternalConfig.initSdkId = initParams.initTxnId
        InternalConfig.issuingBanks = initParams.bankIds.map { it.pluginName }
        DeviceInfoManager.createDeviceInfo(activity)
        setAuthToken(authToken)
        initCLServices(payUUPIProCallBack) { clServices ->
            val appUPIRegistrationManager =
                AppUPIRegistrationManager(clServices, npciRepository)
            activityRef?.get()?.let {
                credentialManager = CLCredentialManager(
                    it, clServices,
                    appUPIRegistrationManager,
                    ListKeysPayloadManager(npciRepository, clServices)
                )
            }
            registrationService =
                PluginServiceLocator.getRegistrationService(
                    activity, appUPIRegistrationManager
                )
            managementService =
                PluginServiceLocator.getManagementService(activity, credentialManager)
            transactionService =
                PluginServiceLocator.getTransactionService(activity, credentialManager)
            registrationService.checkDeviceStatus(
                false, initParams.otpVerificationInterface, payUUPIProCallBack
            )
        }
    }

    fun setAuthToken(authToken: String) {
        AuthTokenRepository.authToken = authToken
    }

    private fun initCLServices(
        payUUPIProCallBack: PayUUPIBoltCallBack,
        getCLServices: (clServices: CLServices) -> Unit
    ) {
        val fnStartTime = System.currentTimeMillis()
        val referenceId = UUID.randomUUID().toString()
        if (::clServicesRef.isInitialized.not()) {
            CLServices.initService(
                activityRef?.get(), object : ServiceConnectionStatusNotifier {
                    override fun serviceConnected(clServices: CLServices) {
                        clServicesRef = WeakReference(clServices)
                        getCLServices.invoke(clServices)
                    }

                    override fun serviceDisconnected() {
                        activityRef?.get()?.baseContext?.let {
                            AnalyticsUtils.logKibana(
                                it,
                                PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                                PayUResponseMessages.PAYU_CL_SERVICE_DISCONNECTED,
                                true, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                                time = Utils.getTimeDifferenceInMilliSeconds(fnStartTime),
                                referenceId = referenceId,
                                code = PayUResponseCodes.PAYU_HANDSHAKE_FAILED.toString(),
                                status = CLConstant.PAYU_EVENT_FAILURE
                            )
                        }
                        payUUPIProCallBack.onPayUFailure(
                            PayUUPIBoltResponse(
                                PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                                PayUResponseCodes.PAYU_HANDSHAKE_FAILED,
                                PayUResponseMessages.PAYU_CL_SERVICE_DISCONNECTED
                            )
                        )
                    }
                })
        } else {
            getCLServices.invoke(clServicesRef.get()!!)
        }
    }

    fun getBankList(payuApiCallBack: PayUUPIBoltCallBack) {
        registrationService.getBankList(payuApiCallBack)
    }

    fun getAccountsList(
        bankCode: String? = null,
        requestType: String? = null,
        iin: String? = null,
        bankId: String? = null,
        accountType: String? = null,
        payuApiCallBack: PayUUPIBoltCallBack
    ) {
        registrationService.getAccountsList(
            bankCode, requestType, iin, bankId, accountType, payuApiCallBack
        )
    }

    fun setMpin(
        account: AccountInfo, cardNo: String, exp: String, payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        managementService.setMpin(account, cardNo, exp, payUUpiProCallBack)
    }

    fun changeMpin(account: AccountInfo, callback: PayUUPIBoltCallBack) {
        managementService.changeMpin(account, callback)
    }

    fun checkBalance(account: AccountInfo, callback: PayUUPIBoltCallBack) {
        managementService.checkBalance(account, callback)
    }

    fun raiseDispute(
        txnId: String, refId: String?, query: String, queryType: String?,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        managementService.raiseDispute(txnId, refId, query, queryType, payUUpiProCallBack)
    }

    fun addAccount(
        accountInfo: AccountInfo,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        managementService.addAccount(accountInfo, payUUpiProCallBack)
    }

    fun registerVPA(
        vpa: String, accountInfo: AccountInfo,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        registrationService.registerVPA(vpa, accountInfo, payUUpiProCallBack)
    }

    fun getProfile(payUUpiProCallBack: PayUUPIBoltCallBack) {
        transactionService.getProfile(payUUpiProCallBack)
    }

    fun getDisputeList(payUUpiProCallBack: PayUUPIBoltCallBack) {
        managementService.getDisputeList(payUUpiProCallBack)
    }

    fun removeAccount(account: AccountInfo, payUUpiProCallBack: PayUUPIBoltCallBack) {
        managementService.removeAccount(account, payUUpiProCallBack)
    }

    fun deregisterVpa(payUUpiProCallBack: PayUUPIBoltCallBack) {
        managementService.deregisterVpa(payUUpiProCallBack)
    }

    fun makePayment(
        amount: String, account: AccountInfo, payeeName: String, payeeVpa: String,
        note: String? = null, txnId: String, mcc: String? = null, merchantId: String? = null, initMode: String,
        purpose: String, refUrl: String? = null, payeeAccountNo: String? = null,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        transactionService.makePayment(amount, account, payeeName, payeeVpa, note, txnId, mcc,
            merchantId, initMode, purpose, refUrl, payeeAccountNo, payUUpiProCallBack, false,
            InternalConfig.sdkInitParams?.apiFailureCallback)
    }

    fun getTransactionHistoryList(
        startDate: String,
        endDate: String,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        managementService.getTransactionHistoryList(startDate, endDate, payUUpiProCallBack)
    }

    fun getDisputeTypeList(upiTxnRefNo: String, payUUpiProCallBack: PayUUPIBoltCallBack) {
        managementService.getDisputeTypeList(upiTxnRefNo, payUUpiProCallBack)
    }

    fun checkDisputeStatus(txnId: String, payUUpiProCallBack: PayUUPIBoltCallBack) {
        managementService.checkDisputeStatus(txnId, payUUpiProCallBack)
    }

    fun getRegisteredMobileNumber(context: Context): String? {
        return PayUSPUtils.getStringFromSP(context, UPIConstants.REGISTERED_MOBILE)
    }

    fun clearCache() {
        InternalConfig.clearAll()
    }

    fun clearData() {
        clearCache()
        activityRef?.get()?.baseContext?.let { PayUSPUtils.clearSharedPreferences(it) }
    }
}