package com.payu.upiboltcore.features.management

import com.payu.commonmodelssdk.constants.ApiConstant
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseCodes.PAYU_FAILED_STATUS
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.PayUAsyncTaskResponse
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.models.AccountInfo
import com.payu.upiboltcore.models.AddAccountRequest
import com.payu.upiboltcore.models.AddAccountResponse
import com.payu.upiboltcore.models.BalanceEnquiryRequest
import com.payu.upiboltcore.models.BalanceEnquiryResponse
import com.payu.upiboltcore.models.ChangeMPinRequest
import com.payu.upiboltcore.models.ChangeMPinResponse
import com.payu.upiboltcore.models.CheckTransactionStatusRequest
import com.payu.upiboltcore.models.CheckTransactionStatusResponse
import com.payu.upiboltcore.models.ComplaintInfo
import com.payu.upiboltcore.models.Credential
import com.payu.upiboltcore.models.DeRegisterVPARequest
import com.payu.upiboltcore.models.DeRegisterVPAResponse
import com.payu.upiboltcore.models.DisputeListResponse
import com.payu.upiboltcore.models.DisputeTypeListRequest
import com.payu.upiboltcore.models.DisputeTypeListResponse
import com.payu.upiboltcore.models.GetDisputeListRequest
import com.payu.upiboltcore.models.InitiateMPinRequest
import com.payu.upiboltcore.models.InitiateMPinResponse
import com.payu.upiboltcore.models.PayerType
import com.payu.upiboltcore.models.RaiseDisputeRequest
import com.payu.upiboltcore.models.RaiseDisputeResponse
import com.payu.upiboltcore.models.RemoveAccountRequest
import com.payu.upiboltcore.models.RemoveAccountResponse
import com.payu.upiboltcore.models.SetMPinRequest
import com.payu.upiboltcore.models.SetMPinResponse
import com.payu.upiboltcore.models.TransactionHistoryListResponse
import com.payu.upiboltcore.models.TransactionHistoryRequest
import com.payu.upiboltcore.models.UserInfo
import com.payu.upiboltcore.network.BaseRepository
import com.payu.upiboltcore.network.CommonHeaders
import com.payu.upiboltcore.network.UpiCoreApiService
import com.payu.upiboltcore.npci.DeviceInfoManager

class ManagementRepository(
    private val commonHeaders: CommonHeaders,
    private val upiCoreApiService: UpiCoreApiService
) : BaseRepository() {

    fun initiateMPinSetup(
        accountInfo: AccountInfo,
        payerType: PayerType,
        cardNo: String,
        expiry: String,
        npciTxnId: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        val headers = commonHeaders.getAuthTokenHeaders()
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val request = InitiateMPinRequest(
                accountId = accountInfo.accountId,
                accountInfo = accountInfo,
                deviceInfo = deviceInfo,
                payerType = payerType,
                setPinType = "DEBIT",
                lastSixDigitDebitCardNo = cardNo,
                expiryDate = expiry,
                virtualAddress = accountInfo.vpa ?: "",
                npciTransactionId = npciTxnId,
                refUrl = accountInfo.refUrl,
                refId = InternalConfig.sdkInitParams?.refId,
                referenceId = referenceId
            ).getJSONObject().toString()
            upiCoreApiService.initiateMPinSetup(headers, request, object : PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val initMPinResponse = InitiateMPinResponse.getInitMPinResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, initMPinResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback, PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun setMPin(
        accountId: String,
        expiry: String,
        format: String,
        lastSixCardDigits: String,
        credentials: List<Credential>,
        npciTxnId: String,
        otpNpciTxnId: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        val headers = commonHeaders.getAuthTokenHeaders()
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->

            if (credentials.size > 1) {
                val otpCredential =
                    credentials.firstOrNull { it.credentialSubType == UPIConstants.OTP_CREDENTIAL_SUB_TYPE }
                val atmPinCredential =
                    credentials.firstOrNull { it.credentialSubType == UPIConstants.CARD_CREDENTIAL_SUB_TYPE }
                val mPinCredential =
                    credentials.firstOrNull { it.credentialSubType == UPIConstants.ATM_CREDENTIAL_SUB_TYPE }

                val request = SetMPinRequest(otpCredentials = otpCredential, deviceInfo = deviceInfo,
                    accountId = accountId, expiryDate = expiry, formatType = format,
                    lastSixDigitDebitCardNo = lastSixCardDigits, mpinCredentials = mPinCredential,
                    atmPinCredentials = atmPinCredential, npciTranId = npciTxnId,
                    otpNpciTranId = otpNpciTxnId,referenceId = referenceId).getJSONObject().toString()

                upiCoreApiService.setMPin(headers, request, object : PayUAsyncTaskResponse {
                    override fun onSuccess(response: Any) {
                        getResponse(response, apiCallback) {
                            try {
                                val setMPinResponse = SetMPinResponse.getSetMPinResponseFromJSON(it)
                                sendSuccessCallback(apiCallback, setMPinResponse)
                            }
                            catch (e: Exception){
                                sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                    e.message.toString()
                                )
                            }

                        }
                    }

                    override fun onFailure(errorCode: Int, errorMessage: String) {
                        sendFailureCallback(apiCallback, errorCode, errorMessage)
                    }
                })
            }
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun changeMPin(
        accountId: String,
        credentials: List<Credential>,
        payerType: PayerType,
        vpa: String,
        npciTxnId: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            if (credentials.size > 1) {
                val oldCredential = credentials.first { it.credentialSubType == UPIConstants.ATM_CREDENTIAL_SUB_TYPE }
                val newCredential = credentials.first { it.credentialSubType == UPIConstants.NEW_PIN_CREDENTIAL_SUB_TYPE }
                newCredential.credentialSubType = UPIConstants.ATM_CREDENTIAL_SUB_TYPE
                val request = ChangeMPinRequest(accountId, oldCredential, newCredential,
                    npciTxnId, deviceInfo, payerType, vpa,referenceId = referenceId).getJSONObject().toString()
                upiCoreApiService.changeMPin(headers, request, object:
                    PayUAsyncTaskResponse {
                    override fun onSuccess(response: Any) {
                        getResponse(response, apiCallback) {
                            try {
                                val changeMPinResponse = ChangeMPinResponse.getChangeMPinResponseFromJSON(it)
                                if (changeMPinResponse.result.status == "S") {
                                    sendSuccessCallback(apiCallback, changeMPinResponse)
                                } else {
                                    sendFailureCallback(
                                        apiCallback,
                                        PayUResponseCodes.PAYU_FAILED_STATUS,
                                        changeMPinResponse.result.statusDesc
                                    )
                                }
                            }
                            catch (e: Exception){
                                sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                    e.message.toString()
                                )
                            }
                        }
                    }

                    override fun onFailure(errorCode: Int, errorMessage: String) {
                        sendFailureCallback(apiCallback, errorCode, errorMessage)
                    }
                })
            }
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun balanceEnquiry(
        accountId: String,
        credential: Credential,
        vpa: String,
        npciTxnId: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = BalanceEnquiryRequest(
                accountId,
                deviceInfo = deviceInfo,
                mpinCredential = credential,
                npciTxnId = npciTxnId,
                virtualAddress = vpa,
                referenceId = referenceId
            ).getJSONObject().toString()
            upiCoreApiService.balanceEnquiry(headers, request, object :
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val balanceEnquiryResponse =
                                BalanceEnquiryResponse.getBalanceEnquiryResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, balanceEnquiryResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }

            })
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun raiseDispute(complaintInfo: ComplaintInfo, referenceId:String?, apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = RaiseDisputeRequest(complaintInfo, deviceInfo,referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.raiseDispute(headers, request, object:
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val raiseDisputeResponse = RaiseDisputeResponse.getRaiseDisputeResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, raiseDisputeResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun addAccount(userInfo: UserInfo, referenceId:String?, apiCallback: ApiBaseCallback) {
        val headers = commonHeaders.getAuthTokenHeaders()
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val request = AddAccountRequest(deviceInfo, userInfo,referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.addAccount(headers, request, object : PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val addAccountResponse = AddAccountResponse.getAddAccountResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, addAccountResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun getDisputeList(referenceId:String?,apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = GetDisputeListRequest(deviceInfo,referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.getDisputeList(headers, request, object: PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val disputeListResponse = DisputeListResponse.getDisputeListResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, disputeListResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun removeAccount(
        accountList: List<AccountInfo>,
        userInfo: UserInfo,
        referenceId: String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = RemoveAccountRequest(
                deviceInfo, userInfo, UPIConstants.REMOVE_ACCOUNT_REQUEST_TYPE, accountList,referenceId = referenceId
            ).getJSONObject().toString()
            upiCoreApiService.removeAccount(headers, request, object : PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val removeAccountResponse = RemoveAccountResponse.getRemoveAccountResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, removeAccountResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun deRegisterVPA(referenceId:String?,apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = DeRegisterVPARequest(deviceInfo,referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.deRegisterVPA(headers, request, object:
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val deRegisterVPAResponse = DeRegisterVPAResponse.getDeRegisterVPAResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, deRegisterVPAResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun getTransactionHistoryList(startDate:String,endDate:String, referenceId:String?,apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = TransactionHistoryRequest(startDate,endDate,deviceInfo.mobileNo,referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.getTransactionHistoryList(headers, request, object: PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val transactionHistoryResponse = TransactionHistoryListResponse.getTransactionHistoryListResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, transactionHistoryResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun getDisputeTypeList(upiTransactionRefNo: String, referenceId:String?, apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = DisputeTypeListRequest(deviceInfo, upiTransactionRefNo, referenceId).getJSONObject().toString()
            upiCoreApiService.getDisputeTypeList(headers, request, object: PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    try {
                        getResponse(response, apiCallback) {
                            val disputeListResponse =
                                DisputeTypeListResponse.getDisputeTypeListResponseFromJSON(it)
                            disputeListResponse?.let {
                                sendSuccessCallback(apiCallback, disputeListResponse)
                            } ?: kotlin.run {
                                sendFailureCallback(
                                    apiCallback, PAYU_FAILED_STATUS,
                                    PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                                )
                            }
                        }
                    } catch (e: Exception) {
                        sendFailureCallback(
                            apiCallback, PAYU_FAILED_STATUS,
                            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                        )
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun checkDisputeStatus(
        txnId: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = CheckTransactionStatusRequest(
                deviceInfo = deviceInfo,
                actionType = PayUUpiConstant.PAYU_TRANSACTION_STATUS,
                onBehalfOf = PayUUpiConstant.PAYU_PAYEE,
                txnId = txnId,
                referenceId = referenceId
            ).getJSONObject().toString()
            upiCoreApiService.checkTransactionStatus(headers, request, object :
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val checkTxnStatusResponse = CheckTransactionStatusResponse.getTransactionStatusResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, checkTxnStatusResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }
}