package com.payu.upiboltcore.features.management

import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.KibanaConstants
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.PayUUPIPlugin.clearData
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.features.BaseService
import com.payu.upiboltcore.interfaces.ManagementService
import com.payu.upiboltcore.models.AccountInfo
import com.payu.upiboltcore.models.AddAccountResponse
import com.payu.upiboltcore.models.BalanceEnquiryResponse
import com.payu.upiboltcore.models.ChangeMPinResponse
import com.payu.upiboltcore.models.CheckTransactionStatusResponse
import com.payu.upiboltcore.models.ComplaintInfo
import com.payu.upiboltcore.models.Credential
import com.payu.upiboltcore.models.DeRegisterVPAResponse
import com.payu.upiboltcore.models.DisputeListResponse
import com.payu.upiboltcore.models.DisputeTypeListResponse
import com.payu.upiboltcore.models.InitiateMPinResponse
import com.payu.upiboltcore.models.PayInfo
import com.payu.upiboltcore.models.PayerType
import com.payu.upiboltcore.models.RaiseDisputeResponse
import com.payu.upiboltcore.models.RemoveAccountResponse
import com.payu.upiboltcore.models.SetMPinResponse
import com.payu.upiboltcore.models.TransactionHistoryListResponse
import com.payu.upiboltcore.models.UserInfo
import com.payu.upiboltcore.npci.CLCredentialManager
import com.payu.upiboltcore.utils.PayUValidationUtils
import com.payu.upiboltcore.utils.TransformerUtils
import com.payu.upiboltcore.utils.Utils
import org.json.JSONObject

class ManagementServiceImpl(
    activity: FragmentActivity,
    private val managementRepository: ManagementRepository,
    private val credentialManager: CLCredentialManager
) : BaseService(activity), ManagementService {

    override fun setMpin(
        account: AccountInfo,
        cardNo: String,
        exp: String,
        payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateActivateAccountParams(cardNo, exp)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack
                )
                return@onDeviceRegistered
            }
            val payerType = PayerType(
                account.accountId,
                account.accountNumber,
                account.bankName ?: "",
                account.vpa ?: ""
            )
            val npciTxnId = Utils.getUniqueTxnId()
            val expiryDate = exp.split("/").run {
                "${get(0)}${get(1).substring(get(1).length -2, get(1).length)}"
            }
            logRequestEvent(
                "${eventName}_${KibanaConstants.EVENT_NAME_INITIATE_MPIN}",
                JSONObject().apply {
                    put(KibanaConstants.ACCOUNT_NO, account.accountNumber)
                    put(KibanaConstants.VPA, Utils.getMaskedString(account.vpa))
                    put(KibanaConstants.CARD_NO, cardNo)
                    put(KibanaConstants.EXPIRY, exp)
                    put(KibanaConstants.NPCI_TXN_ID, npciTxnId)
                }.toString()
            )
            managementRepository.initiateMPinSetup(
                account, payerType, cardNo, expiryDate, npciTxnId, referenceId,
                object :
                    ApiBaseCallback {
                    override fun onApiSuccess(response: Any) {
                        (response as InitiateMPinResponse).result.run {
                            logSuccessEvent(
                                "${eventName}_${KibanaConstants.EVENT_NAME_INITIATE_MPIN}",
                                this.getJSONObject(maskPi = true, logInfo = true).toString()
                            )
                            val credential = credential ?: TransformerUtils.getCredentialFromAccount(account)
                            credential.let {
                                val payInfo = PayInfo(
                                    account.accountName,
                                    account.accountNumber,
                                    mobileNo = InternalConfig.sdkInitParams?.getPhone(),
                                    refId = InternalConfig.sdkInitParams?.refId
                                )
                                credentialManager.getCLCredential(
                                    UPIConstants.CRED_TYPE_SET_PIN,
                                    credential,
                                    credential.atmFormatType,
                                    payerAddress = account.vpa,
                                    payeeAddress = account.vpa,
                                    payInfo = payInfo,
                                    txnId = npciTranId,
                                    onSuccess = { credentials ->
                                        submitMpin(
                                            credentials, account, expiryDate, cardNo, npciTranId.ifEmpty { npciTxnId },
                                            otpNpciTranId, payUUpiProCallBack
                                        )
                                    },
                                    onFailure = { errorCode, errorMessage ->
                                        logEventAndSendFailureCallback(
                                            PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                                            errorCode, errorMessage,
                                            "${eventName}_${KibanaConstants.EVENT_NAME_INITIATE_MPIN}",
                                            payUUpiProCallBack
                                        )
                                    }, responseType = PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG
                                )
                            }
                        }
                    }

                    override fun onApiError(errorCode: Int, errorMessage: String) {
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                            errorCode, errorMessage,
                            "${eventName}_${KibanaConstants.EVENT_NAME_INITIATE_MPIN}",
                            payUUpiProCallBack
                        )
                    }
                }
            )
        }
    }

    override fun changeMpin(account: AccountInfo, callback: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_CHANGE_MPIN, callback) { eventName ->
            val payerType = PayerType(
                account.accountId,
                account.accountNumber,
                account.bankName ?: "",
                account.vpa ?: ""
            )
            val payInfo = PayInfo(
                account.accountName,
                account.accountNumber,
                mobileNo = InternalConfig.sdkInitParams?.getPhone(),
                refId = InternalConfig.sdkInitParams?.refId
            )

            val credential = TransformerUtils.getCredentialFromAccount(account)
            val txnId = Utils.getUniqueTxnId()
            credentialManager.getCLCredential(
                UPIConstants.CRED_TYPE_CHANGE_PIN,
                credential,
                credential.atmFormatType,
                payerAddress = account.vpa ?: "",
                payeeAddress = account.vpa ?: "",
                payInfo = payInfo,
                txnId = txnId,
                onSuccess = { credentials ->
                    logRequestEvent(
                        eventName,
                        JSONObject().apply {
                            put(KibanaConstants.ACCOUNT_NO, account.accountNumber)
                            put(KibanaConstants.VPA, Utils.getMaskedString(account.vpa))
                            put(KibanaConstants.NPCI_TXN_ID, txnId)
                        }.toString()
                    )
                    managementRepository.changeMPin(account.accountId,
                        credentials, payerType,
                        account.vpa ?: "", txnId, referenceId, object : ApiBaseCallback {
                            override fun onApiSuccess(response: Any) {
                                (response as ChangeMPinResponse).let { changeMpinResponse ->
                                    val result = changeMpinResponse.result

                                    logEventAndSendSuccessCallback(
                                        PayUResponseTypes.REQUEST_GET_CHANGE_MPIN,
                                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                        result.getJSONObject().toString(),
                                        eventName, callback, result
                                    )
                                }
                            }

                            override fun onApiError(errorCode: Int, errorMessage: String) {
                                logEventAndSendFailureCallback(
                                    PayUResponseTypes.REQUEST_GET_CHANGE_MPIN,
                                    errorCode, errorMessage, eventName, callback
                                )
                            }
                        })
                },
                onFailure = { errorCode, errorMessage ->
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_CHANGE_MPIN, errorCode, errorMessage, eventName, callback
                    )
                }, responseType = PayUResponseTypes.REQUEST_GET_CHANGE_MPIN
            )
        }
    }

    override fun checkBalance(account: AccountInfo, callback: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_BALANCE, callback) { eventName ->
            val payInfo = PayInfo(
                account.accountName,
                account.accountNumber,
                InternalConfig.sdkInitParams?.getPhone(),
                note = "Balance enquiry",
                refId = InternalConfig.sdkInitParams?.refId,
                refUrl = account.refUrl
            )
            val credential = TransformerUtils.getCredentialFromAccount(account)
            val txnId = Utils.getUniqueTxnId()
            credentialManager.getCLCredential(
                UPIConstants.CRED_TYPE_BALANCE_ENQUIRY,
                credential,
                credential.atmFormatType,
                payerAddress = account.vpa ?: "",
                payeeAddress = account.vpa ?: "",
                payInfo = payInfo,
                txnId = txnId,
                onSuccess = { credentialList ->
                    logRequestEvent(
                        eventName,
                        JSONObject().apply {
                            put(KibanaConstants.ACCOUNT_NO, account.accountNumber)
                            put(KibanaConstants.VPA, Utils.getMaskedString(account.vpa))
                            put(KibanaConstants.NPCI_TXN_ID, txnId)
                        }.toString()
                    )
                    managementRepository.balanceEnquiry(account.accountId, credentialList[0],
                        account.vpa ?: "", txnId, referenceId, object : ApiBaseCallback {
                            override fun onApiSuccess(response: Any) {
                                (response as BalanceEnquiryResponse).let { balanceEnquiryRes ->
                                    val result = balanceEnquiryRes.result.accountInfo.accountBalance
                                    result?.toDoubleOrNull()?.let { balance ->
                                        logEventAndSendSuccessCallback(
                                            PayUResponseTypes.REQUEST_GET_BALANCE,
                                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                            balanceEnquiryRes.result.getJSONObject(maskPi = true, logInfo = true).toString(),
                                            eventName, callback, balance.toString()
                                        )
                                    } ?: kotlin.run {
                                        logEventAndSendFailureCallback(
                                            PayUResponseTypes.REQUEST_GET_BALANCE,
                                            PayUResponseCodes.PAYU_FAILED_STATUS,
                                            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE,
                                            eventName, callback
                                        )
                                    }
                                }
                            }

                            override fun onApiError(errorCode: Int, errorMessage: String) {
                                logEventAndSendFailureCallback(
                                    PayUResponseTypes.REQUEST_GET_BALANCE, errorCode, errorMessage,
                                    eventName, callback
                                )
                            }
                        })
                },
                onFailure = { errorCode, errorMessage ->
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_BALANCE, errorCode, errorMessage,
                        eventName, callback
                    )
                }, responseType = PayUResponseTypes.REQUEST_GET_BALANCE
            )
        }
    }

    override fun raiseDispute(
        txnId: String,
        refId: String?,
        query: String,
        queryType: String?,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateRaiseDisputeRequestParams(txnId, query)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack,
                    txnId = txnId
                )
                return@onDeviceRegistered
            }
            val complaintInfo = ComplaintInfo(
                disputeType = queryType,
                orgTxnId = txnId,
                rrn = refId,
                userMsg = query,
                txnId = Utils.getUniqueTxnId()
            )

            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.TXN_ID, txnId)
                    put(KibanaConstants.REF_ID, refId)
                    put(KibanaConstants.QUERY, query)
                    put(KibanaConstants.QUERY_TYPE, queryType)
                }.toString(),
                txnId = txnId
            )
            managementRepository.raiseDispute(complaintInfo, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as RaiseDisputeResponse).let { disputeResponse ->
                        val result = disputeResponse.result
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            result.getJSONObject().toString(),
                            eventName, payUUpiProCallBack, result, txnId = txnId
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack, txnId = txnId
                    )
                }
            })
        }
    }

    override fun addAccount(accountInfo: AccountInfo, payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_ADD_ACCOUNT, payUUpiProCallBack) { eventName ->
            val vpa = InternalConfig.registeredAccountsMap?.values?.firstOrNull()
            val userInfo = UserInfo(
                name = InternalConfig.sdkInitParams?.email,
                defaultVPAStatus = false,
                accountId = accountInfo.accountId,
                virtualAddress = vpa,
                defaultFlag = "F"
            )
            logRequestEvent(
                eventName,
                accountInfo.getJSONObject(maskPi = true, logInfo = true).toString()
            )
            managementRepository.addAccount(userInfo, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as AddAccountResponse).let { addCountResponse ->
                        val result = addCountResponse.result
                        updateRegisteredAccountsMap(accountInfo.accountId, vpa)

                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_ADD_ACCOUNT,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            result.getJSONObject().toString(),
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_ADD_ACCOUNT,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun getDisputeList(payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_LIST_QUERIES_V3, payUUpiProCallBack) { eventName ->
            logRequestEvent(eventName, "")
            managementRepository.getDisputeList(referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as DisputeListResponse).let { disputeListResponse ->
                        val result = disputeListResponse.result.listofCustDispute

                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_LIST_QUERIES_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            "${KibanaConstants.DISPUTE_LIST_COUNT}${result.size}",
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_LIST_QUERIES_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun removeAccount(account: AccountInfo, payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3, payUUpiProCallBack) { eventName ->
            val userInfo = UserInfo(
                defaultVPAStatus = false,
                accountId = account.accountId,
                virtualAddress = account.vpa
            )

            logRequestEvent(eventName,
                account.getJSONObject(maskPi = true, logInfo = true).toString()
            )
            managementRepository.removeAccount(listOf(account), userInfo, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as RemoveAccountResponse).let { removeAccountResponse ->
                        val result = removeAccountResponse.result

                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            result.getJSONObject(maskPi = true, logInfo = true).toString(),
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun deregisterVpa(payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3, payUUpiProCallBack) { eventName ->
            logRequestEvent(eventName, "")
            managementRepository.deRegisterVPA(referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as DeRegisterVPAResponse).let { deRegisterVpaResponse ->
                        clearData()
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            deRegisterVpaResponse.result.getJSONObject().toString(),
                            eventName, payUUpiProCallBack, deRegisterVpaResponse.result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun getTransactionHistoryList(
        startDate: String,
        endDate: String,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateTxnHistoryRequestParams(startDate, endDate)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack
                )
                return@onDeviceRegistered
            }
            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.START_DATE, startDate)
                    put(KibanaConstants.END_DATE, endDate)
                }.toString()
            )
            managementRepository.getTransactionHistoryList(Utils.convertStringDateToFormat(
                startDate, UPIConstants.PAYU_YYYY_MM_DD
            ), Utils.convertStringDateToFormat(
                endDate, UPIConstants.PAYU_YYYY_MM_DD
            ), referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as TransactionHistoryListResponse).let { transactionHistoryResponse ->
                        val result = transactionHistoryResponse.result

                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            "${KibanaConstants.TXN_HISTORY_COUNT}${result.size}",
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun getDisputeTypeList(upiTxnRefNo: String, payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_QUERY_TYPE_LIST, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateDisputeTypeListParams(upiTxnRefNo)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack
                )
                return@onDeviceRegistered
            }
            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.UPI_TXN_REF_NO, upiTxnRefNo)
                }.toString()
            )
            managementRepository.getDisputeTypeList(upiTxnRefNo, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as? DisputeTypeListResponse)?.let {
                        val list = it.result.disputeTypeList
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            "${KibanaConstants.DISPUTE_TYPE_LIST_COUNT}${list.size}",
                            eventName, payUUpiProCallBack, list
                        )
                    } ?: kotlin.run {
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                            PayUResponseCodes.PAYU_RUNTIME_ERROR,
                            PayUResponseMessages.RUNTIME_ERROR_MESSAGE,
                            eventName, payUUpiProCallBack
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun checkDisputeStatus(
        txnId: String,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(
            PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateDisputeStatusParams(txnId)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack
                )
                return@onDeviceRegistered
            }
            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.TXN_ID, txnId)
                }.toString(),
                txnId = txnId
            )
            managementRepository.checkDisputeStatus(txnId, referenceId, object: ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as? CheckTransactionStatusResponse)?.let {
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            it.result.getJSONObject().toString(),
                            eventName, payUUpiProCallBack, it.result,
                            txnId = txnId
                        )
                    } ?: kotlin.run {
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS,
                            PayUResponseCodes.PAYU_RUNTIME_ERROR,
                            PayUResponseMessages.RUNTIME_ERROR_MESSAGE,
                            eventName, payUUpiProCallBack,
                            txnId = txnId
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS,
                        errorCode, errorMessage, eventName, payUUpiProCallBack,
                        txnId = txnId
                    )
                }
            })
        }
    }

    private fun submitMpin(
        credentials: List<Credential>,
        account: AccountInfo,
        exp: String,
        cardNo: String,
        txnId: String,
        otpTxnId: String,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        val eventName = "${PayUUpiConstant.PAYU_SET_MPIN}_${KibanaConstants.EVENT_NAME_SET_PIN}"
        logRequestEvent(
            eventName,
            JSONObject().apply {
                put(KibanaConstants.ACCOUNT_NO, account.accountNumber)
                put(KibanaConstants.VPA, Utils.getMaskedString(account.vpa))
                put(KibanaConstants.CARD_NO, cardNo)
                put(KibanaConstants.EXPIRY, exp)
                put(KibanaConstants.NPCI_TXN_ID, txnId)
                put(KibanaConstants.OTP_TXN_ID, otpTxnId)
            }.toString()
        )
        managementRepository.setMPin(
            account.accountId, exp, account.formatType, cardNo,
            credentials, txnId, otpTxnId, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as SetMPinResponse).let { setMpinResponse ->
                        val result = setMpinResponse.result

                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            result.getJSONObject(true).toString(),
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            }
        )
    }
}