package com.payu.upiboltcore.features.registration

import com.payu.commonmodelssdk.constants.ApiConstant
import com.payu.commonmodelssdk.constants.PayUResponseCodes.PAYU_FAILED_STATUS
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.PayUAsyncTaskResponse
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.constants.URLConstants
import com.payu.upiboltcore.models.AccountListResponse
import com.payu.upiboltcore.models.AccountsListRequest
import com.payu.upiboltcore.models.BanksListResponse
import com.payu.upiboltcore.models.CheckDeviceStatusRequest
import com.payu.upiboltcore.models.CheckDeviceStatusResponse
import com.payu.upiboltcore.models.GenerateOTPResponse
import com.payu.upiboltcore.models.GenerateOTPResponseResult
import com.payu.upiboltcore.models.GetBankListRequest
import com.payu.upiboltcore.models.RegisterVPARequest
import com.payu.upiboltcore.models.RegisterVPAResponse
import com.payu.upiboltcore.models.UserInfo
import com.payu.upiboltcore.models.ValidateOTPRequest
import com.payu.upiboltcore.models.ValidateOTPResponse
import com.payu.upiboltcore.models.VerifyServerGenIDRequest
import com.payu.upiboltcore.models.VerifyServerGenIDResponse
import com.payu.upiboltcore.network.BaseRepository
import com.payu.upiboltcore.network.CommonHeaders
import com.payu.upiboltcore.network.UpiCoreApiService
import com.payu.upiboltcore.npci.DeviceInfoManager
import org.json.JSONObject

class RegistrationRepository(
    private val commonHeaders: CommonHeaders,
    private val upiCoreApiService: UpiCoreApiService
): BaseRepository() {

    fun checkDeviceStatus(
        issuingBanksList: List<String>,
        clientId: String,
        referenceId: String?,
        smsGatewayNo: String? = null,
        registeredMobile: String? = null,
        isSmsSent: Boolean = false,
        isOtpVerified: Boolean = false,
        apiCallback: ApiBaseCallback
    ) {
        val callback = object : PayUAsyncTaskResponse {
            override fun onSuccess(response: Any) {
                getResponse(response, apiCallback) {
                    val json = JSONObject(it)
                    val result = json.getJSONObject(URLConstants.RES_PARAM_RESULT)
                    if (result.has(GenerateOTPResponseResult.OTP_SENT)) {
                        val generateOtpRes = GenerateOTPResponse.getGenerateOTPResponseFromJSON(it)
                        sendSuccessCallback(apiCallback, generateOtpRes)
                    } else {
                        val deviceStatusRes =
                            CheckDeviceStatusResponse.getCheckDeviceStatusResponseFromJSON(it)
                        sendSuccessCallback(apiCallback, deviceStatusRes)
                    }
                }
            }

            override fun onFailure(errorCode: Int, errorMessage: String) {
                sendFailureCallback(apiCallback, errorCode, errorMessage)
            }
        }

        DeviceInfoManager.deviceInfo?.let {
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = CheckDeviceStatusRequest(
                issuingBanks = issuingBanksList,
                clientId = clientId,
                deviceInfo = it,
                referenceId = referenceId,
                smsGateWayNo = smsGatewayNo,
                registeredMobile = registeredMobile,
                isSmsSent = isSmsSent,
                isOtpVerified = isOtpVerified
            ).getJSONObject().toString()
            upiCoreApiService.checkDeviceStatus(headers, request, callback)
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun  verifyServerGenId(referenceId:String?,apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = VerifyServerGenIDRequest(deviceInfo = deviceInfo, requestType = "UD",referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.verifyServerGenId(headers, request, object: PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val verifyServerGenIDResponse = VerifyServerGenIDResponse.getVerifyServerGenIDResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, verifyServerGenIDResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback, PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun getBanksList(referenceId:String?,apiCallback: ApiBaseCallback) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = GetBankListRequest(deviceInfo = deviceInfo,referenceId = referenceId).getJSONObject().toString()

            upiCoreApiService.getBanksList(headers,request, object :
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val banksListResponse = BanksListResponse.getBanksListResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, banksListResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun getAccountsList(
        bankCode: String?,
        requestType: String?,
        vpa: String?,
        referenceId: String?,
        iin: String? = null,
        bankId: String? = null,
        accountType: String? = null,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            var defaultVPA=vpa
            if(defaultVPA.isNullOrEmpty()) {
                defaultVPA = "${deviceInfo.mobileNo}${UPIConstants.HDFC_HANDLE}"
            }
            val userInfo = UserInfo(
                iin = iin,
                bankId = bankId,
                accountType = accountType,
                virtualAddress = defaultVPA
            )
            val request = AccountsListRequest(
                bankCode,
                deviceInfo = deviceInfo,
                requestType = requestType,
                userInfo = userInfo,
                referenceId = referenceId
            ).getJSONObject().toString()
            upiCoreApiService.getAccountsList(headers, request, object:
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val accountListResponse = AccountListResponse.getAccountListResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, accountListResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }

                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        }
    }

    fun registerVPA(
        userInfo: UserInfo,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = RegisterVPARequest(
                customerEmail = InternalConfig.sdkInitParams?.email,
                deviceInfo = deviceInfo,
                userInfo = userInfo,
                pspRespRefNo = InternalConfig.pspRespRefNo,
                referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.registerVPA(headers, request, object :
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val registerVPAResponse =
                                RegisterVPAResponse.getRegisterVPAResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, registerVPAResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }

    fun validateOTP(
        otp: String,
        referenceId:String?,
        apiCallback: ApiBaseCallback
    ) {
        DeviceInfoManager.deviceInfo?.let { deviceInfo ->
            val headers = commonHeaders.getAuthTokenHeaders()
            val request = ValidateOTPRequest(
                otp = otp,
                deviceInfo = deviceInfo,
                referenceId = referenceId).getJSONObject().toString()
            upiCoreApiService.validateOTP(headers, request, object :
                PayUAsyncTaskResponse {
                override fun onSuccess(response: Any) {
                    getResponse(response, apiCallback) {
                        try {
                            val validateOTPResponse =
                                ValidateOTPResponse.getValidateOTPResponseFromJSON(it)
                            sendSuccessCallback(apiCallback, validateOTPResponse)
                        }
                        catch (e: Exception){
                            sendFailureCallback(apiCallback,PAYU_FAILED_STATUS,
                                e.message.toString()
                            )
                        }
                    }
                }

                override fun onFailure(errorCode: Int, errorMessage: String) {
                    sendFailureCallback(apiCallback, errorCode, errorMessage)
                }
            })
        } ?: kotlin.run {
            sendFailureCallback(
                apiCallback, ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
        }
    }
}