package com.payu.upiboltcore.npci

import android.content.Context
import android.util.Base64
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.KibanaConstants
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.ApiRetryCallback
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.network.NPCIRepository
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.utils.AnalyticsUtils
import com.payu.upiboltcore.utils.PayUSPUtils
import com.payu.upiboltcore.utils.Utils
import org.json.JSONObject
import org.npci.upi.security.services.CLServices
import java.math.BigInteger
import java.security.SecureRandom
import java.util.Date
import java.util.UUID


class AppUPIRegistrationManager(
    private val clServices: CLServices,
    private val npciRepository: NPCIRepository
) {

    fun getUPIToken(
        context: Context,
        getToken: (token: String) -> Unit,
        onFailure: (errorCode: Int, errorMessage: String) -> Unit,
        refreshToken: Boolean? = false,
        responseType: Int
    ) {
//        if (refreshToken == true) {
            fetchUPIToken(
                context, UPIConstants.CHALLENGE_TYPE_INITIAL, responseType, getToken, onFailure
            )
//        } else {
//            getSavedUPIToken(context)?.let { savedToken ->
//                if (isTokenExpired(context)) {
//                    fetchUPIToken(
//                        context, UPIConstants.CHALLENGE_TYPE_ROTATE, responseType, getToken,
//                        onFailure
//                    )
//                } else {
//                    getToken.invoke(savedToken)
//                }
//            } ?: kotlin.run {
//                fetchUPIToken(
//                    context, UPIConstants.CHALLENGE_TYPE_INITIAL, responseType, getToken, onFailure
//                )
//            }
//        }
    }

    private fun fetchUPIToken(
        context: Context,
        credType: String,
        responseType: Int,
        getToken: (token: String) -> Unit,
        onFailure: (errorCode: Int, errorMessage: String) -> Unit,
        apiCount: Int = 0
    ) {
        DeviceInfoManager.deviceInfo?.run {
            val referenceID = UUID.randomUUID().toString()
            val startTime = System.currentTimeMillis()

            val tokenChallenge = NPCIUtils.getChallenge(credType, clServices)
            val tokenDataValue = "$deviceId|$appName|$mobileNo|$tokenChallenge"
            val txnId = Utils.getUniqueTxnId()

            AnalyticsUtils.logKibana(
                context,
                PayUUpiConstant.PAYU_NPCI_TOKEN,
                JSONObject().apply {
                    put(KibanaConstants.TXN_ID, txnId)
                }.toString(),
                false, refType = PayUUpiConstant.PLUGIN_CORE_REQUEST,
                referenceId = referenceID
            )
            npciRepository.getUPIToken(
                tokenDataValue, credType.uppercase(), txnId, referenceID,
                object :
                    ApiBaseCallback {
                    override fun onApiSuccess(response: Any) {
                        val decodedToken = Base64.decode(response as String, Base64.NO_WRAP)
                        val hexEncodedToken = String.format("%040x", BigInteger(1, decodedToken))

                        AnalyticsUtils.logKibana(
                            context,
                            PayUUpiConstant.PAYU_NPCI_TOKEN,
                            JSONObject().apply {
                                put(KibanaConstants.NPCI_TOKEN, Utils.getMaskedString(response))
                            }.toString(),
                            false, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                            time = Utils.getTimeDifferenceInMilliSeconds(startTime),
                            referenceId = referenceID,
                            status = CLConstant.PAYU_EVENT_SUCCESS
                        )
                        registerAppToCL(context, hexEncodedToken, getToken, onFailure)
                    }

                    override fun onApiError(errorCode: Int, errorMessage: String) {
                        if (PayUResponseCodes.PAYU_DEVICE_REGISTRATION_FAILED == errorCode) {
                            InternalConfig.sdkInitParams?.apiFailureCallback?.onApiFailed(
                                PayUUPIBoltResponse(responseType, errorCode, errorMessage),
                                object : ApiRetryCallback {
                                    override fun retry(shouldRetry: Boolean) {
                                        if (shouldRetry) {
                                            fetchUPIToken(
                                                context, UPIConstants.CHALLENGE_TYPE_INITIAL,
                                                responseType, getToken, onFailure, apiCount + 1
                                            )
                                        } else {
                                            sendFailureCallback(
                                                context, errorCode, errorMessage, startTime,
                                                referenceID, onFailure
                                            )
                                        }
                                    }
                                }, apiCount
                            ) ?: run {
                                sendFailureCallback(
                                    context, errorCode, errorMessage, startTime, referenceID,
                                    onFailure
                                )
                            }
                        } else {
                            sendFailureCallback(
                                context, errorCode, errorMessage, startTime, referenceID, onFailure
                            )
                        }
                    }
                }
            )
        }
    }

    private fun sendFailureCallback(
        context: Context, errorCode: Int, errorMessage: String, startTime: Long,
        referenceID: String, onFailure: (errorCode: Int, errorMessage: String) -> Unit
    ) {
        AnalyticsUtils.logKibana(
            context,
            PayUUpiConstant.PAYU_NPCI_TOKEN,
            errorMessage,
            true, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
            time = Utils.getTimeDifferenceInMilliSeconds(startTime),
            referenceId = referenceID,
            code = errorCode.toString(),
            status = CLConstant.PAYU_EVENT_FAILURE
        )
        onFailure.invoke(errorCode, errorMessage)
    }

    private fun getSavedUPIToken(context: Context): String? {
        return Utils.getEncryptedStringFromSP(
            context, getTokenPrefKey(), UPIConstants.UPI_TOKEN_ALIAS
        )
    }

    private fun saveUPIToken(context: Context, token: String) {
        Utils.saveEncryptedStringInSP(
            context, getTokenPrefKey(), token, UPIConstants.UPI_TOKEN_ALIAS
        )
        PayUSPUtils.saveStringInSP(
            context, UPIConstants.UPI_TOKEN_CREATED_TIME_KEY, System.currentTimeMillis().toString()
        )
    }

    private fun getTokenPrefKey() =
        "${UPIConstants.UPI_TOKEN_PREF_KEY}_${InternalConfig.issuingBanks[0]}"

    private fun isTokenExpired(context: Context): Boolean {
        val time = PayUSPUtils.getStringFromSP(context, UPIConstants.UPI_TOKEN_CREATED_TIME_KEY)?.toLong()
        return time?.let {
            val creationDate = Date(time)
            val date = Utils.getDateBeforeDays(Date(), UPIConstants.UPI_TOKEN_EXPIRY_DAYS)
            return date.compareTo(creationDate) >= 0
        } ?: kotlin.run {
            true
        }
    }

    fun registerApp(
        context: Context,
        responseType: Int,
        onSuccess: () -> Unit,
        onFailure: (errorCode: Int, errorMessage: String) -> Unit
    ) {
        getUPIToken(context, {
            onSuccess.invoke()
        }, onFailure, false, responseType)
    }

    private fun registerAppToCL(
        context: Context,
        token: String,
        getToken: (token: String) -> Unit,
        onFailure: (errorCode: Int, errorMessage: String) -> Unit
    ) {
        DeviceInfoManager.deviceInfo?.run {
            generateUPITokenHMAC(token) { hmac, random ->
                val status = clServices.registerApp(appName, mobileNo, deviceId, hmac, random)
                if (status) {
                    saveUPIToken(context, token)
                    getToken.invoke(token)
                } else {
                    onFailure.invoke(PayUResponseTypes.REQUEST_SDK_HANDSHAKE, PayUResponseMessages.PAYU_SDK_INITIALIZED_ERROR_MESSAGE)
                }
            }
        }
    }

    private fun generateUPITokenHMAC(token: String, hmac: (hmac: String?, random: String) -> Unit) {
        DeviceInfoManager.deviceInfo?.run {
            try {
                val secureRandom = SecureRandom()
                val random = ByteArray(16)
                secureRandom.nextBytes(random)
                val randomString = Base64.encodeToString(random, Base64.NO_WRAP)
                val concatenatedString = "$appName|$mobileNo|$deviceId"
                val hashBytes = Utils.generateSHA256Hash(concatenatedString, random)

                val encryptedHashString =
                    Utils.encrypt(Utils.hexStringToByteArray(token), hashBytes, random)
                val hmacValue = Base64.encodeToString(encryptedHashString, Base64.DEFAULT)
                hmac.invoke(hmacValue, randomString)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }
}