package com.payu.upiboltcore.npci

import android.app.Activity
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.os.Build
import android.os.CountDownTimer
import android.telephony.SmsManager
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.KibanaConstants
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.DeviceRegistration
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.features.registration.RegistrationRepository
import com.payu.upiboltcore.models.CheckDeviceStatusResponse
import com.payu.upiboltcore.models.CheckDeviceStatusResult
import com.payu.upiboltcore.models.DeviceRegistrationStatus
import com.payu.upiboltcore.models.VerifyServerGenIDResponse
import com.payu.upiboltcore.utils.PayUAndroidUtils
import com.payu.upiboltcore.utils.AnalyticsUtils
import com.payu.upiboltcore.utils.Utils
import org.json.JSONObject
import java.util.UUID
import kotlin.Exception


internal class SMSVerificationManager(
    private val context: Context,
    private val registrationRepository: RegistrationRepository,
    private val receiverMobile: String,
    content: String,
    private val isNewUPIRegistration: Boolean,
    private val registeredMobile: String? = null,
    private val onVerificationSuccess: (response: CheckDeviceStatusResult?) -> Unit,
    private val onVerificationFailure: (errorCode: Int, errorMessage: String) -> Unit
) : DefaultLifecycleObserver {

    private val sentBroadcastReceiver = SmsBroadcastReceiver()
    private val deliveredBroadcastReceiver = SmsBroadcastReceiver()
    private var isSMSDelivered = false
    private var isSMSSent = false
    private var isCheckDeivceSS = false
    private var deviceBindingTimer: CountDownTimer? = null
    private var checkDeviceTimer: CountDownTimer? = null
    private var verifyCount = 0
    private val maxVerifyCount =
        UPIConstants.SMS_VERIFICATION_TIMEOUT_MILLIS / UPIConstants.SMS_VERIFICATION_CHECK_INTERVAL_MILLIS
    private val TAG = "SMSVerificationManager"
    private var observer: SMSVerificationManager = this

    private val airplaneModeChangeReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            Log.d("AirplaneMode", "Service state changed")
            val isAirplaneModeOn = intent.getBooleanExtra("state", false)
            if (isAirplaneModeOn) {
                sendSMSVerificationFailedCallback(errorMsg = PayUResponseMessages.PAYU_AIRPLANE_MODE_ON_ERROR_MESSAGE)
            }
        }
    }

    init {
        if (PayUAndroidUtils.isAirplaneModeOn(context)) {
            onVerificationFailure.invoke(
                PayUResponseCodes.PAYU_DEVICE_BINDING_FAILED,
                PayUResponseMessages.PAYU_AIRPLANE_MODE_ON_ERROR_MESSAGE
            )
        } else {
            (context as AppCompatActivity).lifecycle.addObserver(observer)
            ContextCompat.registerReceiver(
                context, airplaneModeChangeReceiver,
                IntentFilter("android.intent.action.AIRPLANE_MODE"), ContextCompat.RECEIVER_EXPORTED
            )
            sendSMS(context, receiverMobile, content)
        }
    }


    inner class SmsBroadcastReceiver : BroadcastReceiver() {

        override fun onReceive(context: Context, intent: Intent) {

            if (intent.action == UPIConstants.ACTION_SENT) {
                context.unregisterReceiver(sentBroadcastReceiver)
                if (resultCode == Activity.RESULT_OK) {
                    isSMSSent = true
                } else {
                    sendSMSVerificationFailedCallback()
                }
            } else if (intent.action == UPIConstants.ACTION_DELIVERED) {
                context.unregisterReceiver(deliveredBroadcastReceiver)
                if (resultCode == Activity.RESULT_OK) {
                    isSMSDelivered = true
                    startVerifyDeviceBindingTimer()
                } else {
                    sendSMSVerificationFailedCallback()
                }
            }
        }
    }

    private fun getTimer(interval: Long, onTimerFinish: () -> Unit, onInterval: (() -> Unit)? = null): CountDownTimer {
        val timer = object : CountDownTimer(interval, interval) {
            override fun onTick(millisUntilFinished: Long) {
                onInterval?.invoke()
            }

            override fun onFinish() {
                onTimerFinish.invoke()
            }
        }
        return timer
    }

    private fun sendSMS(context: Context, receiverMobile: String, content: String) {
        if (hasPermissions(
                context, arrayOf(
                    android.Manifest.permission.SEND_SMS
                )
            )
        ) {
            val sentIntent = Intent(UPIConstants.ACTION_SENT)
            val sentPI =
                PendingIntent.getBroadcast(context, 0, sentIntent, PendingIntent.FLAG_IMMUTABLE)
            val deliveredIntent = Intent(UPIConstants.ACTION_DELIVERED)
            val deliveredPI = PendingIntent.getBroadcast(
                context,
                1,
                deliveredIntent,
                PendingIntent.FLAG_IMMUTABLE
            )

            val smsManager = InternalConfig.sdkInitParams?.subscriptionId?.let {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                    context.getSystemService(SmsManager::class.java).createForSubscriptionId(it.toInt())
                } else if (Build.VERSION.SDK_INT < Build.VERSION_CODES.S
                    && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1
                ) {
                    SmsManager.getSmsManagerForSubscriptionId(it.toInt())
                } else {
                    SmsManager.getDefault()
                }
            } ?: kotlin.run {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    context.getSystemService(SmsManager::class.java)
                } else {
                    SmsManager.getDefault()
                }
            }

            ContextCompat.registerReceiver(
                context, sentBroadcastReceiver,
                IntentFilter(UPIConstants.ACTION_SENT), ContextCompat.RECEIVER_EXPORTED
            )
            ContextCompat.registerReceiver(
                context, deliveredBroadcastReceiver,
                IntentFilter(UPIConstants.ACTION_DELIVERED), ContextCompat.RECEIVER_EXPORTED
            )
            try {
                smsManager?.sendTextMessage(receiverMobile, null, content, sentPI, deliveredPI)
                deviceBindingTimer = getTimer(UPIConstants.SMS_VERIFICATION_TIMEOUT_MILLIS, onTimerFinish = {
                    if (isCheckDeivceSS) {
                        onVerificationSuccess.invoke(null)
                    } else {
                        sendSMSVerificationFailedCallback(errorMsg = PayUResponseMessages.PAYU_DEVICE_BINDING_TIMEOUT_MESSAGE)
                    }
                })
                deviceBindingTimer?.start()
            } catch (ex: Exception) {
                sendSMSVerificationFailedCallback(
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    ex.localizedMessage ?: ex.message ?: PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                )
            }
        } else {
            sendSMSVerificationFailedCallback(
                PayUResponseCodes.PAYU_FAILED_STATUS,
                PayUResponseMessages.PAYU_SMS_PERMISSION_ERROR_MESSAGE
            )
        }
    }

    private fun hasPermissions(context: Context, permissions: Array<String>): Boolean =
        permissions.all {
            ActivityCompat.checkSelfPermission(context, it) == PackageManager.PERMISSION_GRANTED
        }

    private fun sendSMSVerificationFailedCallback(
        errorCode: Int = PayUResponseCodes.PAYU_DEVICE_BINDING_FAILED,
        errorMsg: String = PayUResponseMessages.PAYU_SMS_VERIFICATION_FAILED
    ) {
        (context as AppCompatActivity).lifecycle.removeObserver(observer)
        sendVerificationFailureResponse(errorCode, errorMsg)
    }

    private fun callCheckDeviceAPI() = synchronized(Unit) {
        val referenceID = UUID.randomUUID().toString()
        val startTime = System.currentTimeMillis()

        AnalyticsUtils.logKibana(
            context,
            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
            JSONObject().apply {
                put(KibanaConstants.ISSUING_BANKS, InternalConfig.issuingBanks.joinToString(","))
            }.toString(),
            false, refType = PayUUpiConstant.PLUGIN_CORE_REQUEST,
            referenceId = referenceID
        )
        registrationRepository.checkDeviceStatus(
            InternalConfig.issuingBanks,
            InternalConfig.clientId ?: "",
            referenceID, receiverMobile,
            registeredMobile = registeredMobile,
            isSmsSent = true,
            isOtpVerified = false,
            object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as? CheckDeviceStatusResponse)?.result?.run {
                        AnalyticsUtils.logKibana(
                            context,
                            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                            this.getJSONObject().toString(),
                            false, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                            time = Utils.getTimeDifferenceInMilliSeconds(startTime),
                            referenceId = referenceID,
                            status = CLConstant.PAYU_EVENT_SUCCESS
                        )
                        verifyCount += 1
                        if (smsStatus == UPIConstants.SMS_RECEIVED_STATUS) {
                            deviceBindingTimer?.cancel()
                            checkDeviceTimer?.cancel()
                            if (!isNewUPIRegistration) {
                                val reVerifyStartTime = System.currentTimeMillis()
                                AnalyticsUtils.logKibana(
                                    context,
                                    PayUUpiConstant.PAYU_RE_VERIFY_SERVER_GEN_ID, "",
                                    false, refType = PayUUpiConstant.PLUGIN_CORE_REQUEST,
                                    referenceId = referenceID
                                )
                                registrationRepository.verifyServerGenId(
                                    referenceID,
                                    object : ApiBaseCallback {
                                        override fun onApiSuccess(response: Any) {

                                            val verifyServerGenIDResponse =
                                                response as VerifyServerGenIDResponse
                                            AnalyticsUtils.logKibana(
                                                context,
                                                PayUUpiConstant.PAYU_RE_VERIFY_SERVER_GEN_ID,
                                                verifyServerGenIDResponse.result.getJSONObject().toString(),
                                                false, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                                                time = Utils.getTimeDifferenceInMilliSeconds(reVerifyStartTime),
                                                referenceId = referenceID,
                                                status = CLConstant.PAYU_EVENT_SUCCESS
                                            )

                                            isCheckDeivceSS = true
                                            (context as AppCompatActivity).lifecycle.removeObserver(observer)
                                            DeviceInfoManager.setAppGenId(
                                                context, verifyServerGenIDResponse.result.servGenId
                                            )
                                            InternalConfig.deviceRegistrationStatus =
                                                DeviceRegistrationStatus(DeviceRegistration.REGISTERED)
                                            onVerificationSuccess.invoke(this@run)
                                        }

                                        override fun onApiError(errorCode: Int, errorMessage: String) {
                                            AnalyticsUtils.logKibana(
                                                context,
                                                PayUUpiConstant.PAYU_RE_VERIFY_SERVER_GEN_ID,
                                                errorMessage,
                                                true, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                                                time = Utils.getTimeDifferenceInMilliSeconds(reVerifyStartTime),
                                                referenceId = referenceID,
                                                code = errorCode.toString(),
                                                status = CLConstant.PAYU_EVENT_FAILURE
                                            )
                                            sendSMSVerificationFailedCallback(
                                                errorCode, errorMessage
                                            )
                                        }
                                    })
                            } else {
                                isCheckDeivceSS = true
                                (context as AppCompatActivity).lifecycle.removeObserver(observer)
                                InternalConfig.deviceRegistrationStatus =
                                    DeviceRegistrationStatus(DeviceRegistration.REGISTERED)
                                onVerificationSuccess.invoke(this)
                            }
                        } else if (smsStatus == UPIConstants.SMS_NOT_RECEIVED_STATUS) {
                            startVerifyDeviceBindingTimer()
                        }
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    sendVerificationFailureResponse(errorCode, errorMessage)
                    AnalyticsUtils.logKibana(
                        context,
                        PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                        errorMessage,
                        true, refType = PayUUpiConstant.PLUGIN_CORE_RESPONSE,
                        time = Utils.getTimeDifferenceInMilliSeconds(startTime),
                        referenceId = referenceID,
                        code = errorCode.toString(),
                        status = CLConstant.PAYU_EVENT_FAILURE
                    )
                }
            }
        )
    }

    private fun startVerifyDeviceBindingTimer() {
        checkDeviceTimer = getTimer(UPIConstants.SMS_VERIFICATION_CHECK_INTERVAL_MILLIS,
            onTimerFinish = {
                if (verifyCount <= maxVerifyCount) {
                    callCheckDeviceAPI()
                } else {
                    sendSMSVerificationFailedCallback(errorMsg = PayUResponseMessages.PAYU_DEVICE_BINDING_TIMEOUT_MESSAGE)
                }
            })
        checkDeviceTimer?.start()
    }
    override fun onStop(owner: LifecycleOwner) {
        super.onStop(owner)
        Log.d(TAG, "App toggled")
        sendSMSVerificationFailedCallback(errorMsg = PayUResponseMessages.PAYU_APP_IN_BACKGROUND_ERROR)
    }

    private fun sendVerificationFailureResponse(errorCode: Int, responseMsg: String) {
        deviceBindingTimer?.cancel()
        checkDeviceTimer?.cancel()
        try {
            context.unregisterReceiver(airplaneModeChangeReceiver)
            context.unregisterReceiver(sentBroadcastReceiver)
            context.unregisterReceiver(deliveredBroadcastReceiver)
        } catch (ex: Exception) {
            Log.d(TAG, "Unregister exception")
        }
        onVerificationFailure.invoke(errorCode, responseMsg)
    }
}