package com.payu.upiboltcore.features

import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_REQUEST_UNKNOWN
import com.payu.commonmodelssdk.constants.payuResponseTypeMap
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.DeviceRegistration
import com.payu.upiboltcore.utils.AnalyticsUtils
import java.util.UUID

open class BaseService(private val activity: FragmentActivity) {

    protected var fnStartTime: Long? = 0
    protected var referenceId: String? = null

    protected fun onDeviceRegistered(
        responseType: Int,
        payUUPIProCallBack: PayUUPIBoltCallBack,
        onRegistered: (eventName: String) -> Unit
    ) {
        fnStartTime = System.currentTimeMillis()
        referenceId = UUID.randomUUID().toString()
        val eventName = payuResponseTypeMap[responseType] ?: PAYU_REQUEST_UNKNOWN
        InternalConfig.deviceRegistrationStatus?.deviceStatus?.let { status ->
            if (status == DeviceRegistration.REGISTERED) {
                onRegistered.invoke(eventName)
            } else {
                logEventAndSendFailureCallback(
                    responseType,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_DEVICE_BINDING_FAILED_MESSAGE,
                    eventName,
                    payUUPIProCallBack
                )
            }
        } ?: kotlin.run {
            logEventAndSendFailureCallback(
                responseType,
                PayUResponseCodes.PAYU_FAILED_STATUS,
                PayUResponseMessages.PAYU_DEVICE_BINDING_FAILED_MESSAGE,
                eventName,
                payUUPIProCallBack
            )
        }
    }

    private fun logKibana(
        name: String,
        data: String,
        isError: Boolean,
        severity: AnalyticsUtils.EventSeverity = AnalyticsUtils.EventSeverity.Low,
        refType: String,
        referenceId: String? = null,
        txnId: String? = null,
        time: Long? = null,
        code: String? = null,
        status: String? = null,
        amount: Double? = null
    ) {
        activity.baseContext?.let {
            AnalyticsUtils.logEventNameForKibana(
                it, "${PayUUpiConstant.PAYU_BOLT_CORE_API}$name",
                data, txnId, time, refType,
                code = code, status = status,
                eventType = if (isError)
                    AnalyticsUtils.EventType.Error
                else
                    AnalyticsUtils.EventType.Info,
                eventSeverity = severity,
                referenceId = referenceId,
                amount = amount,
                sdkName = PayUUpiConstant.PAYU_BOLT_CORE_SDK_NAME
            )
        }
    }

    protected fun logRequestEvent(
        eventName: String,
        eventData: String,
        txnId: String? = null,
        amount: Double? = null
    ) {
        logKibana(
            eventName, eventData, false,
            AnalyticsUtils.EventSeverity.Low,
            PayUUpiConstant.PLUGIN_CORE_REQUEST,
            referenceId,
            txnId,
            System.currentTimeMillis() - (fnStartTime ?: 0),
            amount = amount
        )
    }

    protected fun logEventAndSendSuccessCallback(
        responseType: Int,
        code: Int,
        eventData: String,
        eventName: String,
        callback: PayUUPIBoltCallBack,
        result: Any? = null,
        txnId: String? = null,
        amount: Double? = null
    ) {
        logSuccessEvent(eventName, eventData, txnId, amount)
        callback.onPayUSuccess(PayUUPIBoltResponse(responseType, code, result = result))
    }

    protected fun logSuccessEvent(
        eventName: String,
        eventData: String,
        txnId: String? = null,
        amount: Double? = null
    ) {
        logKibana(
            eventName, eventData, false,
            AnalyticsUtils.EventSeverity.Low,
            PayUUpiConstant.PLUGIN_CORE_RESPONSE,
            referenceId, txnId,
            System.currentTimeMillis() - (fnStartTime ?: 0),
            status = CLConstant.PAYU_EVENT_SUCCESS,
            amount = amount
        )
    }

    protected fun logEventAndSendFailureCallback(
        responseType: Int,
        code: Int,
        message: String,
        eventName: String,
        callback: PayUUPIBoltCallBack,
        txnId: String? = null,
        amount: Double? = null
    ) {
        logKibana(
            eventName, message, true,
            AnalyticsUtils.EventSeverity.High,
            PayUUpiConstant.PLUGIN_CORE_RESPONSE,
            referenceId, txnId,
            System.currentTimeMillis() - (fnStartTime ?: 0),
            code = code.toString(),
            status = CLConstant.PAYU_EVENT_FAILURE,
            amount = amount
        )
        callback.onPayUFailure(PayUUPIBoltResponse(responseType, code, message))
    }

    protected fun updateRegisteredAccountsMap(accId: String, vpa: String?) {
        if (InternalConfig.registeredAccountsMap == null)
            InternalConfig.registeredAccountsMap = HashMap()
        InternalConfig.registeredAccountsMap?.let { map ->
            map[accId] = vpa
        }
    }
}