package com.payu.upiboltcore.features.registration

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import androidx.core.content.ContextCompat
import com.google.android.gms.auth.api.phone.SmsRetriever
import com.google.android.gms.common.api.CommonStatusCodes
import com.google.android.gms.common.api.Status
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import kotlinx.coroutines.delay
import java.util.regex.Pattern


class OTPVerificationManager(
    private val context: Context,
    private val onReceiverStarted: (otpVerificationManager: OTPVerificationManager) -> Unit,
    private val onFailure: (errorCode: Int, errorMessage: String) -> Unit
) {

    private var smsReceiver: SmsBroadcastReceiver? = null
    private var otp: String? = null

    init {
        startSMSObserver()
    }

    private fun registerSMSReceiver() {
        smsReceiver = SmsBroadcastReceiver(onSuccess = { otp ->
            this@OTPVerificationManager.otp = otp
            unregisterReceiver()
        }, onFailure = {
            sendFailureCallback()
        })
        ContextCompat.registerReceiver(
            context, smsReceiver, IntentFilter(SmsRetriever.SMS_RETRIEVED_ACTION),
            ContextCompat.RECEIVER_EXPORTED
        )
    }

    private fun startSMSObserver() {
        val client = SmsRetriever.getClient(context)
        val task = client.startSmsRetriever()

        task.addOnSuccessListener {
            registerSMSReceiver()
            onReceiverStarted.invoke(this)
        }

        task.addOnFailureListener {
            sendFailureCallback()
        }
    }

    private fun sendFailureCallback() {
        unregisterReceiver()
        onFailure.invoke(
            PayUResponseCodes.PAYU_FAILED_STATUS,
            PayUResponseMessages.PAYU_SMS_RETRIEVAL_FAILED
        )
    }

    fun unregisterReceiver() {
        if (smsReceiver != null) {
            context.unregisterReceiver(smsReceiver)
            smsReceiver = null
        }
    }

    suspend fun awaitOtp(timeout: Int): String? {
        var elapsedTime = 0L
        while (otp.isNullOrEmpty()) {
            delay(1000)
            elapsedTime += 1000
            if (elapsedTime >= (timeout * 1000)) {
                break
            }
        }
        unregisterReceiver()
        return otp
    }
}

class SmsBroadcastReceiver(
    private val onSuccess: ((otp: String) -> Unit)? = null,
    private val onFailure: (() -> Unit)? = null
) : BroadcastReceiver() {

    override fun onReceive(context: Context, intent: Intent) {
        if (SmsRetriever.SMS_RETRIEVED_ACTION == intent.action) {
            val extras = intent.extras
            val status = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                extras?.getParcelable(SmsRetriever.EXTRA_STATUS, Status::class.java)
            } else {
                extras?.getParcelable(SmsRetriever.EXTRA_STATUS)
            }

            when (status?.statusCode) {
                CommonStatusCodes.SUCCESS -> {
                    val message = extras?.getString(SmsRetriever.EXTRA_SMS_MESSAGE)
                    if (message != null) {
                        val pattern = Pattern.compile("\\b\\d{4,6}\\b")
                        val matcher = pattern.matcher(message)
                        if (matcher.find()) {
                            val otp = matcher.group(0)
                            if (otp.isNullOrEmpty().not()) {
                                onSuccess?.invoke(otp)
                            } else
                                onFailure?.invoke()
                        }
                    }
                }

                CommonStatusCodes.TIMEOUT -> {
                    onFailure?.invoke()
                }
            }
        }
    }
}