package com.payu.upiboltcore.features.registration

import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.KibanaConstants
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_REQUEST_UNKNOWN
import com.payu.commonmodelssdk.constants.payuResponseTypeMap
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.OTPVerificationInterface
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.PayUUPIPlugin
import com.payu.upiboltcore.constants.DeviceRegistration
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.features.BaseService
import com.payu.upiboltcore.interfaces.RegistrationService
import com.payu.upiboltcore.models.AccountInfo
import com.payu.upiboltcore.models.AccountListResponse
import com.payu.upiboltcore.models.Bank
import com.payu.upiboltcore.models.BanksListResponse
import com.payu.upiboltcore.models.CheckDeviceStatusResponse
import com.payu.upiboltcore.models.DeviceRegistrationStatus
import com.payu.upiboltcore.models.GenerateOTPResponse
import com.payu.upiboltcore.models.GenerateOTPResponseResult
import com.payu.upiboltcore.models.RegisterVPAResponse
import com.payu.upiboltcore.models.UserInfo
import com.payu.upiboltcore.models.ValidateOTPResponse
import com.payu.upiboltcore.npci.AppUPIRegistrationManager
import com.payu.upiboltcore.npci.DeviceInfoManager
import com.payu.upiboltcore.npci.SMSVerificationManager
import com.payu.upiboltcore.utils.PayUAndroidUtils
import com.payu.upiboltcore.utils.PayUSPUtils
import com.payu.upiboltcore.utils.PayUValidationUtils
import com.payu.upiboltcore.utils.Utils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONObject

class RegistrationServiceImpl(
    private val activity: FragmentActivity,
    private val registrationRepository: RegistrationRepository,
    private val appUPIRegistrationManager: AppUPIRegistrationManager
): BaseService(activity), RegistrationService {

    private val coroutineScope = CoroutineScope(Dispatchers.IO)

    override fun registerApp(
        otpVerificationInterface: OTPVerificationInterface,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        appUPIRegistrationManager.registerApp(activity, onSuccess = {
            InternalConfig.sdkInitParams?.getPhone()?.let { phone ->
                if (PayUAndroidUtils.isSimStateActive(activity, phone)) {
                    checkDeviceStatus(false, otpVerificationInterface, payUUPIProCallBack)
                } else {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                        PayUResponseCodes.PAYU_SIM_INFO_NOT_AVAILABLE,
                        PayUResponseMessages.PAYU_SIM_INFO_NOT_AVAILABLE_MESSAGE,
                        PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                        payUUPIProCallBack
                    )
                }
            } ?: kotlin.run {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + PayUUpiConstant.PHONE,
                    PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                    payUUPIProCallBack
                )
            }
        }, onFailure = { errorCode, errorMessage ->
            logEventAndSendFailureCallback(
                PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                errorCode, errorMessage,
                PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                payUUPIProCallBack
            )
        }, responseType = PayUResponseTypes.REQUEST_SDK_HANDSHAKE)
    }

    override fun checkDeviceStatus(
        isOtpVerified: Boolean,
        otpVerificationInterface: OTPVerificationInterface?,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        logRequestEvent(
            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
            JSONObject().apply {
                put(KibanaConstants.ISSUING_BANKS, InternalConfig.issuingBanks.joinToString(","))
            }.toString()
        )
        if (isOtpVerified.not()) {
            InternalConfig.otpAttempts = (InternalConfig.otpAttempts ?: 0) + 1
            OTPVerificationManager(activity,
                onReceiverStarted = { otpVerificationManager ->
                    invokeCheckDeviceStatus(
                        isOtpVerified, payUUPIProCallBack, otpVerificationManager, otpVerificationInterface
                    )
                },
                onFailure = { errorCode, errorMessage ->
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                        errorCode, errorMessage,
                        PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                        payUUPIProCallBack
                    )
                })
        } else {
            invokeCheckDeviceStatus(
                isOtpVerified, payUUPIProCallBack, null, null
            )
        }
    }

    private fun invokeCheckDeviceStatus(
        isOtpVerified: Boolean,
        payUUPIProCallBack: PayUUPIBoltCallBack,
        otpVerificationManager: OTPVerificationManager? = null,
        otpVerificationInterface: OTPVerificationInterface? = null
    ) {
        registrationRepository.checkDeviceStatus(
            InternalConfig.issuingBanks,
            InternalConfig.clientId ?: "", referenceId,
            registeredMobile = PayUUPIPlugin.getRegisteredMobileNumber(activity),
            isOtpVerified = isOtpVerified,
            apiCallback = object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as? CheckDeviceStatusResponse)?.result?.run {
                        otpVerificationManager?.unregisterReceiver()
                        if (deviceStatus == UPIConstants.DEVICE_STATUS_DR && simStatus == UPIConstants.SIM_STATUS_SNN
                        ) {
                            handleDeviceBindingSuccess(this.getJSONObject().toString(), payUUPIProCallBack)
                        } else {
                            InternalConfig.deviceRegistrationStatus =
                                DeviceRegistrationStatus(
                                    DeviceRegistration.NOT_REGISTERED, smsGateWayNo,
                                    smsGateWayContent, smsGateWayKey
                                )
                            val isNewUPIRegistration =
                                (deviceStatus == UPIConstants.DEVICE_STATUS_DN && simStatus == UPIConstants.SIM_STATUS_SNM)
                            deviceInfo?.appGenId?.let {
                                DeviceInfoManager.setAppGenId(activity, it)
                            }
                            registerDevice(
                                isNewUPIRegistration,
                                PayUUPIPlugin.getRegisteredMobileNumber(activity),
                                payUUPIProCallBack
                            )
                        }
                    } ?: (response as? GenerateOTPResponse)?.result?.let { res ->
                        coroutineScope.launch {
                            handleGenerateOtpResponse(
                                res, otpVerificationManager, otpVerificationInterface, payUUPIProCallBack
                            )
                        }
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    otpVerificationManager?.unregisterReceiver()
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_SDK_HANDSHAKE, errorCode, errorMessage,
                        PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                        payUUPIProCallBack
                    )
                }
            }
        )
    }

    private fun handleDeviceBindingSuccess(eventData: String, payUUPIProCallBack: PayUUPIBoltCallBack) {
        InternalConfig.sdkInitParams?.getPhone()?.let { mobile ->
            PayUSPUtils.saveStringInSP(activity, UPIConstants.REGISTERED_MOBILE, mobile)
        }
        InternalConfig.deviceRegistrationStatus =
            DeviceRegistrationStatus(DeviceRegistration.REGISTERED)
        logEventAndSendSuccessCallback(
            PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
            PayUResponseCodes.PAYU_SUCCESS_STATUS,
            eventData,
            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
            payUUPIProCallBack
        )
    }

    private suspend fun handleGenerateOtpResponse(
        generateOTPResponseResult: GenerateOTPResponseResult,
        otpVerificationManager: OTPVerificationManager?,
        otpVerificationInterface: OTPVerificationInterface?,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        if (generateOTPResponseResult.otpSent == true) {
            otpVerificationInterface?.onOTPListenerStarted()
            val otp = otpVerificationManager?.awaitOtp(
                generateOTPResponseResult.otpExpiresIn
                    ?: PayUUpiConstant.PAYU_OTP_READ_TIMEOUT_SECONDS
            )
            if (otp.isNullOrEmpty().not()) {
                otpVerificationInterface?.onOTPReceived()
                validateOtp(otp ?: "", otpVerificationInterface, payUUPIProCallBack)
            } else {
                withContext(Dispatchers.Main.immediate) {
                    if ((InternalConfig.otpAttempts ?: 0) <= (generateOTPResponseResult.otpAttemptsRem ?: 0)) {
                        otpVerificationInterface?.onTimeElapsed()
                    } else {
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                            PayUResponseCodes.PAYU_FAILED_STATUS,
                            PayUResponseMessages.PAYU_SMS_RETRIEVAL_FAILED,
                            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                            payUUPIProCallBack
                        )
                    }
                }
            }
        } else {
            withContext(Dispatchers.Main.immediate) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_SMS_RETRIEVAL_FAILED,
                    PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                    payUUPIProCallBack
                )
            }
        }
    }

    override fun getBankList(payuApiCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_LIST_BANKS, payuApiCallBack) { eventName ->
            logRequestEvent(eventName, "")
            registrationRepository.getBanksList(referenceId, object :
                ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as BanksListResponse).let { banksListResponse ->
                        val bankDataList = banksListResponse.result.bankMasterList.map {
                            Bank.getBankData(it)
                        }
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_LIST_BANKS,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            "${KibanaConstants.BANK_LIST_COUNT}${bankDataList.size}",
                            eventName, payuApiCallBack, bankDataList
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_LIST_BANKS, errorCode, errorMessage,
                        eventName, payuApiCallBack
                    )
                }
            })
        }
    }

    private fun registerDevice(
        isNewUPIRegistration: Boolean,
        registeredMobile: String? = null,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        InternalConfig.deviceRegistrationStatus?.let { status ->
            if (status.deviceStatus != DeviceRegistration.REGISTERED
                && status.smsReceiverNumber.isNullOrEmpty().not()
                && status.smsContent.isNullOrEmpty().not()
            ) {
                SMSVerificationManager(
                    activity,
                    registrationRepository,
                    status.smsReceiverNumber!!,
                    status.smsContent ?: "",
                    isNewUPIRegistration,
                    registeredMobile = registeredMobile,
                    onVerificationSuccess = { response ->
                        InternalConfig.sdkInitParams?.getPhone()?.let { mobile ->
                            PayUSPUtils.saveStringInSP(
                                activity, UPIConstants.REGISTERED_MOBILE, mobile
                            )
                        }
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            response?.getJSONObject()?.toString()
                                ?: KibanaConstants.SMS_VERIFICATION_DONE,
                            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                            payUUPIProCallBack
                        )
                    },
                    onVerificationFailure = { errorCode, errorMessage ->
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                            errorCode, errorMessage,
                            PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                            payUUPIProCallBack,
                        )
                    }
                )
            }
        } ?: kotlin.run {
            logEventAndSendFailureCallback(
                PayUResponseTypes.REQUEST_SDK_HANDSHAKE,
                PayUResponseCodes.PAYU_HANDSHAKE_FAILED,
                PayUResponseMessages.PAYU_DEVICE_BINDING_FAILED_MESSAGE,
                PayUUpiConstant.PAYU_CHECK_DEVICE_STATUS,
                payUUPIProCallBack
            )
        }
    }

    override fun getAccountsList(
        bankCode: String?,
        requestType: String?,
        iin: String?,
        bankId: String?,
        accountType: String?,
        payuApiCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(
            PayUResponseTypes.REQUEST_FETCH_ACCOUNT_V3,
            payuApiCallBack
        ) { eventName ->
            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.BANK_CODE, bankCode)
                    put(KibanaConstants.REQUEST_TYPE, requestType)
                    put(KibanaConstants.IIN, iin)
                    put(KibanaConstants.BANK_ID, bankId)
                    put(KibanaConstants.ACCOUNT_TYPE, accountType)
                }.toString()
            )
            registrationRepository.getAccountsList(
                bankCode, requestType, null, referenceId, iin, bankId, accountType,
                object :
                    ApiBaseCallback {
                    override fun onApiSuccess(response: Any) {
                        (response as AccountListResponse).let { accountListResponse ->
                            InternalConfig.registeredAccountsMap?.let { map ->
                                accountListResponse.result.accountList.forEach {
                                    it.vpa = map[it.accountId]
                                }
                            }
                            InternalConfig.pspRespRefNo = accountListResponse.result.pspRespRefNo
                            logEventAndSendSuccessCallback(
                                PayUResponseTypes.REQUEST_FETCH_ACCOUNT_V3,
                                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                "${KibanaConstants.ACCOUNTS_COUNT}${accountListResponse.result.accountList.size}",
                                eventName,
                                payuApiCallBack,
                                accountListResponse.result.accountList
                            )
                        }
                    }

                    override fun onApiError(errorCode: Int, errorMessage: String) {
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_FETCH_ACCOUNT_V3,
                            errorCode, errorMessage, eventName, payuApiCallBack
                        )
                    }
                }
            )
        }
    }

    override fun registerVPA(
        vpa: String,
        accountInfo: AccountInfo,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3, payUUpiProCallBack) { eventName ->
            val param = PayUValidationUtils.validateRegisterVPAParams(vpa)
            if (param.isNotEmpty()) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                    eventName, payUUpiProCallBack
                )
                return@onDeviceRegistered
            }
            val userInfo = UserInfo(
                name = InternalConfig.sdkInitParams?.email,
                defaultVPAStatus = false,
                accountId = accountInfo.accountId,
                virtualAddress = vpa,
                accountNo = accountInfo.accountNumber
            )
            logRequestEvent(
                eventName,
                JSONObject().apply {
                    put(KibanaConstants.ACCOUNT_NO, accountInfo.accountNumber)
                    put(KibanaConstants.VPA, Utils.getMaskedString(vpa))
                }.toString()
            )
            registrationRepository.registerVPA(userInfo, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as RegisterVPAResponse).let { registerVPAResponse ->
                        val result = registerVPAResponse.result
                        result.userInfo.accountId?.let { accId ->
                            updateRegisteredAccountsMap(accId, result.userInfo.virtualAddress)
                        }
                        activity.baseContext?.let {
                            if (result.servGenId.isNullOrEmpty().not())
                                DeviceInfoManager.setAppGenId(it, result.servGenId!!)
                        }
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            registerVPAResponse.result.getJSONObject(true).toString(),
                            eventName, payUUpiProCallBack, result
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3, errorCode,
                        errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun validateOtp(
        otp: String,
        otpVerificationInterface: OTPVerificationInterface?,
        payUUpiProCallBack: PayUUPIBoltCallBack
    ) {
        val eventName =
            payuResponseTypeMap[PayUResponseTypes.REQUEST_VALIDATE_OTP] ?: PAYU_REQUEST_UNKNOWN
        logRequestEvent(eventName, "")
        registrationRepository.validateOTP(otp, referenceId, object : ApiBaseCallback {
            override fun onApiSuccess(response: Any) {
                (response as? ValidateOTPResponse)?.result?.let { validateOTPResponse ->
                    if (validateOTPResponse.isOtpVerified == true
                        && validateOTPResponse.isExistingUser == true
                        && validateOTPResponse.isCustomerActive == true
                    ) {
                        handleDeviceBindingSuccess(
                            validateOTPResponse.getJSONObject().toString(), payUUpiProCallBack
                        )
                    } else {
                        checkDeviceStatus(true, otpVerificationInterface, payUUpiProCallBack)
                    }
                }
            }

            override fun onApiError(errorCode: Int, errorMessage: String) {
                logEventAndSendFailureCallback(
                    PayUResponseTypes.REQUEST_VALIDATE_OTP, errorCode,
                    errorMessage, eventName, payUUpiProCallBack
                )
            }
        })
    }
}