package com.payu.upiboltcore.features.transaction

import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.KibanaConstants
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.commonmodelssdk.listeners.ApiFailedCallback
import com.payu.commonmodelssdk.listeners.ApiRetryCallback
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.features.BaseService
import com.payu.upiboltcore.interfaces.TransactionService
import com.payu.upiboltcore.models.AccountInfo
import com.payu.upiboltcore.models.CheckVPAResponse
import com.payu.upiboltcore.models.GetProfileResponse
import com.payu.upiboltcore.models.MakePaymentResponse
import com.payu.upiboltcore.models.MerchantDetails
import com.payu.upiboltcore.models.PayInfo
import com.payu.upiboltcore.models.PayeeType
import com.payu.upiboltcore.models.PayerType
import com.payu.upiboltcore.npci.CLCredentialManager
import com.payu.upiboltcore.utils.PayUValidationUtils
import com.payu.upiboltcore.utils.TransformerUtils
import com.payu.upiboltcore.utils.Utils
import org.json.JSONObject

class TransactionServiceImpl(
    activity: FragmentActivity,
    private val transactionRepository: TransactionRepository,
    private val credentialManager: CLCredentialManager
) : BaseService(activity), TransactionService {

    override fun checkVPA(
        vpa: String,
        requestType: String,
        payUUPIProCallBack: PayUUPIBoltCallBack
    ) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_GET_PROFilE_VPA_V3, payUUPIProCallBack) {
            transactionRepository.checkVPA(vpa, requestType, referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as CheckVPAResponse).let { checkVpaResponse ->
                        val result = checkVpaResponse.result

                        val payUUPIBoltResponse = PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_GET_PROFilE_VPA_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            PayUResponseMessages.PAYU_SUCCESS_MESSAGE,
                            result
                        )
                        payUUPIProCallBack.onPayUSuccess(
                            payUUPIBoltResponse
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    payUUPIProCallBack.onPayUFailure(
                        PayUUPIBoltResponse(
                            errorCode,
                            PayUResponseCodes.PAYU_FAILED_STATUS,
                            errorMessage
                        )
                    )
                }
            })
        }
    }

    override fun getProfile(payUUpiProCallBack: PayUUPIBoltCallBack) {
        onDeviceRegistered(PayUResponseTypes.REQUEST_ALL_ACCOUNTS_V3, payUUpiProCallBack) { eventName ->
            logRequestEvent(eventName, "")
            transactionRepository.getProfile(referenceId, object : ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    (response as GetProfileResponse).let { getProfileResponse ->
                        val result = getProfileResponse.result
                        result.vpaAccountDetails?.forEach { account ->
                            account?.run {
                                updateRegisteredAccountsMap(accountId, vpa)
                            }
                        }
                        val accounts = result.vpaAccountDetails ?: emptyList()
                        logEventAndSendSuccessCallback(
                            PayUResponseTypes.REQUEST_ALL_ACCOUNTS_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            "${KibanaConstants.ACCOUNTS_COUNT}${accounts.size}",
                            eventName, payUUpiProCallBack, accounts
                        )
                    }
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_ALL_ACCOUNTS_V3,
                        errorCode, errorMessage, eventName, payUUpiProCallBack
                    )
                }
            })
        }
    }

    override fun makePayment(
        txnAmount: String, account: AccountInfo, payeeName: String, payeeVpa: String, note: String?,
        txnId: String, mcc: String?, merchantId: String?, initMode: String, purpose: String,
        refUrl: String?, payeeAccountNo: String?, payUUpiProCallBack: PayUUPIBoltCallBack,
        refreshToken: Boolean?, retryCallback: ApiFailedCallback?, apiCount: Int
    ) {

        onDeviceRegistered(PayUResponseTypes.REQUEST_PAY, payUUpiProCallBack) { eventName ->
            Utils.formatAmount(txnAmount)?.let { amount ->
                val param = PayUValidationUtils.validatePaymentParams(
                    amount, payeeName, payeeVpa, txnId, initMode, purpose
                )
                if (param.isNotEmpty()) {
                    logEventAndSendFailureCallback(
                        PayUResponseTypes.REQUEST_PAY,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param,
                        eventName, payUUpiProCallBack,
                        txnId = txnId, amount = txnAmount.toDoubleOrNull()
                    )
                    return@onDeviceRegistered
                }
                val npciTxnId = Utils.getUniqueTxnId()
                val payInfo = PayInfo(
                    payeeName, payeeAccountNo, InternalConfig.sdkInitParams?.getPhone(),
                    txnAmount = amount, note = note
                )
                val credential = TransformerUtils.getCredentialFromAccount(account)
                credentialManager.getCLCredential(
                    UPIConstants.CRED_TYPE_PAY, credential,
                credential.atmFormatType, payerAddress = account.vpa, payeeAddress = payeeVpa,
//                    credential.atmFormatType, payerAddress = "${account.vpa}21", payeeAddress = "onida@mypsp2",
                    payInfo = payInfo, txnId = npciTxnId,
                    onSuccess = { credentials ->
                        val payee = PayeeType(
                            mcc = mcc, name = payeeName, merchantName = payeeName,
                        virtualAddress = payeeVpa
//                            virtualAddress = "onida@mypsp2"
                        )
                        val payer = PayerType(
                            account.accountId, account.accountNumber, account.bankName ?: "",
                        account.vpa ?: "",
//                            "${account.vpa}21",
                            account.accountName, isMerchant = false,
                            accountType = account.accountType, payerBankCode = account.bankCode
                        )
                        val merchantDetails = MerchantDetails(merchantId = merchantId)
                        logRequestEvent(
                            eventName,
                            JSONObject().apply {
                                put(KibanaConstants.AMOUNT, txnAmount)
                                put(
                                    KibanaConstants.ACCOUNT_INFO,
                                    account.getJSONObject(maskPi = true, logInfo = true).toString()
                                )
                                put(KibanaConstants.PAYEE_VPA, Utils.getMaskedString(payeeVpa))
                                put(KibanaConstants.PAYER_VPA, Utils.getMaskedString(account.vpa))
                                put(KibanaConstants.PAYER_ACC_NO, account.accountNumber)
                                put(KibanaConstants.NPCI_TXN_ID, npciTxnId)
                            }.toString(),
                            txnId = txnId, amount = txnAmount.toDoubleOrNull()
                        )
                        transactionRepository.makePayment(
                            amount, credentials[0], merchantDetails, payee, payer, note, txnId,
                            initMode, purpose, null, refUrl, npciTxnId, referenceId,
                            object : ApiBaseCallback {
                                override fun onApiSuccess(response: Any) {
                                    (response as MakePaymentResponse).let { makePaymentResponse ->
                                        val result = makePaymentResponse.result

                                        logEventAndSendSuccessCallback(
                                            PayUResponseTypes.REQUEST_PAY,
                                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                            "${KibanaConstants.PAYMENT_SUCCESS}${account.vpa} to $payeeVpa",
                                            eventName, payUUpiProCallBack, result,
                                            txnId, amount = txnAmount.toDoubleOrNull()
                                        )
                                    }
                                }

                                override fun onApiError(errorCode: Int, errorMessage: String) {
                                    if (PayUResponseCodes.PAYU_DEVICE_FINGERPRINT_MISMATCHED == errorCode) {
                                        retryCallback?.onApiFailed(
                                            PayUUPIBoltResponse(
                                                PayUResponseTypes.REQUEST_PAY, errorCode
                                            ), object : ApiRetryCallback {
                                                override fun retry(shouldRetry: Boolean) {
                                                    if (shouldRetry) {
                                                        makePayment(
                                                            amount, account, payeeName, payeeVpa,
                                                            note, txnId, mcc, merchantId, initMode,
                                                            purpose, refUrl, payeeAccountNo,
                                                            payUUpiProCallBack, true, retryCallback,
                                                            apiCount + 1
                                                        )
                                                    } else {
                                                        logEventAndSendFailureCallback(
                                                            PayUResponseTypes.REQUEST_PAY,
                                                            errorCode, errorMessage, eventName,
                                                            payUUpiProCallBack, txnId,
                                                            amount = txnAmount.toDoubleOrNull()
                                                        )
                                                    }
                                                }
                                            }, apiCount) ?: run {
                                            logEventAndSendFailureCallback(
                                                PayUResponseTypes.REQUEST_PAY, errorCode,
                                                errorMessage, eventName, payUUpiProCallBack, txnId,
                                                amount = txnAmount.toDoubleOrNull()
                                            )
                                        }
                                    } else {
                                        logEventAndSendFailureCallback(
                                            PayUResponseTypes.REQUEST_PAY, errorCode, errorMessage,
                                            eventName, payUUpiProCallBack, txnId,
                                            amount = txnAmount.toDoubleOrNull()
                                        )
                                    }
                                }
                            }
                        )
                    },
                    onFailure = { errorCode, errorMessage ->
                        logEventAndSendFailureCallback(
                            PayUResponseTypes.REQUEST_PAY, errorCode, errorMessage, eventName,
                            payUUpiProCallBack, txnId, amount = txnAmount.toDoubleOrNull()
                        )
                    }, refreshToken, PayUResponseTypes.REQUEST_PAY
                )
            }
        }
    }
}