package com.payu.upiboltcore.network

import com.payu.commonmodelssdk.constants.ApiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.PayUAsyncTaskResponse
import com.payu.commonmodelssdk.model.PayUNetworkData
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.models.ErrorResponse
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import java.io.IOException
import java.util.concurrent.TimeUnit

internal class HttpApiCall {
    private var call: Call? = null

    fun makePostAPICall(
        httpMethod: String = ApiConstant.PAYU_POST,
        payUNetworkData: PayUNetworkData,
        payUAsyncTaskResponse: PayUAsyncTaskResponse
    ) {

        if (httpMethod == ApiConstant.PAYU_POST && payUNetworkData.postRequest.isNullOrEmpty()) {
            payUAsyncTaskResponse.onFailure(
                ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
            )
            return
        }
        if (payUNetworkData.url.isEmpty()) {
            payUAsyncTaskResponse.onFailure(
                ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                ApiConstant.REST_API_PATH_CANNOT_BE_NULL
            )
            return
        }
        val okHttpClientBuilder: OkHttpClient.Builder = OkHttpClient.Builder()
        val client = okHttpClientBuilder.connectTimeout(60, TimeUnit.SECONDS)
            .writeTimeout(60, TimeUnit.SECONDS)
            .readTimeout(60, TimeUnit.SECONDS).build()

        val mediaType: MediaType = payUNetworkData.contentType.toMediaType()

        val request: Request.Builder = Request.Builder()

        request.addHeader(ApiConstant.PAYU_CONTENT_TYPE, ApiConstant.PAYU_APPLICATION_JSON)
        when (httpMethod) {
            ApiConstant.PAYU_POST -> {
                val body: RequestBody? = payUNetworkData.postRequest?.toRequestBody(mediaType)
                if (body != null) {
                    request.post(body)
                } else {
                    payUAsyncTaskResponse.onFailure(
                        ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                        ApiConstant.REQUEST_BODY_CANNOT_BE_NULL
                    )
                }
                request.url(payUNetworkData.url)
            }
            PayUUpiConstant.PAYU_GET -> {
                if (payUNetworkData.getRequest != null) {
                    var queryRequest = ""
                    for ((key, value) in payUNetworkData.getRequest!!) {
                        queryRequest += "$key=$value&"
                    }
                    request.url(payUNetworkData.url + "?${queryRequest.dropLast(1)}")
                }
            }
            PayUUpiConstant.PAYU_DELETE -> {
                payUNetworkData.postRequest?.let {
                    request.delete(it.toRequestBody(mediaType))
                    request.url(payUNetworkData.url)
                }
            }
            PayUUpiConstant.PAYU_PATCH -> {
                payUNetworkData.postRequest?.let {
                    request.patch(it.toRequestBody(mediaType))
                    request.url(payUNetworkData.url)
                }
            }
        }
        if (payUNetworkData.headerMap != null) {
            for ((key, value) in payUNetworkData.headerMap!!) {
                request.addHeader(key, value)
            }
        }

        call = client.newCall(request.build())
        call?.let { InternalConfig.activeApiCalls.add(it) }
        call?.enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                payUAsyncTaskResponse.onFailure(
                    ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                    com.payu.commonmodelssdk.constants.PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                )
                InternalConfig.activeApiCalls.remove(call)
            }

            override fun onResponse(call: Call, response: Response) {
                if(response.code == 200) {
                    payUAsyncTaskResponse.onSuccess(response)
                } else {
                    response.body?.string()?.let {
                        val error = ErrorResponse.getErrorResponseFromJSON(it)
                        payUAsyncTaskResponse.onFailure(response.code, error.message)
                    } ?: kotlin.run {
                        payUAsyncTaskResponse.onFailure(
                            ApiConstant.SDK_RESPONSE_STATUS_CODE_1,
                            com.payu.commonmodelssdk.constants.PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                        )
                    }
                }
                InternalConfig.activeApiCalls.remove(call)
            }
        })
    }
}