package com.payu.upiboltcore.npci

import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.os.ResultReceiver
import android.util.Base64
import android.util.Log
import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.models.Credential
import com.payu.upiboltcore.models.PayInfo
import com.payu.upiboltcore.utils.Utils
import org.json.JSONArray
import org.json.JSONObject
import org.npci.upi.security.services.CLRemoteResultReceiver
import org.npci.upi.security.services.CLServices
import java.security.SecureRandom

class CLCredentialManager(
    private val activity: FragmentActivity,
    private val clServices: CLServices,
    private val registrationManager: AppUPIRegistrationManager,
    private val listKeysPayloadManager: ListKeysPayloadManager
) {

    fun getCLCredential(
        credType: String,
        credential: Credential,
        format: String?,
        payerAddress: String? = null,
        payeeAddress: String? = null,
        payInfo: PayInfo,
        txnId: String,
        onSuccess: (credentials: List<Credential>) -> Unit,
        onFailure: (errorCode: Int, errorMessage: String) -> Unit,
        refreshToken: Boolean? = false,
        responseType: Int
    ) {
        registrationManager.getUPIToken(activity, { token ->
            val payload = getListKeysPayload()
            if (payload.isNullOrEmpty().not()) {
                val control = getControls(credType, credential, format)
                val random = ByteArray(16)
                SecureRandom().nextBytes(random)
                val randomString = Base64.encodeToString(random, Base64.NO_WRAP)
                val salt = getSaltJSON(
                    credType, payerAddress, payeeAddress, payInfo.txnAmount, txnId, randomString
                ).toString()
                Log.d("CL_Logs_Salt",salt)
                Log.d("CL_Logs_Random",random.toString())
                getTrust(
                    credType, payerAddress, payeeAddress, payInfo.txnAmount, txnId, random, token
                ) { trust ->
                    payInfo.refId = txnId
                    payInfo.refUrl = "https://psp1.com"
                    val payInfoJson = payInfo.getPayInfoJSON()
                    val configJson = getConfigJSON().toString()

                    Log.d(
                        "CL_Logs_SDK",
                        "${UPIConstants.CREDENTIAL_KEY_CODE}, ${payload}, $control, ${configJson}, ${salt}, ${payInfoJson}, ${trust}, ${UPIConstants.CREDENTIAL_PREF_LANGUAGE}"
                    )
                    clServices.getCredential(
                        UPIConstants.CREDENTIAL_KEY_CODE,
                        payload, control.toString(), configJson, salt,
                        payInfoJson, trust, UPIConstants.CREDENTIAL_PREF_LANGUAGE,
                        CLRemoteResultReceiver(
                            CredResultReceiver(credType, control, onSuccess, onFailure)
                        )
                    )
                }
            } else {
                onFailure.invoke(
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                )
            }
        }, onFailure, refreshToken, responseType)
    }

    private fun getListKeysPayload(): String? {
        return String(
            Base64.decode(listKeysPayloadManager.getListKeysPayload(), Base64.NO_WRAP)
        )
    }

    private fun getControls(
        credType: String,
        credential: Credential,
        format: String?
    ): JSONObject {
        return when (credType) {
            UPIConstants.CRED_TYPE_SET_PIN -> getSetMpinCredentials(format, credential)
            UPIConstants.CRED_TYPE_BALANCE_ENQUIRY,
            UPIConstants.CRED_TYPE_PAY -> getMpinControls(credential)
            UPIConstants.CRED_TYPE_CHANGE_PIN -> getChangeMPinControls(credential)
            else -> JSONObject()
        }
    }

    private fun getCredJSON(type: String?, subType: String?, dType: String?, dLength: String?): JSONObject {
        return JSONObject().apply {
            put(UPIConstants.CREDENTIAL_KEY_TYPE, type)
            put(UPIConstants.CREDENTIAL_KEY_SUB_TYPE, subType)
            put(UPIConstants.CREDENTIAL_KEY_D_TYPE, dType)
            put(UPIConstants.CREDENTIAL_KEY_D_LENGTH, dLength)
        }
    }

    private fun getConfigJSON(): JSONObject {
        return JSONObject().apply {
            put("resendAadhaarOTPFeature", false)
            put("resendIssuerOTPFeature", false)
            put("issuerResendOTPLimit", 2)
            put("aadhaarResendOTPLimit", 1)
            put("forgotUpiPINEnabled", false)
            put("captureCardDetails", false)
        }
    }

    private fun getSaltJSON(
        credType: String,
        payerAddress: String? = null,
        payeeAddress: String? = null,
        txnAmount: String,
        txnId: String,
        random: String
    ): JSONObject {
        return JSONObject().apply {
            put("txnId", JSONArray().put(txnId))
            put("txnAmount", txnAmount)
            DeviceInfoManager.deviceInfo?.run {
                put("deviceId", deviceId)
                put("appId", appName)
                put("mobileNumber", mobileNo)
            }
            put("credType", JSONArray().put(credType))
            payerAddress?.let {
                put("payerAddr", payerAddress)
                put("payeeAddr", payeeAddress ?: payerAddress)
            }
            put("random", random)
        }
    }

    private fun getTrust(
        credType: String,
        payerAddress: String? = null,
        payeeAddress: String? = null,
        txnAmount: String,
        txnId: String,
        random: ByteArray,
        token: String,
        getTrust: (trust: String) -> Unit,
    ) {
        DeviceInfoManager.deviceInfo?.run {
            val concatenatedString =
                "$credType|$txnId|$appName|$mobileNo|$deviceId|$payerAddress|${payeeAddress ?: payerAddress}|$txnAmount"

            Log.d("CL_Logs_RawTrust",concatenatedString)
            val hashBytes = Utils.generateSHA256Hash(concatenatedString, random)
            val encryptedHashString =
                Utils.encrypt(Utils.hexStringToByteArray(token), hashBytes, random)
            val trust = Base64.encodeToString(encryptedHashString, Base64.DEFAULT)
            val trustJson = JSONObject().apply {
                put(credType, trust)
            }.toString()
            Log.d("CL_Logs_Token", token)
            Log.d("CL_Logs_EnCryptedTrust", trust)
            Log.d("CL_Logs_TrustJson", trustJson)
            getTrust.invoke(trustJson)
        }
    }

    inner class CredResultReceiver(
        private val credType: String,
        private val controls: JSONObject,
        private val onSuccess: (credentials: List<Credential>) -> Unit,
        private val onFailure: (errorCode: Int, errorMessage: String) -> Unit
    ) : ResultReceiver(Handler(Looper.getMainLooper())) {

        override fun onReceiveResult(resultCode: Int, resultData: Bundle) {
            super.onReceiveResult(resultCode, resultData)

            Log.d("CredResultReceiver", "resultCode - $resultCode, resultData - $resultData")
            if (resultCode == 1) {
                if (resultData.containsKey("credBlocks")) {
                    val cred = resultData.get("credBlocks") as HashMap<String, String>
                    Log.d("CredResultReceiver", "cred - ${cred.entries}")
                    if (cred.isNotEmpty()) {
                        val credentialList = mutableListOf<Credential>()
                        val controlsArray = controls.getJSONArray(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED)
                        for(i in 0 until controlsArray.length()) {
                            val subType = controlsArray.optJSONObject(i)?.optString(UPIConstants.CREDENTIAL_KEY_SUB_TYPE)
                            val type = controlsArray.optJSONObject(i)?.optString(UPIConstants.CREDENTIAL_KEY_TYPE)
                            val jsonString = cred[subType]
                            if (jsonString.isNullOrEmpty().not()) {
                                val json = JSONObject(jsonString)
                                val credTypeJson = JSONObject(json.optString(credType) ?: "")
                                val data = credTypeJson.optJSONObject("data")
                                val code = data?.optString("code")
                                val encData = data?.optString("encryptedBase64String")
                                val ki = data?.optString("ki")
                                val resultCred = Credential(
                                    credentialDataCode = code,
                                    credentialDataKi = ki,
                                    credentialDataValue = Base64.encodeToString(encData?.toByteArray(), Base64.NO_WRAP),
                                    credentialSubType = subType,
                                    credentialType = type
                                )
                                credentialList.add(resultCred)
                            }
                        }
                        onSuccess.invoke(credentialList)
                    } else {
                        onFailure.invoke(
                            PayUResponseCodes.PAYU_FAILED_STATUS,
                            PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                        )
                    }
                } else {
                    onFailure.invoke(
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                    )
                }
            } else {
                onFailure.invoke(
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                )
            }
        }
    }

    private fun getSetMpinCredentials(format: String?, credential: Credential): JSONObject {
        return if (format?.contains(UPIConstants.CREDENTIAL_FORMAT_1) == true) {
            JSONObject().apply {
                val jsonArray = JSONArray().apply {
                    put(
                        0, getCredJSON(
                            UPIConstants.OTP_CREDENTIAL_TYPE,
                            UPIConstants.OTP_CREDENTIAL_SUB_TYPE,
                            credential.otpCredentialType,
                            credential.otpCredentialLength
                        )
                    )
                    put(
                        1, getCredJSON(
                            UPIConstants.ATM_CREDENTIAL_TYPE,
                            UPIConstants.ATM_CREDENTIAL_SUB_TYPE,
                            credential.credentialDataType,
                            credential.credentialDataLength
                        )
                    )
                }
                put(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED, jsonArray)
            }
        } else if (format?.contains(UPIConstants.CREDENTIAL_FORMAT_2) == true) {
            JSONObject().apply {
                val jsonArray = JSONArray().apply {
                    put(
                        0, getCredJSON(
                            UPIConstants.OTP_CREDENTIAL_TYPE,
                            UPIConstants.OTP_CREDENTIAL_SUB_TYPE,
                            credential.otpCredentialType,
                            credential.otpCredentialLength
                        )
                    )
                    put(
                        1, getCredJSON(
                            UPIConstants.ATM_CREDENTIAL_TYPE,
                            UPIConstants.CARD_CREDENTIAL_SUB_TYPE,
                            credential.atmCredentialType,
                            credential.atmCredentialLength
                        )
                    )
                    put(
                        2, getCredJSON(
                            UPIConstants.ATM_CREDENTIAL_TYPE,
                            UPIConstants.ATM_CREDENTIAL_SUB_TYPE,
                            credential.credentialDataType,
                            credential.credentialDataLength
                        )
                    )
                }
                put(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED, jsonArray)
            }
        } else {
            JSONObject().apply {
                val jsonArray = JSONArray().apply {
                    put(
                        0, getCredJSON(
                            UPIConstants.OTP_CREDENTIAL_TYPE,
                            UPIConstants.OTP_CREDENTIAL_SUB_TYPE,
                            credential.otpCredentialType,
                            credential.otpCredentialLength
                        )
                    )
                    put(
                        1, getCredJSON(
                            UPIConstants.ATM_CREDENTIAL_TYPE,
                            UPIConstants.ATM_CREDENTIAL_SUB_TYPE,
                            credential.credentialDataType,
                            credential.credentialDataLength
                        )
                    )
                    put(
                        2, getCredJSON(
                            UPIConstants.OTP_CREDENTIAL_TYPE,
                            UPIConstants.AADHAAR_CREDENTIAL_TYPE,
                            credential.otpCredentialType,
                            credential.aadharCrdLength
                        )
                    )
                }
                put(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED, jsonArray)
            }
        }
    }

    private fun getMpinControls(credential: Credential): JSONObject {
        return JSONObject().apply {
            val jsonArray = JSONArray().apply {
                put(
                    0, getCredJSON(
                        UPIConstants.ATM_CREDENTIAL_TYPE,
                        UPIConstants.ATM_CREDENTIAL_SUB_TYPE,
                        credential.credentialDataType,
                        credential.credentialDataLength
                    )
                )
            }
            put(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED, jsonArray)
        }
    }

    private fun getChangeMPinControls(credential: Credential): JSONObject {
        return JSONObject().apply {
            val jsonArray = JSONArray().apply {
                put(
                    0, getCredJSON(
                        UPIConstants.ATM_CREDENTIAL_TYPE,
                        UPIConstants.ATM_CREDENTIAL_SUB_TYPE,
                        credential.credentialDataType,
                        credential.credentialDataLength
                    )
                )
                put(
                    1, getCredJSON(
                        UPIConstants.ATM_CREDENTIAL_TYPE,
                        UPIConstants.NEW_PIN_CREDENTIAL_SUB_TYPE,
                        credential.credentialDataType,
                        credential.credentialDataLength
                    )
                )
            }
            put(UPIConstants.CREDENTIAL_KEY_CRED_ALLOWED, jsonArray)
        }
    }
}