package com.payu.upiboltcore.npci

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.Address
import android.location.Geocoder
import android.location.Geocoder.GeocodeListener
import android.location.Location
import android.location.LocationManager
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import androidx.core.app.ActivityCompat
import com.payu.upiboltcore.InternalConfig
import com.payu.upiboltcore.constants.UPIConstants
import com.payu.upiboltcore.models.DeviceInfo
import com.payu.upiboltcore.utils.AnalyticsUtils
import com.payu.upiboltcore.utils.PayUSPUtils
import com.payu.upiboltcore.utils.Utils
import java.net.Inet4Address
import java.net.NetworkInterface
import java.util.Collections
import java.util.Locale


internal object DeviceInfoManager {

    var deviceInfo: DeviceInfo? = null
        private set
    private var appGenId: String? = null

    fun createDeviceInfo(context: Context, forceRefresh: Boolean? = false) {
        val deviceId = Utils.getDeviceId(context, forceRefresh)
        val appGenId = appGenId ?: Utils.getAppGenId(context)
        val location = getCurrentLocation(context)
        location?.let {
            getCityNameFromLocation(context, it) { city ->
                deviceInfo?.location = city ?: UPIConstants.DEVICE_INFO_LOCATION
            }
        }
        InternalConfig.sdkInitParams?.run {
            if (deviceId.isEmpty().not() && appId.isEmpty().not()
                && getPhone().isEmpty().not()
            ) {
                deviceInfo = DeviceInfo(
                    androidId = deviceId,
                    appGenId = appGenId,
                    appName = context.packageName,
                    appVersionCode = context.packageManager?.getPackageInfo(
                        context.packageName, 0)?.versionCode?.toString() ?: "0",
                    capability = UPIConstants.DEVICE_INFO_CAPABILITY,
                    deviceId = deviceId,
                    deviceType = UPIConstants.DEVICE_TYPE_MOB,
                    geoCode = location?.let { location ->
                        "${location.latitude},${location.longitude}"
                    } ?: UPIConstants.DEVICE_INFO_LAT_LNG,
                    ip = getIPAddress(context) ?: UPIConstants.DEVICE_INFO_IP,
                    location = UPIConstants.DEVICE_INFO_LOCATION,
                    mobileNo = getPhone(),
                    os = UPIConstants.DEVICE_INFO_OS,
                    selectedSimSlot = "",
                    simId = subscriptionId,
                    manufacturer = Build.MANUFACTURER,
                    model = Build.MODEL,
                    osVersion = Build.VERSION.RELEASE
                )
            }
        }
    }

    fun setAppGenId(context: Context, appGenId: String) {
        this.appGenId = appGenId
        PayUSPUtils.saveStringInSP(context, UPIConstants.APP_GEN_ID_KEY, appGenId)
        createDeviceInfo(context)
    }

    private fun getIPAddress(context: Context): String? {
        val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            return connectivityManager?.activeNetwork?.let { network ->
                val capabilities = connectivityManager.getNetworkCapabilities(network)
                if (capabilities?.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) == true) {
                    var ip: String? = null
                    connectivityManager.getLinkProperties(network)?.linkAddresses?.forEach { linkAddress ->
                        ip = if (linkAddress.address is Inet4Address) {
                            linkAddress.address.hostAddress
                        } else null
                    }
                    return@let ip
                } else if (capabilities?.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) == true) {
                    getMobileNetworkIPAddress()
                } else null
            }
        }
        return null
    }

    private fun getMobileNetworkIPAddress(): String? {
        try {
            val interfaces: List<NetworkInterface> =
                Collections.list(NetworkInterface.getNetworkInterfaces())
            for (intf in interfaces) {
                val addresses = Collections.list(intf.inetAddresses)
                for (address in addresses) {
                    if (!address.isLoopbackAddress) {
                        val sAddr = address.hostAddress
                        sAddr?.let {
                            val isIPv4 = sAddr.indexOf(':') < 0
                            return if (isIPv4)
                                sAddr
                            else {
                                val delim = sAddr.indexOf('%') // drop ip6 zone suffix
                                if (delim < 0)
                                    sAddr.uppercase(Locale.getDefault())
                                else
                                    sAddr.substring(0, delim).uppercase(Locale.getDefault())
                            }
                        }
                    }
                }
            }
        } catch (ignored: Exception) {
            return null
        }
        return null
    }

    private fun getCurrentLocation(context: Context): Location? {
        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)
            == PackageManager.PERMISSION_GRANTED &&
            ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION)
            == PackageManager.PERMISSION_GRANTED
        ) {
            try {
                val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as? LocationManager
                return locationManager?.getLastKnownLocation(LocationManager.GPS_PROVIDER) ?:
                locationManager?.getLastKnownLocation(LocationManager.NETWORK_PROVIDER)
            } catch (ex: Exception) {
                AnalyticsUtils.logEventNameForKibana(context, "getLocation", ex.message)
            }
        }
        return null
    }

    private fun getCityNameFromLocation(
        context: Context,
        location: Location,
        getCity: (city: String?) -> Unit
    ) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            Geocoder(context).getFromLocation(location.latitude, location.longitude, 1,
                object : GeocodeListener {
                    override fun onGeocode(addresses: MutableList<Address>) {
                        getCity.invoke(addresses[0].locality)
                    }

                    override fun onError(errorMessage: String?) {
                        super.onError(errorMessage)
                        getCity.invoke(null)
                    }
                })
        } else {
            val addresses = Geocoder(context)
                .getFromLocation(location.latitude, location.longitude, 1)
            getCity.invoke(addresses?.get(0)?.locality)
        }
    }
}