package com.payu.upiboltcore.utils

import android.app.ActivityManager
import android.content.Context
import android.os.Build
import android.util.Log
import com.payu.commonmodelssdk.constants.KibanaEvents
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.DD_MM_YY_HH_MM_SS_DATE_FORMAT
import com.payu.payuanalytics.analytics.factory.AnalyticsFactory
import com.payu.payuanalytics.analytics.model.AnalyticsType
import com.payu.payuanalytics.analytics.model.PayUAnalytics
import com.payu.upiboltcore.BuildConfig
import com.payu.upiboltcore.InternalConfig
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale


/**
 * AnalyticsUtils class for kibana logging
 * */
object AnalyticsUtils {

    enum class EventSeverity{
        High,
        Low
    }
    enum class EventType{
        Info,
        Error
    }
    private var eventIndex = 0

    private fun getAnalyticsJsonWithBasicProperties(context: Context): JSONObject {
        val analyticsJson = JSONObject()

        analyticsJson.put(KibanaEvents.PAYU_TIMESTAMP, getCurrentDateInDDMMYYHHMMSSFormat())
        analyticsJson.put(
            KibanaEvents.PAYU_SDK_BOLT_APPLICATION_PACKAGE,
            context.applicationContext?.packageName
        )
        try {
            val versionName = context.packageManager
                .getPackageInfo(context.packageName, 0).versionName
            analyticsJson.put(
                KibanaEvents.PAYU_SDK_BOLT_APPLICATION_VERSION,
                versionName
            )
        } catch (e: Exception) {
            Log.d("AnalyticsUtils", "getAnalyticsJsonWithBasicProperties1 " + e.message)
        }
        analyticsJson.put(KibanaEvents.LOGGING_SDK,BuildConfig.LIBRARY_PACKAGE_NAME)
        analyticsJson.put(KibanaEvents.PAYU_SDK_VERSION, BuildConfig.VERSION_NAME)
        analyticsJson.put(
            KibanaEvents.PAYU_MERCHANT_KEY, InternalConfig.sdkInitParams?.mKey
        )
        analyticsJson.put(
            KibanaEvents.PAYU_SDK_BOLT_REF_ID,
            InternalConfig.sdkInitParams?.refId
        )
        val deviceName = Build.MANUFACTURER + " " + Build.MODEL
        val deviceOsVersion = Build.VERSION.SDK_INT
        val deviceTotalMemory = getTotalMemoryInfo(context)
        val availMemory = getAvailableMemoryInfo(context)
        var memoryInfo: Long = 0
        try {
            memoryInfo = availMemory * 100 / deviceTotalMemory
        } catch (ex: Exception) {
            Log.d("AnalyticsUtils", "getAnalyticsJsonWithBasicProperties2 " + ex.message)

        }
        analyticsJson.put(
            KibanaEvents.PAYU_DEVICE_DETAILS,
            deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
        )
        return analyticsJson
    }

    private fun getTotalMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return (memoryInfo.totalMem / (1024 * 1024))
    }

    private fun getAvailableMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return memoryInfo.availMem / (1024 * 1024)
    }


    fun logEventNameForKibana(
        context: Context,
        eventName: String,
        eventData: String? = null,
        txnId: String? = null,
        timeDiff: Long? = null,
        refType: String? = null,
        code: String? = null,
        status: String? = null,
        screenName: String? = null,
        eventType : EventType = EventType.Info,
        eventSeverity : EventSeverity = EventSeverity.Low,
        referenceId: String? = null,
        amount: Double? = null,
        sdkName: String = PayUUpiConstant.PAYU_BOLT_SDK_NAME,
        name: String? = null
    ) {

        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_KEY, eventName)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_PLATFORM, PayUUpiConstant.ANDROID)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_TYPE, eventType.name)
            analyticsJson.put(KibanaEvents.PAYU_SDK_NAME, sdkName)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_SEVERITY, eventSeverity.name)
            eventData?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_VALUE, eventData) }
            timeDiff?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_TIME, it) }
            analyticsJson.put(KibanaEvents.TXN_ID, txnId ?: InternalConfig.initSdkId)
            referenceId?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_REFERENCE_ID, it) }
            refType?.let { analyticsJson.put(KibanaEvents.PAYU_SDK_REF_TYPE, it) }
            code?.let { analyticsJson.put(KibanaEvents.PAYU_CODE, it) }
            status?.let { analyticsJson.put(KibanaEvents.PAYU_STATUS, it) }
            screenName?.let { analyticsJson.put(KibanaEvents.PAYU_SCREEN_NAME, it) }
            amount?.let { analyticsJson.put(KibanaEvents.PAYU_AMOUNT, it) }
            name?.let { analyticsJson.put(KibanaEvents.PAYU_NAME, it) }
            logKibanaData(context, analyticsJson)
        } catch (e: Exception) {
            Log.d("PAYU", e.message.toString())
        }
    }

    private fun logKibanaData(context: Context, analyticsJson: JSONObject) {
        eventIndex++
        analyticsJson.put(KibanaEvents.PAYU_EVENTS_INDEX, eventIndex)
        val payuAnalytics =
            AnalyticsFactory(context).getAnalyticsClass(AnalyticsType.PAYU_ANALYTICS) as PayUAnalytics
//        Log.d("logKibanaData", analyticsJson.toString())
        payuAnalytics.log(analyticsJson.toString())
    }

    private fun getCurrentDateInDDMMYYHHMMSSFormat(): String {
        val sdf = SimpleDateFormat(DD_MM_YY_HH_MM_SS_DATE_FORMAT, Locale.ENGLISH)
        return sdf.format(Date())
    }

    fun logKibana(
        context: Context,
        name: String,
        data: String,
        isError: Boolean,
        severity: EventSeverity = EventSeverity.Low,
        refType: String,
        referenceId: String? = null,
        txnId: String? = null,
        time: Long? = null,
        code: String? = null,
        status: String? = null,
        amount: Double? = null
    ) {
        logEventNameForKibana(
            context, "${PayUUpiConstant.PAYU_BOLT_CORE_API}$name",
            data, txnId, time, refType,
            code = code, status = status,
            eventType = if (isError)
                EventType.Error
            else
                EventType.Info,
            eventSeverity = severity,
            referenceId = referenceId,
            amount = amount,
            sdkName = PayUUpiConstant.PAYU_BOLT_CORE_SDK_NAME
        )
    }
}