package com.payu.upiboltcore.utils

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import android.provider.Settings
import android.security.KeyPairGeneratorSpec
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64
import com.payu.upiboltcore.constants.UPIConstants
import java.nio.charset.StandardCharsets
import java.security.KeyStore
import java.security.MessageDigest
import java.text.DecimalFormat
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.UUID
import java.util.regex.Matcher
import java.util.regex.Pattern
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.SecretKey
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec


internal object Utils {

    fun getDeviceId(context: Context, forceRefresh: Boolean? = false): String {
        return if (forceRefresh == true) {
            generateDeviceId(context)
        } else {
            PayUSPUtils.getStringFromSP(context, UPIConstants.DEVICE_ID_KEY) ?: kotlin.run {
                generateDeviceId(context)
            }
        }
    }

    private fun generateDeviceId(context: Context): String {
        var deviceId = Settings.Secure.getString(
            context.contentResolver, Settings.Secure.ANDROID_ID
        )
        if (deviceId.isNullOrEmpty()) {
            deviceId = generateUUID()
        }
        PayUSPUtils.saveStringInSP(context, UPIConstants.DEVICE_ID_KEY, deviceId)
        return deviceId
    }

    private fun generateUUID(): String {
        val id = UUID.randomUUID().toString()
        return id.replace("-", "")
    }

    fun getAppGenId(context: Context): String {
        return PayUSPUtils.getStringFromSP(context, UPIConstants.APP_GEN_ID_KEY) ?: kotlin.run {
            val appGenId = generateUUID()
            PayUSPUtils.saveStringInSP(context, UPIConstants.APP_GEN_ID_KEY, appGenId)
            appGenId
        }
    }

    fun getUniqueTxnId() = "payu${System.currentTimeMillis()}${System.currentTimeMillis()}${System.currentTimeMillis()}".substring(0, 35)

    fun encrypt(key: ByteArray, data: ByteArray, ivData: ByteArray? = null ,mode: Int = Cipher.ENCRYPT_MODE): ByteArray {
        val secretKeySpec = SecretKeySpec(key, "AES")
        return encrypt(secretKeySpec, data, ivData, mode)
    }

    private fun encrypt(key: SecretKey, data: ByteArray, ivData: ByteArray? = null, mode: Int = Cipher.ENCRYPT_MODE): ByteArray {
        val ivSpec: IvParameterSpec? = ivData ?.let { IvParameterSpec(ivData) }
        val cipher: Cipher = Cipher.getInstance("AES/GCM/NoPadding")
        cipher.init(mode, key, ivSpec)
        return cipher.doFinal(data)
    }

//    private fun decrypt(key: SecretKey, encryptedData: ByteArray, ivData: ByteArray? = null): ByteArray {
//        return encrypt(key, encryptedData, ivData, Cipher.DECRYPT_MODE)
//    }

    private fun encryptWithIv(context: Context, alias: String, data: String): String? {
        try {
            val key = getKey(context, alias)
            if (key != null) {
                val cipher = Cipher.getInstance("AES/GCM/NoPadding")
                cipher.init(Cipher.ENCRYPT_MODE, key)
                val iv = cipher.iv
                val encrypted = cipher.doFinal(data.toByteArray(StandardCharsets.UTF_8))

                val ivAndCiphertext = ByteArray(iv.size + encrypted.size)
                System.arraycopy(iv, 0, ivAndCiphertext, 0, iv.size)
                System.arraycopy(encrypted, 0, ivAndCiphertext, iv.size, encrypted.size)
                return Base64.encodeToString(ivAndCiphertext, Base64.DEFAULT)
            }
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
        return null
    }

    private fun decryptWithIv(context: Context, alias: String, data: String?): String? {
        try {
            val key = getKey(context, alias)
            if (key != null) {
                val ivAndCiphertext = Base64.decode(data, Base64.DEFAULT)

                val iv = ByteArray(12)
                System.arraycopy(ivAndCiphertext, 0, iv, 0, iv.size)
                val encryptedData = ByteArray(ivAndCiphertext.size - iv.size)
                System.arraycopy(ivAndCiphertext, iv.size, encryptedData, 0, encryptedData.size)

                val cipher = Cipher.getInstance("AES/GCM/NoPadding")
                val spec = GCMParameterSpec(128, iv)
                cipher.init(Cipher.DECRYPT_MODE, key, spec)

                val decrypted = cipher.doFinal(encryptedData)
                return String(decrypted, StandardCharsets.UTF_8)
            }
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
        return null
    }

    private fun getKey(context: Context, alias: String): SecretKey? {
        try {
            val keystore = KeyStore.getInstance("AndroidKeyStore")
            keystore.load(null)
            val keyEntry = keystore.getEntry(alias, null) as? KeyStore.SecretKeyEntry
            return keyEntry?.secretKey ?: createKey(context, alias)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return null
    }

    private fun createKey(context: Context, alias: String): SecretKey? {
        try {
            val generator = KeyGenerator.getInstance("AES", "AndroidKeyStore")
            val spec = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                KeyGenParameterSpec.Builder(
                    alias, KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
                ).setBlockModes(KeyProperties.BLOCK_MODE_GCM)
                    .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
                    .setKeySize(256)
                    .build()
            } else {
                KeyPairGeneratorSpec.Builder(context)
                    .setAlias(alias)
                    .build()
            }
            generator.init(spec)
            return generator.generateKey()
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return null
    }

    fun getEncryptedStringFromSP(context: Context, key: String, alias: String): String? {
        return PayUSPUtils.getStringFromSP(context, key)?.let {
            decryptWithIv(context, alias, it)
        }
    }

    fun saveEncryptedStringInSP(context: Context, key: String, value: String, alias: String) {
        encryptWithIv(context, alias, value)?.let {
            PayUSPUtils.saveStringInSP(context, key, it)
        }
    }

    fun generateSHA256Hash(value: String, randomBytes: ByteArray): ByteArray {
        val md = MessageDigest.getInstance("SHA-256")
        md.update(randomBytes)
        return md.digest(value.toByteArray())
    }

    fun hexStringToByteArray(s: String): ByteArray {
        val b = ByteArray(s.length / 2)
        for (i in b.indices) {
            val index = i * 2
            val v = s.substring(index, index + 2).toInt(16)
            b[i] = v.toByte()
        }
        return b
    }

    fun getDateBeforeDays(date: Date, days: Int): Date {
        val beforeDays: Long = days * 1000L * 60L * 60L * 24L
        val newTime = date.time - beforeDays
        return Date(newTime)
    }

    fun formatMobileNumber(mobile: String): String {
        if (mobile.length == 10) {
            return "91$mobile"
        } else if (mobile.startsWith("+")) {
            return mobile.substringAfter("+")
        }
        return mobile
    }

    fun convertStringDateToFormat(
        inputDate: String,
        outputDateFormat: String,
        inputDateFormat: String = UPIConstants.PAYU_DD_MM_YYYY
    ): String {
        val inputFormat = SimpleDateFormat(inputDateFormat, Locale.getDefault())
        val outputFormat = SimpleDateFormat(outputDateFormat, Locale.getDefault())
        return try {
            val date = inputFormat.parse(inputDate)
            date?.let { outputFormat.format(date) } ?: ""
        } catch (e: Exception) {
            ""
        }
    }
    fun isWifiConnected(context: Context){
        val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

        val networkAvailability = cm.getNetworkCapabilities(cm.activeNetwork)
        if(networkAvailability !=null && networkAvailability.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) && networkAvailability.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED))
        {
            //has network
            if (networkAvailability.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) { //wifi
            }
            else if (networkAvailability.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) { //cellular
            }
        }
    }
    fun validateVPA(upi: String): Boolean {
        val string = upi.substring(0, upi.indexOf("@hdfc"))
        if (string.isEmpty()) {
            return false
        }

        val upiPatt: Pattern = Pattern.compile("^[0-9A-Za-z.-]{2,256}$")
        val matcher: Matcher = upiPatt.matcher(string)
        return matcher.matches()
    }

    fun getMaskedMobileNumber(number : String) : String{
        val regex = ".*\\d(?=\\d{4})".toRegex()
        return number.replace(regex,"*")
    }

    fun getTimeDifferenceInMilliSeconds(apiCalledTime: Long): Long {
        return System.currentTimeMillis() - apiCalledTime
    }

    fun formatAmount(amount: String?): String? {
        if (amount.isNullOrEmpty()) return null
        return try {
            val format = DecimalFormat("0.00")
            format.format(amount.toDouble())
        } catch (ex: Exception) {
            null
        }
    }

    fun convertTxnHistoryDateFormat(
        inputDate: String
    ): String {
        val inputDateFormat = UPIConstants.TXN_HISTORY_INPUT_DATE_FORMAT
        val outputDateFormat = UPIConstants.TXN_HISTORY_OUTPUT_DATE_FORMAT
        val inputFormat = SimpleDateFormat(inputDateFormat, Locale.getDefault())
        val outputFormat = SimpleDateFormat(outputDateFormat, Locale.getDefault())
        return try {
            val date = inputFormat.parse(inputDate)
            date?.let { outputFormat.format(date) } ?: ""
        } catch (e: Exception) {
            ""
        }
    }

    fun getMaskedString(input: String?): String? {
        input?.let {
            if (input.length < 5) return input
            val midpoint = input.length - 5
            return "**" + input.substring(midpoint)
        }
        return null
    }
}