package com.payu.upibolt

import android.content.Context
import androidx.annotation.Keep
import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.enums.PluginType
import com.payu.commonmodelssdk.listeners.PayUHashGenerationListener
import com.payu.upibolt.listeners.PluginInterface
import com.payu.upibolt.models.PayUUPIBoltConfig
import com.payu.upibolt.payUAxisImpl.PayUAxisPlugin
import com.payu.upibolt.payUHDFCImpl.HDFCPluginImpl
import com.payu.upibolt.utils.AnalyticsUtils
import com.payu.upibolt.utils.InternalConfig
import com.payu.upibolt.utils.Utils.isAxisDependencyAvailable
import com.payu.upibolt.utils.Utils.isHDFCDependencyAvailable
import org.json.JSONObject

@Keep
class PayUUPIBolt private constructor() {

    var core: PluginInterface? = null
        private set

    @Keep
    companion object {
        @Volatile
        private var bolt: PayUUPIBolt? = null

        @Keep
        @JvmStatic
        fun getInstance(
            activity: FragmentActivity,
            config: PayUUPIBoltConfig,
            hashGenerationListener: PayUHashGenerationListener
        ) = bolt ?: synchronized(this) { // synchronized to avoid concurrency problem
            bolt ?: PayUUPIBolt().also {
                InternalConfig.config = config
                InternalConfig.appId = activity.applicationInfo.packageName
                bolt = it
                var pluginType = PluginType.AXIS
                if (config.pluginTypes.isNotEmpty()) {
                    pluginType = config.pluginTypes.first()
                }
                InternalConfig.pluginType = pluginType
                bolt?.core = bolt?.initCore(activity, pluginType)
                InternalConfig.payUHashGenerationListener = hashGenerationListener

                AnalyticsUtils.logEventNameForKibana(
                    activity,
                    "${PayUUpiConstant.PAYU_API}${CLConstant.PAYU_CT_BOLT_SDK_INIT}",
                    prepareInitParamsJson(config).toString(),
                    eventType = AnalyticsUtils.EventType.Info,
                    eventSeverity = AnalyticsUtils.EventSeverity.Low,
                    status = CLConstant.PAYU_CT_INITIATED
                )
                AnalyticsUtils.setCTLogsForUPIWithData(
                    activity,
                    CLConstant.PAYU_CT_BOLT_SDK_INIT,
                    status = CLConstant.PAYU_CT_INITIATED
                )
            }
        }

        fun reset(context: Context) {
            InternalConfig.clearAll()
            bolt?.core?.clearCache()
            bolt = null
            AnalyticsUtils.logEventNameForKibana(
                context,
                "${PayUUpiConstant.PAYU_API}${CLConstant.PAYU_CT_BOLT_SDK_RESET}",
                "",
                eventType = AnalyticsUtils.EventType.Info,
                eventSeverity = AnalyticsUtils.EventSeverity.Low,
                status = CLConstant.PAYU_CT_RESET
            )
            AnalyticsUtils.setCTLogsForUPIWithData(
                context,
                CLConstant.PAYU_CT_BOLT_SDK_RESET,
                status = CLConstant.PAYU_CT_RESET
            )
        }

        private fun prepareInitParamsJson(sdkInitParams: PayUUPIBoltConfig?): JSONObject {
            val jsonObject = JSONObject()
            jsonObject.put(PayUUpiConstant.PAYU_MERCHANT_NAME, sdkInitParams?.merchantKey)
            jsonObject.put(PayUUpiConstant.PAYU_MERCHANT_KEY, sdkInitParams?.merchantKey)
            jsonObject.put(PayUUpiConstant.PAYU_EMAIL_ID, sdkInitParams?.email)
            if (sdkInitParams?.pluginTypes?.isEmpty() == false) {
                jsonObject.put(PayUUpiConstant.PAYU_PLUGIN_TYPE, sdkInitParams.pluginTypes.first())
            }
            jsonObject.put(PayUUpiConstant.PAYU_IS_PROD, sdkInitParams?.isProduction)
            jsonObject.put(PayUUpiConstant.PAYU_SDK_BOLT_REF_ID, sdkInitParams?.refId)
            return jsonObject
        }
    }

    fun isRegistered(activity: FragmentActivity, pg: String, callback: (Boolean) -> Unit) {
        if (isDependencyAvailable(pg).not()) {
            callback.invoke(false)
            return
        }
        val pluginInterface = initCore(activity, PluginType.valueOf(pg))
        callback.invoke(pluginInterface.getRegisteredMobile().isNullOrEmpty().not())
    }

    fun clearData(activity: FragmentActivity, pg: String) {
        if (isDependencyAvailable(pg).not()) {
            return
        }
        val pluginInterface = initCore(activity, PluginType.valueOf(pg))
        pluginInterface.clearData()
    }

    private fun isDependencyAvailable(pg: String): Boolean {
        if (pg == PluginType.AXIS.pluginName && isAxisDependencyAvailable().not()) {
            return false
        }
        if ((pg == PluginType.HDFC.pluginName || pg == PluginType.BHIM.pluginName)
            && isHDFCDependencyAvailable().not()) {
            return false
        }
        return true
    }

    private fun initCore(
        activity: FragmentActivity,
        pluginType: PluginType
    ): PluginInterface {
        return when (pluginType) {
            PluginType.AXIS -> {
                PayUAxisPlugin(activity)
            }

            PluginType.HDFC, PluginType.BHIM -> {
                HDFCPluginImpl(activity)
            }
        }
    }
}