package com.payu.upibolt.network

import android.content.Context
import com.payu.commonmodelssdk.constants.ApiConstant
import com.payu.commonmodelssdk.constants.ApiConstant.PAYU_POST
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_API
import com.payu.commonmodelssdk.listeners.PayUAsyncTaskResponse
import com.payu.commonmodelssdk.model.PayUNetworkData
import com.payu.upibolt.utils.AnalyticsUtils
import com.payu.upibolt.utils.InternalConfig
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import org.json.JSONObject
import java.io.IOException
import java.util.concurrent.TimeUnit

/**
 * HttpApiCall class for all api calls
 * */

internal class HttpApiCall(private val context: Context) {
    private var call: Call? = null

    fun makePostAPICall(
        apiName: String,
        httpMethod: String = PAYU_POST,
        payUNetworkData: PayUNetworkData,
        payUAsyncTaskResponse: PayUAsyncTaskResponse,
        timeOut:Long = 60
    ) {
        val eventData = prepareApiRequestEventData(apiName,httpMethod, payUNetworkData)
        AnalyticsUtils.logEventNameForKibana(
            context,
            "${PAYU_API}$apiName",
            eventData,
            eventType = AnalyticsUtils.EventType.Info,
            refType = PayUUpiConstant.SDK_PAYU_REQUEST
        )

        if (httpMethod == PAYU_POST && payUNetworkData.postRequest.isNullOrEmpty()) {
            payUAsyncTaskResponse.onFailure(
                PayUResponseCodes.PAYU_FAILED_STATUS,
                PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE
            )
            return
        }
        val okHttpClientBuilder: OkHttpClient.Builder = OkHttpClient.Builder()
        val client = okHttpClientBuilder.connectTimeout(timeOut, TimeUnit.SECONDS)
            .writeTimeout(timeOut, TimeUnit.SECONDS)
            .readTimeout(timeOut, TimeUnit.SECONDS).build()
        val mediaType: MediaType = payUNetworkData.contentType.toMediaType()

        val request: Request.Builder = Request.Builder()

        request.addHeader(ApiConstant.PAYU_CONTENT_TYPE, payUNetworkData.contentType)
        if (httpMethod == PAYU_POST) {
            val body: RequestBody? = payUNetworkData.postRequest?.toRequestBody(mediaType)
            if (body != null) {
                request.post(body)
            } else {
                payUAsyncTaskResponse.onFailure(
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE
                )
            }
            request.url(payUNetworkData.url)
        } else {
            var queryRequest = ""
            payUNetworkData.getRequest?.let {
                for ((key, value) in it) {
                    queryRequest += "$key=$value&"
                }
            }
            request.url(payUNetworkData.url + "?${queryRequest.dropLast(1)}")
        }

        payUNetworkData.headerMap?.let {
            for ((key, value) in it) {
                request.addHeader(key, value)
            }
        }
        request.tag(apiName)

        call = client.newCall(request.build())
        call?.let { InternalConfig.activeApiCalls.add(it) }
        call?.enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                payUAsyncTaskResponse.onFailure(
                    PayUResponseCodes.PAYU_RUNTIME_ERROR,
                    PayUResponseMessages.RUNTIME_ERROR_MESSAGE
                )
                InternalConfig.activeApiCalls.remove(call)
            }

            override fun onResponse(call: Call, response: Response) {
                if (response.code == 200) {
                    val responseString = response.body?.string()
                    if (responseString != null) {
                        payUAsyncTaskResponse.onSuccess(responseString)
                    } else {
                        payUAsyncTaskResponse.onFailure(response.code, response.message)
                    }
                } else {
                    try {
                        response.body?.string()?.let {
                            val json = JSONObject(it)
                            var message = "";
                            message = if (json.has(PayUUpiConstant.RES_PARAM_MESSAGE)) {
                                json.optString(PayUUpiConstant.RES_PARAM_MESSAGE)
                            } else {
                                json.optString(PayUUpiConstant.RES_PARAM_MSG)
                            }
                            payUAsyncTaskResponse.onFailure(response.code, message)
                        } ?: kotlin.run {
                            payUAsyncTaskResponse.onFailure(response.code, response.message)
                        }
                    } catch (ex: Exception) {
                        payUAsyncTaskResponse.onFailure(response.code, response.message)
                    }
                }
                InternalConfig.activeApiCalls.remove(call)
            }
        })
    }

    private fun prepareApiRequestEventData(
        apiName: String,
        httpMethod: String,
        payUNetworkData: PayUNetworkData
    ): String {
        var body = ""
        if (apiName == PayUUpiConstant.PAYU_AXIS_AUTH_TOKEN) {
            body = AnalyticsUtils.prepareAuthTokenParamsJson(payUNetworkData.postRequest)
        }
        val eventData =
            PayUUpiConstant.SDK_PAYU_REQUEST + PayUUpiConstant.SDK_PAYU_URL + payUNetworkData.url + " " + if (httpMethod == PAYU_POST) {
                body
            } else {
                ""
            } + PayUUpiConstant.SDK_HTTP_METHOD + httpMethod
        return eventData
    }
}