package com.payu.upibolt.utils

import android.Manifest
import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.telephony.SubscriptionInfo
import android.telephony.SubscriptionManager
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.fragment.app.FragmentActivity
import com.olive.upi.transport.model.Account
import com.olive.upi.transport.model.BeneVpa
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.ApiConstant
import com.payu.commonmodelssdk.constants.KibanaEvents
import com.payu.commonmodelssdk.constants.KibanaEvents.PAYU_INIT_PAYMENT
import com.payu.commonmodelssdk.constants.KibanaEvents.PAYU_PAYMENT_HASH_RECEIVED
import com.payu.commonmodelssdk.constants.KibanaEvents.PAYU_REQUEST_PAYMENT_HASH
import com.payu.commonmodelssdk.constants.KibanaEvents.PAYU_REQUEST_VERIFY_PAYMENT_HASH
import com.payu.commonmodelssdk.constants.KibanaEvents.PAYU_VERIFY_PAYMENT_HASH_RECEIVED
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUResponseTypes.REQUEST_CHECK_PAYMENT_STATUS
import com.payu.commonmodelssdk.constants.PayUResponseTypes.REQUEST_PAY
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.AXIS_SDK_RESPONSE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.HASH
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_APP_ID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_CARRIER_NAME
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_CODE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_EMAIL_ID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_GET_SIM_INFO
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_IS_PROD
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_IS_UPI_BOLT_AVAILABLE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_MERCHANT_KEY
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_MOBILE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_PG
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_REQUEST_UNKNOWN
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_SDK_RESPONSE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_TXN_ID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_UNQ_TXNID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PREF_KEY_SUBSCRIPTION_ID
import com.payu.commonmodelssdk.constants.payuResponseTypeMap
import com.payu.commonmodelssdk.enums.PluginType
import com.payu.commonmodelssdk.enums.UpiHashCommand
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.listeners.OTPVerificationInterface
import com.payu.commonmodelssdk.listeners.PayUHashGeneratedListener
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.commonmodelssdk.listeners.ApiFailedCallback
import com.payu.commonmodelssdk.model.PayUSimInfo
import com.payu.commonmodelssdk.model.PgDetails
import com.payu.commonmodelssdk.model.PgDetailsRequest
import com.payu.commonmodelssdk.model.request.CancelApiRequest
import com.payu.commonmodelssdk.model.request.PayUUPIBoltPaymentParams
import com.payu.commonmodelssdk.model.request.PayUUPIParams
import com.payu.commonmodelssdk.model.request.VerifyApiRequest
import com.payu.commonmodelssdk.model.response.PayUAccountDetail
import com.payu.commonmodelssdk.model.response.PayUCustomerBankAccounts
import com.payu.commonmodelssdk.model.response.InitiatePaymentResponse
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upibolt.listeners.UpiInitiateCallback
import com.payu.upibolt.network.CancelApiTask
import com.payu.upibolt.network.InitiatePaymentTask
import com.payu.upibolt.network.PgDetailsTask
import com.payu.upibolt.network.VerifyPaymentTask
import com.payu.upiboltcore.models.RegisterVPAResponseResult
import org.json.JSONArray
import org.json.JSONObject
import java.util.Locale

abstract class CommonPluginImpl(protected val activity: FragmentActivity) {

    protected var upiInitiateCallback: UpiInitiateCallback? = null

    protected var payuStartTimeMap: HashMap<String, Long> = HashMap()

    protected var callBack: HashMap<Int, PayUUPIBoltCallBack>  = HashMap()
    protected var startTime = System.currentTimeMillis()
    protected var axisStartTimeMap: HashMap<String, Long> = HashMap()
    protected var referenceId: String? = null

    protected var initiatePayResponse: InitiatePaymentResponse? = null
    private var amount: String? = null // only for logging

    abstract fun getRegisteredMobile(): String?
    abstract fun initiateWrapperSDK(
        callback: UpiInitiateCallback,
        otpVerificationInterface: OTPVerificationInterface? = null,
        apiFailureCallback: ApiFailedCallback? = null
    )
    abstract fun isSessionValid(): Boolean
    abstract fun makePayment(payUUPIParams: PayUUPIParams)

    protected fun checkDeviceStatus(
        mobileNo: String,
        subscriptionId: String
    ): PayUUPIBoltResponse {

        val response = PayUUPIBoltResponse(
            PayUResponseTypes.CHECK_DEVICE_BINDING_STATUS,
            PayUResponseCodes.PAYU_DEVICE_BINDING_FAILED,
            PayUResponseMessages.PAYU_DEVICE_BINDING_FAILED_MESSAGE
        )

        if (mobileNo.isEmpty() || subscriptionId.isEmpty()) {
            val param = if (mobileNo.isEmpty())
                PayUUpiConstant.PAYU_MOBILE
            else if (subscriptionId.isEmpty())
                PayUUpiConstant.PAYU_SUBSCRIPTION_ID
            else ""
            response.code = PayUResponseCodes.PAYU_FAILED_STATUS
            response.message = PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE + param
            return response
        }

        val subInfoList = getSubscriptionInfoList()
        if (subInfoList?.isEmpty() == true) {
            response.code = PayUResponseCodes.PAYU_SIM_INFO_NOT_AVAILABLE
            response.message = PayUResponseMessages.PAYU_SIM_INFO_NOT_AVAILABLE_MESSAGE
            return response
        }

        val simInfoPref = getSimInfoInSharedPref()
        val deviceSimInfo = subInfoList?.let { getSimInfoList(it) }
        val isValidSimInfo = getValidStoredSimInfo(simInfoPref, deviceSimInfo)

        if (isValidSimInfo.isNullOrEmpty() || getRegisteredMobile() == null
            || getRegisteredMobile() != mobileNo
        ) {
            return response
        }

        response.code = PayUResponseCodes.PAYU_SUCCESS_STATUS
        response.message = null
        return response
    }

    protected fun validateSDKSession(
        responseType: Int,
        callback: UpiInitiateCallback,
        otpVerificationInterface: OTPVerificationInterface? = null,
        apiFailureCallback: ApiFailedCallback? = null
    ) {
        if (InternalConfig.config == null) {
            callback.onSdKError(
                PayUResponseCodes.PAYU_FAILED_STATUS,
                PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE
            )
            return
        }
        val response =
            checkDeviceStatus(InternalConfig.mobile ?: "", InternalConfig.subscriptionId ?: "")
        response.responseType = responseType
        if (response.code != PayUResponseCodes.PAYU_SUCCESS_STATUS) {
            callback.onSdKError(response.code, response.message)
        } else if (isSessionValid()) {
            callback.onSDKSuccess()
        } else {
            //Do refresh token
            initiateWrapperSDK(callback, otpVerificationInterface, apiFailureCallback)
        }
    }

    open fun isUpiBoltEnabled(payUUpiProCallBack: PayUUPIBoltCallBack) {
        referenceId = Utils.generateUUID()
        val startTime = System.currentTimeMillis()
        InternalConfig.pgDetails = PgDetails(
            pgName = "AXIS",
            pgId = "157",
            merchChanId = "AUTOPEAPP1011050256763",
            mcc = "4112",
            merchId = "AUTOPE10110501",
            mKey = "smsplus",
            pluginKey = "",
            bankHandle = "axisbank"
        )
        if (InternalConfig.pgDetails != null) {
            val payUUPIBoltResponse = PayUUPIBoltResponse(
                PayUResponseTypes.REQUEST_UPI_BOLT,
                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                PayUResponseMessages.PAYU_SUCCESS_MESSAGE,
                InternalConfig.pgDetails
            )
            val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
            payUUpiProCallBack.onPayUSuccess(payUUPIBoltResponse)
            logSuccessResponse(
                PAYU_IS_UPI_BOLT_AVAILABLE,
                preparePgDetailsResponseJson(InternalConfig.pgDetails).toString(), timeDiff,
                referenceId = referenceId
            )
            return
        }

        val hashString = UpiAuthHeaderHelper.getUPIHashString(
            UpiHashCommand.PgDetails, null,
            InternalConfig.config?.merchantKey ?: ""
        )

        logPayUFunctionsRequest(
            PayUResponseTypes.REQUEST_UPI_BOLT,
            KibanaEvents.PAYU_REQUEST_PG_DETAILS_HASH + hashString.entries.toString(),
            referenceId = referenceId
        )
        InternalConfig.payUHashGenerationListener?.generateHash(hashString,
            object : PayUHashGeneratedListener {
                override fun onHashGenerated(map: HashMap<String, String?>) {
                    val name = map[PayUUpiConstant.HASH_NAME]
                    InternalConfig.config?.run {
                        val pgName =
                            if (InternalConfig.pluginType == PluginType.BHIM)
                                InternalConfig.config?.issuingBanks?.firstOrNull() ?: ""
                            else
                                InternalConfig.pluginType?.name ?: ""
                        val pgDetailsRequest =
                            PgDetailsRequest(merchantKey, pgName, map[name])
                        logPayUFunctionsRequest(
                            PayUResponseTypes.REQUEST_UPI_BOLT,
                            preparePgDetailsRequestJson(pgDetailsRequest).toString(),
                            referenceId = referenceId
                        )
                        val pgDetailsTask = PgDetailsTask(activity)
                        pgDetailsTask.callApi(pgDetailsRequest, callback = object :
                            ApiBaseCallback {
                            override fun onApiSuccess(response: Any) {
                                InternalConfig.pgDetails = response as PgDetails
                                val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
                                logSuccessResponse(
                                    PAYU_IS_UPI_BOLT_AVAILABLE,
                                    preparePgDetailsResponseJson(InternalConfig.pgDetails).toString(),
                                    timeDiff,
                                    referenceId = referenceId
                                )
                                val payUUPIBoltResponse = PayUUPIBoltResponse(
                                    PayUResponseTypes.REQUEST_UPI_BOLT,
                                    PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                    PayUResponseMessages.PAYU_SUCCESS_MESSAGE,
                                    InternalConfig.pgDetails
                                )
                                payUUpiProCallBack.onPayUSuccess(payUUPIBoltResponse)
                            }

                            override fun onApiError(errorCode: Int, errorMessage: String) {
                                val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
                                logFailureResponse(
                                    PAYU_IS_UPI_BOLT_AVAILABLE, errorCode, errorMessage, timeDiff,
                                    referenceId = referenceId
                                )
                                val payUUPIBoltResponse = PayUUPIBoltResponse(
                                    PayUResponseTypes.REQUEST_UPI_BOLT,
                                    errorCode, errorMessage
                                )
                                payUUpiProCallBack.onPayUFailure(payUUPIBoltResponse)
                            }
                        })
                    }
                }
            })
    }

    open fun getSubscriberInfo(mobile: String, payUUpiProCallBack: PayUUPIBoltCallBack) {
        referenceId = Utils.generateUUID()
        val startTime = System.currentTimeMillis()
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1) {
            if (hasSmsAndPhonePermissions().not()) {
                val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
                logFailureResponse(
                    PAYU_GET_SIM_INFO,
                    PayUResponseCodes.PERMISSION_MISSING_ERROR_CODE,
                    PayUResponseMessages.PAYU_PERMISSION_ERROR_MESSAGE,
                    timeDiff,
                    referenceId = referenceId
                )
                val PayUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_SIM_INFO,
                    PayUResponseCodes.PERMISSION_MISSING_ERROR_CODE,
                    PayUResponseMessages.PAYU_PERMISSION_ERROR_MESSAGE
                )
                payUUpiProCallBack.onPayUFailure(PayUUPIBoltResponse)
            } else {
                val simInfoList = ArrayList<PayUSimInfo>()
                val storedSimInfo = getSimInfoInSharedPref()
                val subInfoList = getSubscriptionInfoList()
                val deviceSimInfo =  subInfoList?.let { getSimInfoList(subInfoList) }
                if (storedSimInfo.mobileNumber.isNotEmpty() && storedSimInfo.subscriberId.isNotEmpty()) {
                    val validSimInfo = getValidStoredSimInfo(storedSimInfo, deviceSimInfo)
                    if (!validSimInfo.isNullOrEmpty()) {
                        simInfoList.addAll(validSimInfo)
                    } else if(deviceSimInfo?.isNotEmpty() == true) {
                        simInfoList.addAll(deviceSimInfo)
                    }
                } else if(deviceSimInfo?.isNotEmpty() == true) {
                    simInfoList.addAll(deviceSimInfo)
                } else if (subInfoList?.isNotEmpty() == true) {
                    simInfoList.add(PayUSimInfo(mobileNumber = Utils.formatMobileNumber(mobile), null, subInfoList[0].subscriptionId.toString(), null))
                }

                val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
                if (simInfoList.isEmpty()) {
                    logFailureResponse(
                        PAYU_GET_SIM_INFO,
                        PayUResponseCodes.PAYU_SIM_INFO_NOT_AVAILABLE,
                        PayUResponseMessages.PAYU_SIM_INFO_NOT_AVAILABLE_MESSAGE,
                        timeDiff,
                        referenceId = referenceId
                    )
                    val PayUUPIBoltResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_SIM_INFO,
                        PayUResponseCodes.PAYU_SIM_INFO_NOT_AVAILABLE,
                        PayUResponseMessages.PAYU_SIM_INFO_NOT_AVAILABLE_MESSAGE
                    )
                    payUUpiProCallBack.onPayUFailure(PayUUPIBoltResponse)
                } else
                    storeSlotSize(simInfoList.size.toString())
                val payUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_SIM_INFO,
                    PayUResponseCodes.PAYU_SUCCESS_STATUS,
                    PayUResponseMessages.PAYU_SUCCESS_MESSAGE,
                    simInfoList
                )
                logSuccessResponse(
                    PAYU_GET_SIM_INFO,
                    simInfoList.map {
                        PayUSimInfo(Utils.getMaskedMobileNumber(it.mobileNumber),
                            it.slotIndex, it.subscriberId, it.carrierName)
                    }.joinToString(","),
                    timeDiff,
                    referenceId = referenceId
                )
                payUUpiProCallBack.onPayUSuccess(payUUPIBoltResponse)
            }
        } else {
            val timeDiff = Utils.getTimeDifferenceInMilliSeconds(startTime)
            logFailureResponse(
                PAYU_GET_SIM_INFO,
                PayUResponseCodes.PAYU_DEVICE_NOT_SUPPORTED_CODE,
                PayUResponseMessages.PAYU_DEVICE_NOT_SUPPORTED,
                timeDiff,
                referenceId = referenceId
            )
            val PayUUPIBoltResponse = PayUUPIBoltResponse(
                PayUResponseTypes.REQUEST_SIM_INFO,
                PayUResponseCodes.PAYU_DEVICE_NOT_SUPPORTED_CODE,
                PayUResponseMessages.PAYU_DEVICE_NOT_SUPPORTED
            )
            payUUpiProCallBack.onPayUFailure(PayUUPIBoltResponse)
        }
    }

    @SuppressLint("MissingPermission")
    protected fun getSubscriptionInfoList(): List<SubscriptionInfo>? {
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1) {

            val sm: SubscriptionManager? =
                ContextCompat.getSystemService(
                    activity.baseContext,
                    SubscriptionManager::class.java
                )

            if (ActivityCompat.checkSelfPermission(
                    activity,
                    Manifest.permission.READ_PHONE_STATE
                ) == PackageManager.PERMISSION_GRANTED
            ) {
                return sm?.activeSubscriptionInfoList
            }
        }
        return null
    }

    protected fun getSimInfoList(list: List<SubscriptionInfo>): List<PayUSimInfo> {
        val simInfoList1 = ArrayList<PayUSimInfo>()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
            for (sim in list) {
                if (sim.carrierName.toString()
                        .uppercase(Locale.ROOT) != PayUUpiConstant.PAYU_CARRIER_NO_SERVICE
                ) {
                    getPhoneNoFromSubscriptionInfo(sim)?.let { phone ->
                        if(phone.isNotEmpty()) {
                            val simInfo = PayUSimInfo(
                                phone,
                                sim.simSlotIndex.toString(),
                                sim.subscriptionId.toString(),
                                sim.carrierName.toString()
                            )
                            simInfoList1.add(simInfo)
                        }
                    }
                }
            }
        }
        return simInfoList1
    }

    @SuppressLint("MissingPermission")
    private fun getPhoneNoFromSubscriptionInfo(sim: SubscriptionInfo): String? {
        var phone: String? = null
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
            val sm = ContextCompat.getSystemService(activity.baseContext, SubscriptionManager::class.java)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                if (ActivityCompat.checkSelfPermission(
                        activity, Manifest.permission.READ_PHONE_NUMBERS)
                    == PackageManager.PERMISSION_GRANTED) {
                    phone = sm?.getPhoneNumber(sim.subscriptionId)
                    if (phone.isNullOrEmpty()) {
                        phone = sim.number
                    }
                }
            } else {
                val permission = if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q)
                    Manifest.permission.READ_PHONE_STATE
                else Manifest.permission.READ_PHONE_NUMBERS
                if (ActivityCompat.checkSelfPermission(activity, permission)
                    == PackageManager.PERMISSION_GRANTED
                ) {
                    phone = sim.number
                }
            }
            phone = phone?.let { Utils.formatMobileNumber(it) }
        }
        return phone
    }

    fun storeSimInfoInSharedPref(subId: String, mobile: String, carrier: String?) {
        val map = HashMap<String, String>().apply {
            put(PayUUpiConstant.PREF_KEY_SUBSCRIPTION_ID, subId)
            put(PayUUpiConstant.PREF_KEY_MOBILE, mobile)
            carrier?.let { put(PayUUpiConstant.PREF_KEY_CARRIER, carrier) }
        }
        SharedPrefsUtils.saveStringValues(activity, map)
    }

    private fun storeSlotSize(slotSize: String) {
        SharedPrefsUtils.saveStringValue(activity, PayUUpiConstant.PAYU_SLOT_SIZE, slotSize)
    }

    protected fun getValidStoredSimInfo(
        storedSimInfo: PayUSimInfo, deviceSimInfo: List<PayUSimInfo>?): List<PayUSimInfo>? {
        deviceSimInfo?.let { info ->
            return info.filter { it.mobileNumber == storedSimInfo.mobileNumber
                    && it.subscriberId == storedSimInfo.subscriberId }
        }
        return null
    }

    protected fun getSimInfoInSharedPref(): PayUSimInfo {
        val number = SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PREF_KEY_MOBILE)
        val id = SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PREF_KEY_SUBSCRIPTION_ID)
        val index = SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PREF_KEY_SLOT_INDEX)
        val carrier = SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PREF_KEY_CARRIER)
        return PayUSimInfo(number ?: "", index ?: "", id ?: "", carrier)
    }

    private fun getSlotIndexFromPreference() =
        SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PREF_KEY_SLOT_INDEX)

    fun getSlotSizeFromPreference() =
        SharedPrefsUtils.getStringValue(activity, PayUUpiConstant.PAYU_SLOT_SIZE)

    fun hasSmsAndPhonePermissions() =
        hasPermissions(activity.baseContext, PayUUpiConstant.PERMISSIONS.keys.toTypedArray())

    private fun hasPermissions(context: Context, permissions: Array<String>): Boolean =
        permissions.all {
            ActivityCompat.checkSelfPermission(context, it) == PackageManager.PERMISSION_GRANTED
        }

    internal fun getPayUUpiParams(paymentParams: PayUUPIBoltPaymentParams, paymentHash: String) =
        PayUUPIParams(
            InternalConfig.config?.merchantKey,
            email = InternalConfig.config?.email,
            phone = InternalConfig.mobile,
            hash = paymentHash,
            bankCode = PayUUpiConstant.PG_DETAIL_TYPE_IN_APP,
            params = paymentParams
        )

    protected fun clearPayUSdk() {
        SharedPrefsUtils.clearPreference(activity)
        InternalConfig.clearNonInitParams()
    }

    internal fun logFailureResponse(
        requestType: Int,
        errorCode: Int? = null,
        errorMessage: String?,
        referenceId: String? = null,
        txnId: String ?= null,
        amount: String? = null,
        eventType: AnalyticsUtils.EventType = AnalyticsUtils.EventType.Error,
        eventSeverity: AnalyticsUtils.EventSeverity = AnalyticsUtils.EventSeverity.High,
        axisErrorCode: String? = null
    ) {
        val responseType =
            payUAxisResponseTypeMap[requestType] ?: PayUResponseTypes.DEFAULT_FAILURE
        val responseTypeString =
            payuResponseTypeMap[responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN

        logFailureResponse(
            eventKey = responseTypeString,
            errorCode = errorCode,
            errorMessage = errorMessage ?: "",
            timeDiff = Utils.getTimeDifferenceInMilliSeconds(
                axisStartTimeMap[responseTypeString] ?: 0
            ),
            txnId = txnId,
            referenceId = referenceId,
            amount = amount,
            eventType, eventSeverity,
            axisErrorCode = axisErrorCode
        )
    }

    internal fun logFailureResponse(
        eventKey: String,
        errorCode: Int? = null,
        errorMessage: String,
        timeDiff: Long,
        txnId: String? = null,
        referenceId: String? = null,
        amount: String? = null,
        eventType: AnalyticsUtils.EventType = AnalyticsUtils.EventType.Error,
        eventSeverity: AnalyticsUtils.EventSeverity = AnalyticsUtils.EventSeverity.High,
        axisErrorCode: String? = null
    ) {
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_API}$eventKey",
            errorMessage,
            txnId = txnId,
            timeDiff = timeDiff,
            refType = PAYU_SDK_RESPONSE,
            code = errorCode?.toString() ?: axisErrorCode ?: PayUResponseCodes.PAYU_FAILED_STATUS.toString(),
            status = CLConstant.PAYU_EVENT_FAILURE,
            eventType = eventType,
            eventSeverity = eventSeverity,
            referenceId = referenceId,
            amount = amount?.toDouble()
        )
    }

    internal fun logAxisFailureResponse(
        requestType: Int,
        errorCode: String = PayUResponseCodes.PAYU_FAILED_STATUS.toString(),
        errorMessage: String?,
        referenceId: String? = null,
        txnId: String ?= null,
        eventType: AnalyticsUtils.EventType = AnalyticsUtils.EventType.Error,
        eventSeverity: AnalyticsUtils.EventSeverity = AnalyticsUtils.EventSeverity.High
    ) {
        val responseType =
            payUAxisResponseTypeMap[requestType] ?: PayUResponseTypes.DEFAULT_FAILURE
        val responseTypeString =
            payuResponseTypeMap[responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_AXIS_API}$responseTypeString",
            errorMessage,
            txnId = txnId,
            timeDiff = Utils.getTimeDifferenceInMilliSeconds(
                axisStartTimeMap[responseTypeString] ?: 0
            ),
            refType = AXIS_SDK_RESPONSE,
            code = errorCode,
            status = CLConstant.PAYU_EVENT_FAILURE,
            eventType = eventType,
            eventSeverity = eventSeverity,
            referenceId = referenceId
        )
    }

    internal fun logSuccessResponse(
        requestType: Int,
        response: String,
        txnId: String? = null
    ) {
        val responseType =
            payUAxisResponseTypeMap[requestType] ?: PayUResponseTypes.DEFAULT_FAILURE
        val responseTypeString =
            payuResponseTypeMap[responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN

        logSuccessResponse(
            eventKey = responseTypeString,
            eventData = response,
            timeDiff = Utils.getTimeDifferenceInMilliSeconds(
                axisStartTimeMap[responseTypeString] ?: 0
            ),
            txnId = txnId,
            referenceId = referenceId
        )
    }

    internal fun logSuccessResponse(
        eventKey: String,
        eventData: String,
        timeDiff: Long,
        txnId: String? = null,
        referenceId: String? = null,
        amount: String? = null
    ) {
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_API}$eventKey",
            eventData,
            txnId = txnId,
            timeDiff = timeDiff,
            refType = PAYU_SDK_RESPONSE,
            status = CLConstant.PAYU_EVENT_SUCCESS,
            eventType = AnalyticsUtils.EventType.Info,
            referenceId = referenceId,
            amount = amount?.toDouble()
        )
    }

    internal fun logAxisSuccessResponse(
        requestType: Int,
        response: String,
        txnId: String? = null
    ) {
        val responseType =
            payUAxisResponseTypeMap[requestType] ?: PayUResponseTypes.DEFAULT_FAILURE
        val responseTypeString =
            payuResponseTypeMap[responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_AXIS_API}$responseTypeString",
            response,
            txnId = txnId,
            timeDiff = Utils.getTimeDifferenceInMilliSeconds(
                axisStartTimeMap[responseTypeString] ?: 0
            ),
            refType = AXIS_SDK_RESPONSE,
            status = CLConstant.PAYU_EVENT_SUCCESS,
            eventType = AnalyticsUtils.EventType.Info,
            referenceId = referenceId
        )
    }

    internal fun logAxisFunctionsRequest(
        responseType: Int,
        request: String,
        txnId: String? = null,
        referenceId: String? = null,
        amount: String? = null
    ) {
        val responseTypeString = payuResponseTypeMap[responseType] ?: PAYU_REQUEST_UNKNOWN
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_AXIS_API}$responseTypeString",
            request,
            refType = PayUUpiConstant.AXIS_PAYU_REQUEST,
            status = CLConstant.PAYU_CT_INITIATED,
            eventType = AnalyticsUtils.EventType.Info,
            txnId = txnId,
            referenceId = referenceId,
            amount = amount?.toDouble()
        )
    }

    internal fun logPayUFunctionsRequest(
        responseType: Int,
        request: String,
        txnId: String? = null,
        referenceId: String? = null,
        amount: String? = null
    ) {
        val responseTypeString = payuResponseTypeMap[responseType] ?: PAYU_REQUEST_UNKNOWN
        AnalyticsUtils.logEventNameForKibana(
            activity,
            "${PayUUpiConstant.PAYU_API}$responseTypeString",
            request,
            refType = PayUUpiConstant.SDK_PAYU_REQUEST,
            status = CLConstant.PAYU_CT_INITIATED,
            eventType = AnalyticsUtils.EventType.Info,
            txnId = txnId,
            referenceId = referenceId,
            amount = amount?.toDouble()
        )
    }

    protected fun prepareAccountJson(accountList: ArrayList<PayUCustomerBankAccounts>): JSONArray {
        val jsonArray = JSONArray()
        for (account in accountList) {
            val jsonObject = JSONObject()
            jsonObject.put(PayUUpiConstant.PAYU_MASKED_ACT_NUMBER, account.bankAccounts.maskedAccnumber)
            jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(account.bankAccounts.vpa))
            jsonObject.put(PayUUpiConstant.PAYU_AEBA, account.bankAccounts.aeba)
            jsonObject.put(PayUUpiConstant.PAYU_MBEBA, account.bankAccounts.mbeba)
            jsonObject.put(PayUUpiConstant.VERIFY_STATUS, account.bankAccounts.status)
            jsonObject.put(PayUUpiConstant.PAYU_TYPE, account.bankAccounts.type)
            jsonArray.put(jsonObject)
        }
        return jsonArray
    }
    protected fun prepareAxisAccountJson(accountList: ArrayList<Account>): JSONArray {
        val jsonArray = JSONArray()
        for (account in accountList) {
            val jsonObject = JSONObject()
            jsonObject.put(PayUUpiConstant.PAYU_MASKED_ACT_NUMBER, account.maskedAccnumber)
            jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(account.vpa))
            jsonObject.put(PayUUpiConstant.PAYU_AEBA, account.aeba)
            jsonObject.put(PayUUpiConstant.PAYU_MBEBA, account.mbeba)
            jsonObject.put(PayUUpiConstant.VERIFY_STATUS, account.status)
            jsonObject.put(PayUUpiConstant.PAYU_TYPE, account.type)
            jsonArray.put(jsonObject)
        }
        return jsonArray
    }

    protected fun prepareAxisCustomerAccountJson(customerBankAccounts: ArrayList<com.olive.upi.transport.model.CustomerBankAccounts>): JSONArray {
        val jsonArray = JSONArray()
        for (customerBankAccount in customerBankAccounts) {
            for (account in customerBankAccount.accounts) {
                val jsonObject = prepareJsonFromAccount(account)
                jsonArray.put(jsonObject)
            }
        }
        return jsonArray
    }

    protected fun prepareJsonFromAccount(accountDetail: PayUAccountDetail): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_ACC_REF_NUMBER, Utils.getMaskedString(accountDetail.accRefNumber))
        jsonObject.put(PayUUpiConstant.PAYU_AEBA, accountDetail.aeba)
        jsonObject.put(PayUUpiConstant.PAYU_ATM_D_LENGTH, accountDetail.atmdLength)
        jsonObject.put(PayUUpiConstant.PAYU_D_LENGTH, accountDetail.dLength)
        jsonObject.put(PayUUpiConstant.PAYU_D_TYPE, accountDetail.dType)
        jsonObject.put(PayUUpiConstant.PAYU_FORMAT_TYPE, accountDetail.formatType)
        jsonObject.put(PayUUpiConstant.PAYU_IFSC, accountDetail.ifsc)
        jsonObject.put(PayUUpiConstant.PAYU_IIN, accountDetail.iin)
        jsonObject.put(PayUUpiConstant.PAYU_MMID, accountDetail.mmid)
        jsonObject.put(PayUUpiConstant.PAYU_BALANCE, accountDetail.balance)
        jsonObject.put(PayUUpiConstant.PAYU_BAL_TIME, accountDetail.balTime)
        jsonObject.put(PayUUpiConstant.PAYU_BANK_CODE, accountDetail.bankCode)
        jsonObject.put(PayUUpiConstant.PAYU_BANK_NAME, accountDetail.bankName)
        jsonObject.put(PayUUpiConstant.PAYU_BANK_ID, accountDetail.bankId)
        jsonObject.put(PayUUpiConstant.PAYU_MASKED_ACT_NUMBER, accountDetail.maskedAccnumber)
        jsonObject.put(PayUUpiConstant.PAYU_MBEBA, accountDetail.mbeba)
        jsonObject.put(PayUUpiConstant.PAYU_NAME, accountDetail.name)
        jsonObject.put(PayUUpiConstant.PAYU_OTP_D_LENGTH, accountDetail.otpdLength)
        jsonObject.put(PayUUpiConstant.PAYU_OTP_D_TYPE, accountDetail.otpdType)
        jsonObject.put(PayUUpiConstant.PAYU_STATUS, accountDetail.status)
        jsonObject.put(PayUUpiConstant.PAYU_TYPE, accountDetail.type)
        jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(accountDetail.vpa))
        return jsonObject
    }

    protected fun prepareJsonFromAccount(account: Account): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_NAME, account.name)
        jsonObject.put(PayUUpiConstant.PAYU_ACC_REF_NUMBER, Utils.getMaskedString(account.accRefNumber))
        jsonObject.put(PayUUpiConstant.PAYU_IFSC, account.ifsc)
        jsonObject.put(PayUUpiConstant.PAYU_MASKED_ACCNUMBER, account.maskedAccnumber)
        jsonObject.put(PayUUpiConstant.PAYU_TYPE, account.type)
        jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(account.vpa))
        jsonObject.put(PayUUpiConstant.PAYU_IIN, account.iin)
        jsonObject.put(PayUUpiConstant.PAYU_MMID, account.mmid)
        jsonObject.put(PayUUpiConstant.PAYU_AEBA, account.aeba)
        jsonObject.put(PayUUpiConstant.PAYU_MBEBA, account.mbeba)
        jsonObject.put(PayUUpiConstant.PAYU_D_LENGTH, account.dLength)
        jsonObject.put(PayUUpiConstant.PAYU_D_TYPE, account.dType)
        jsonObject.put(PayUUpiConstant.PAYU_STATUS, account.status)
        jsonObject.put(PayUUpiConstant.PAYU_ATMPIN_FORMAT, account.atmpinFormat)
        jsonObject.put(PayUUpiConstant.PAYU_ATMPIN_LENGTH, account.atmpinlength)
        jsonObject.put(PayUUpiConstant.PAYU_OTP_LENGTH, account.otpLength)
        jsonObject.put(PayUUpiConstant.PAYU_OTP_FORMAT, account.otpFormat)
        jsonObject.put(PayUUpiConstant.PAYU_BALANCE, account.balance)
        jsonObject.put(PayUUpiConstant.PAYU_BAL_TIME, account.balTime)
        return jsonObject
    }

    protected fun prepareActivateAccountJson(account: Account, cardNumber: String, exp: String): JSONObject {
        val jsonObject = prepareJsonFromAccount(account)
        val maskedCardNumber = cardNumber.replace(PayUUpiConstant.PAYU_CARD_MASK_REGEX, "*")
        jsonObject.put(PayUUpiConstant.PAYU_MASKED_CARD_NUMBER,maskedCardNumber)
        jsonObject.put(PayUUpiConstant.PAYU_EXPIRY,exp)
        return jsonObject
    }
    protected fun prepareActivatePayuAccountJson(account: PayUAccountDetail, cardNumber: String, exp: String): JSONObject {
        val jsonObject = prepareJsonFromAccount(account)
        val maskedCardNumber = cardNumber.replace(PayUUpiConstant.PAYU_CARD_MASK_REGEX, "*")
        jsonObject.put(PayUUpiConstant.PAYU_MASKED_CARD_NUMBER,maskedCardNumber)
        jsonObject.put(PayUUpiConstant.PAYU_EXPIRY,exp)
        return jsonObject
    }

    protected fun prepareSaveVpaJson(vpa: String, nickName: String): JSONObject {
        val jSONObject = JSONObject()
        jSONObject.put(PayUUpiConstant.PAYU_VPA,Utils.getMaskedString(vpa))
        jSONObject.put(PayUUpiConstant.PAYU_NICK_NAME,nickName)
        return jSONObject
    }

    protected fun prepareQueryRequestJson(txnId: String, refId: String, amount:Double, query:String): JSONObject {
        val jSONObject = JSONObject()
        jSONObject.put(PayUUpiConstant.PAYU_TXN_ID,txnId)
        jSONObject.put(PayUUpiConstant.PAYU_SDK_BOLT_REF_ID,refId)
        jSONObject.put(PayUUpiConstant.PAYU_AMOUNT,amount)
        jSONObject.put(PayUUpiConstant.PAYU_QUERY,query)
        return jSONObject
    }

    protected fun txnHistoryRequestJson(fromDate: String, toDate: String): JSONObject {
        val jSONObject = JSONObject()
        jSONObject.put(PayUUpiConstant.PAYU_FROM_DATE, fromDate)
        jSONObject.put(PayUUpiConstant.PAYU_TO_DATE, toDate)
        return jSONObject
    }

    protected fun prepareInitiatePayJson(
        account: PayUAccountDetail,
        beneVpa: BeneVpa,
        amount: String,
        productInfo: String,
        txnId: String
    ): JSONObject {
        val jsonObject = prepareJsonFromAccount(account)
        val beneVpaJson = prepareSaveVpaJson(beneVpa.vpa, beneVpa.nickname)
        jsonObject.put(PayUUpiConstant.PAYU_BENE_VPA, beneVpaJson)
        jsonObject.put(PayUUpiConstant.PAYU_AMOUNT, amount)
        jsonObject.put(PayUUpiConstant.PAYU_PRODUCT_INFO, productInfo)
        jsonObject.put(PayUUpiConstant.PAYU_TXN_ID, txnId)
        return jsonObject
    }

    protected fun preparePgDetailsRequestJson(pgDetailsRequest: PgDetailsRequest): JSONObject{
        val jsonObject = JSONObject()
        jsonObject.put(PAYU_MERCHANT_KEY,pgDetailsRequest.merchantKey)
        jsonObject.put(PAYU_PG,pgDetailsRequest.pg)
        jsonObject.put(HASH,pgDetailsRequest.hash)
        return jsonObject
    }

    private fun preparePgDetailsResponseJson(pgDetails: PgDetails?): JSONObject{
        val jsonObject = JSONObject()
        jsonObject.put(PAYU_MERCHANT_KEY,pgDetails?.mKey)
        jsonObject.put(PAYU_PG,pgDetails?.pgId)
        return jsonObject
    }

    protected fun prepareCheckDeviceRequestJson(subscriptionId: String, mobile: String): JSONObject{
        val jsonObject = JSONObject()
        jsonObject.put(PREF_KEY_SUBSCRIPTION_ID,subscriptionId)
        jsonObject.put(PAYU_MOBILE, Utils.getMaskedMobileNumber(mobile))
        return jsonObject
    }

    protected fun prepareSdkInitParamJson(): JSONObject {
        val jSONObject = JSONObject()
        jSONObject.put(PAYU_MERCHANT_KEY, InternalConfig.config?.merchantKey)
        jSONObject.put(PAYU_PG, InternalConfig.pluginType?.name)
        jSONObject.put(PAYU_APP_ID, InternalConfig.appId)
        jSONObject.put(PREF_KEY_SUBSCRIPTION_ID, InternalConfig.subscriptionId)
        InternalConfig.mobile?.let {
            jSONObject.put(PAYU_MOBILE, Utils.getMaskedMobileNumber(it))
        }
        jSONObject.put(PAYU_UNQ_TXNID, InternalConfig.handshakeId)
        jSONObject.put(PAYU_EMAIL_ID,InternalConfig.config?.email)
        jSONObject.put(PAYU_IS_PROD,InternalConfig.config?.isProduction)
        jSONObject.put(PAYU_CARRIER_NAME, getCarrierNameFromPhone(InternalConfig.mobile))
        return jSONObject
    }

    protected fun prepareVerifyRequestJson(verifyApiRequest: VerifyApiRequest): JSONObject{
        val jsonObject = JSONObject()
        jsonObject.put(PAYU_MERCHANT_KEY,verifyApiRequest.merchantKey)
        jsonObject.put(PAYU_TXN_ID,verifyApiRequest.txnId)
        jsonObject.put(HASH,verifyApiRequest.verifyPaymentHash)
        return jsonObject
    }

    protected fun initPayment(
        paymentParams: PayUUPIBoltPaymentParams,
        callback: PayUUPIBoltCallBack,
        fnCalledTime: Long
    ) {
        InternalConfig.paymentTxnId = paymentParams.txnId
        amount = paymentParams.amount
        logPayUFunctionsRequest(
            REQUEST_PAY,
            PAYU_INIT_PAYMENT,
            paymentParams.txnId,
            referenceId = referenceId,
            amount = paymentParams.amount
        )
        payuStartTimeMap[PayUUpiConstant.PAYU_REQUEST_PAY] = fnCalledTime
        if (paymentParams.accountDetail?.mbeba == PayUUpiConstant.PAYU_VALUE_Y && InternalConfig.config != null) {
            callBack[PayUResponseTypes.PAYU_CALLBACK_KEY] = callback

            val hashMap =
                UpiAuthHeaderHelper.getUPIHashString(
                    UpiHashCommand.PaymentHash,
                    paymentParams,
                    InternalConfig.config?.merchantKey ?: "",
                    InternalConfig.config?.email ?: ""
                )
            logPayUFunctionsRequest(
                REQUEST_PAY,
                PAYU_REQUEST_PAYMENT_HASH + hashMap.entries.toString(),
                paymentParams.txnId,
                referenceId = referenceId,
                amount = paymentParams.amount
            )
            InternalConfig.payUHashGenerationListener?.generateHash(
                hashMap,
                object : PayUHashGeneratedListener {
                    override fun onHashGenerated(map: HashMap<String, String?>) {
                        logPayUFunctionsRequest(
                            REQUEST_PAY,
                            PAYU_PAYMENT_HASH_RECEIVED + map.entries.toString(),
                            paymentParams.txnId,
                            referenceId = referenceId,
                            amount = paymentParams.amount
                        )
                        val hashName = map[PayUUpiConstant.HASH_NAME]
                        val hash = map[hashName]
                        if (!hash.isNullOrEmpty()) {
                            val payUUPIParams = getPayUUpiParams(paymentParams, hash)

                            InitiatePaymentTask(activity).callApi(
                                payUUPIParams,
                                InternalConfig.config?.isProduction ?: true,
                                object : ApiBaseCallback {
                                    override fun onApiSuccess(response: Any) {
                                        initiatePayResponse =
                                            InitiatePaymentResponse.getInitiatePaymentResponse(response.toString())
                                        makePayment(payUUPIParams)
                                    }

                                    override fun onApiError(errorCode: Int, errorMessage: String) {
                                        val code = if (errorCode == PayUResponseCodes.PAYU_FAILED_STATUS) {
                                            PayUResponseCodes.PAYU_INITIATE_PAY_FAILED
                                        } else {
                                            errorCode
                                        }
                                        val response = PayUUPIBoltResponse(
                                            PayUResponseTypes.REQUEST_PAY, code, errorMessage
                                        )
                                        logFailureResponse(
                                            PayUUpiConstant.PAYU_REQUEST_PAY,
                                            response.code,
                                            response.message ?: "",
                                            Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                                            paymentParams.txnId,
                                            referenceId = referenceId,
                                            amount = paymentParams.amount
                                        )
                                        callback.onPayUFailure(response)
                                    }
                                })
                        } else {
                            val response = PayUUPIBoltResponse(
                                PayUResponseTypes.REQUEST_PAY,
                                PayUResponseCodes.PAYU_FAILED_STATUS,
                                PayUResponseMessages.HASH_NOT_RECEIVED
                            )
                            logFailureResponse(
                                PayUUpiConstant.PAYU_REQUEST_PAY,
                                response.code,
                                response.message ?: "",
                                Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                                paymentParams.txnId,
                                referenceId = referenceId,
                                amount = paymentParams.amount
                            )
                            callback.onPayUFailure(response)
                        }
                    }
                }
            )
        } else {
            val response = PayUUPIBoltResponse(
                PayUResponseTypes.REQUEST_PAY,
                PayUResponseCodes.PAYU_FAILED_STATUS, PayUResponseMessages.PAYU_PIN_NOT_SET
            )
            logFailureResponse(
                PayUUpiConstant.PAYU_REQUEST_PAY,
                response.code,
                response.message ?: "",
                Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                paymentParams.txnId,
                referenceId = referenceId,
                amount = paymentParams.amount
            )
            callback.onPayUFailure(response)
        }
    }

    open fun verifyTransactionStatus(txnId: String, callback: PayUUPIBoltCallBack) {
        val fnCalledTime = System.currentTimeMillis()
        val mKey = InternalConfig.config?.merchantKey ?: ""
        val request = Utils.generateVerifyPaymentPostData(txnId)
        val hashMap =
            UpiAuthHeaderHelper.getUPIHashString(
                UpiHashCommand.VerifyPayment,
                merchantKey = mKey,
                email = InternalConfig.config?.email ?: "",
                requestJson = request,
                txnId = txnId
            )

        logPayUFunctionsRequest(
            REQUEST_CHECK_PAYMENT_STATUS,
            PAYU_REQUEST_VERIFY_PAYMENT_HASH + hashMap.entries.toString(),
            txnId,
            referenceId = referenceId,
            amount = amount
        )
        InternalConfig.payUHashGenerationListener?.generateHash(
            hashMap,
            object : PayUHashGeneratedListener {
                override fun onHashGenerated(map: HashMap<String, String?>) {
                    val verifyHashName = map[PayUUpiConstant.HASH_NAME]
                    val hash = map[verifyHashName]
                    if (hash.isNullOrEmpty().not()) {
                        val verifyApiRequest = VerifyApiRequest(activity, InternalConfig.config?.merchantKey, txnId, hash)
                        logPayUFunctionsRequest(
                            REQUEST_CHECK_PAYMENT_STATUS,
                            PAYU_VERIFY_PAYMENT_HASH_RECEIVED + "_" + prepareVerifyRequestJson(verifyApiRequest).toString(),
                            txnId, referenceId = referenceId,
                            amount = amount
                        )
                        val verifyPaymentTask = VerifyPaymentTask(activity)
                        verifyPaymentTask.callApi(verifyApiRequest, true, object :
                            ApiBaseCallback {
                            override fun onApiSuccess(response: Any) {
                                val payUResponse = PayUUPIBoltResponse(
                                    PayUResponseTypes.REQUEST_CHECK_PAYMENT_STATUS,
                                    PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                    null, response
                                )
                                val resultJson = JSONObject(response.toString())
                                val status = resultJson.getString(PayUUpiConstant.VERIFY_STATUS)
                                if (status == PayUUpiConstant.VERIFY_STATUS_FAILURE) {
                                    logFailureResponse(
                                        PayUUpiConstant.PAYU_VERIFY_PAYMENT,
                                        PayUResponseCodes.PAYU_FAILED_STATUS,
                                        resultJson.toString(),
                                        Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                                        txnId, referenceId = referenceId,
                                        amount = amount
                                    )
                                } else {
                                    logSuccessResponse(
                                        PayUUpiConstant.PAYU_VERIFY_PAYMENT,
                                        ApiConstant.SUCCESS,
                                        Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                                        txnId, referenceId = referenceId,
                                        amount = amount
                                    )
                                }
                                callback.onPayUSuccess(payUResponse)
                            }

                            override fun onApiError(errorCode: Int, errorMessage: String) {
                                val response = PayUUPIBoltResponse(
                                    PayUResponseTypes.REQUEST_CHECK_PAYMENT_STATUS,
                                    errorCode, errorMessage
                                )
                                logFailureResponse(
                                    PayUUpiConstant.PAYU_VERIFY_PAYMENT,
                                    response.code,
                                    response.message ?: "",
                                    Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                                    txnId, referenceId = referenceId,
                                    amount = amount
                                )
                                callback.onPayUFailure(response)
                            }
                        })
                    } else {
                        val response = PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_CHECK_PAYMENT_STATUS,
                            PayUResponseCodes.PAYU_FAILED_STATUS,
                            PayUResponseMessages.HASH_NOT_RECEIVED
                        )

                        logFailureResponse(
                            PayUUpiConstant.PAYU_REQUEST_PAY,
                            response.code,
                            response.message ?: "",
                            Utils.getTimeDifferenceInMilliSeconds(fnCalledTime),
                            txnId, referenceId = referenceId,
                            amount = amount
                        )
                        callback.onPayUFailure(response)
                    }
                }
            }
        )
    }

    open fun cancelTransaction(callback: PayUUPIBoltCallBack) {
        if(initiatePayResponse?.returnUrl.isNullOrEmpty().not() &&  initiatePayResponse?.token.isNullOrEmpty().not()) {
            val cancelApiRequest = CancelApiRequest(initiatePayResponse?.token!!, initiatePayResponse?.returnUrl!!)
            val cancelApiTask = CancelApiTask(activity)
            cancelApiTask.callApi(cancelApiRequest, callback = object: ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    initiatePayResponse?.token = null
                    initiatePayResponse?.returnUrl = null
//                    Log.d("cancelapi", response.toString())
                    callback.onPayUSuccess(
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_CANCEL_TRANSACTION,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS
                        )
                    )
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
//                    Log.d("cancelapi", errorMessage)
                    callback.onPayUFailure(
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_CANCEL_TRANSACTION,
                            errorCode, errorMessage
                        )
                    )
                }
            })
        }
    }

    protected fun getCarrierNameFromPhone(phone: String?): String? {
        if(phone.isNullOrEmpty().not()) {
            val infoList = getSubscriptionInfoList()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
                return infoList?.find { getPhoneNoFromSubscriptionInfo(it)?.contains(phone ?: "") == true }?.carrierName?.toString()
            }
        }
        return null
    }

    // Request from Merchant
    protected fun prepareJsonFromAccountsInn(
        bankName: String?,
        bankCode: String?,
        iin: String?,
        vpa: String?,
        isCCTxnEnabled: Boolean,
        requestType: String?
    ): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_BANK_NAME, bankName)
        jsonObject.put(PayUUpiConstant.PAYU_BANK_CODE, bankCode)
        jsonObject.put(PayUUpiConstant.PAYU_IIN, iin)
        jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(vpa))
        jsonObject.put(PayUUpiConstant.PAYU_IS_CC_TXN_ENABLED, isCCTxnEnabled)
        jsonObject.put(PayUUpiConstant.PAYU_REQUEST_TYPE, requestType)
        return jsonObject
    }

    // Request to Axis
    protected fun prepareJsonFromAccountsInn(iin: String?, isCCTxnEnabled: Boolean): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_IIN, iin)
        jsonObject.put(PayUUpiConstant.PAYU_TXN_ACCOUNT_TYPE, if (isCCTxnEnabled) "CREDIT" else "")
        return jsonObject
    }

    // Response from Axis
    protected fun prepareJsonFromCheckBalanceResponse(result: String, code: String): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_RESULT, result)
        jsonObject.put(PayUUpiConstant.PAYU_CODE, code)
        return jsonObject
    }

    // response to Merchant
    protected fun prepareJsonForCheckBalanceMerchant(response: PayUUPIBoltResponse): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put(PayUUpiConstant.PAYU_RESPONSE_TYPE, response.responseType)
        jsonObject.put(PAYU_CODE, response.code)
        return jsonObject
    }

    protected fun preparePluginCoreAccountJson(registerVPAResponseResult: RegisterVPAResponseResult): JSONArray {
        registerVPAResponseResult.run {
            val jsonArray = JSONArray()
            val jsonObject = JSONObject()
            jsonObject.put(PayUUpiConstant.PAYU_NAME, userInfo.name)
            jsonObject.put(
                PayUUpiConstant.PAYU_ACC_REF_NUMBER,
                Utils.getMaskedString(userInfo.accountId)
            )
            jsonObject.put(PayUUpiConstant.PAYU_IFSC, userInfo.ifscCode)
            jsonObject.put(PayUUpiConstant.PAYU_MASKED_ACCNUMBER, userInfo.accountNo)
            jsonObject.put(PayUUpiConstant.PAYU_TYPE, userInfo.accountType)
            jsonObject.put(PayUUpiConstant.PAYU_VPA, Utils.getMaskedString(userInfo.virtualAddress))
            jsonArray.put(jsonObject)
            return jsonArray
        }
    }
}