package com.payu.upibolt.payUAxisImpl

import android.content.Context
import android.util.Base64
import android.util.Log
import com.olive.upi.transport.model.sdk.SDKHandshake
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.listeners.ApiBaseCallback
import com.payu.commonmodelssdk.model.PgDetails
import com.payu.upibolt.models.PayUUPIBoltConfig
import com.payu.upibolt.BuildConfig
import com.payu.upibolt.utils.InternalConfig
import com.payu.upibolt.utils.Utils
import org.json.JSONObject
import java.security.MessageDigest
import java.sql.Timestamp

class AxisAuthTokenManager(
    private val context: Context,
    private val payUUPIBoltConfig: PayUUPIBoltConfig,
    private val pgDetails: PgDetails,
    private val onSuccess: (sdkHandshake: SDKHandshake) -> Unit,
    private val onFailure: (errorCode: Int, errorMessage: String) -> Unit
) {

    init {
        val txnId = PayUUpiConstant.PAYU_TXNID_PREFIX + System.currentTimeMillis().toString()
        InternalConfig.handshakeId = txnId
        val params = prepareAuthTokenPostData(txnId)
        if (params.isNotEmpty()) {
            AuthTokenTask(context).callApi(params, payUUPIBoltConfig.isProduction, object :
                ApiBaseCallback {
                override fun onApiSuccess(response: Any) {
                    val authToken = response.toString()
                    val sdkHandshake = prepareSdkHandShakeParams(authToken, txnId)
                    onSuccess.invoke(sdkHandshake)
                }

                override fun onApiError(errorCode: Int, errorMessage: String) {
                    onFailure.invoke(errorCode, errorMessage)
                }
            })
        } else {
            onFailure.invoke(
                PayUResponseCodes.PAYU_FAILED_STATUS,
                PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE
            )
        }
    }

    private fun prepareAuthTokenPostData(txnId: String): String {
        val currentTimeMillis = System.currentTimeMillis()
        val timestamp = Timestamp(currentTimeMillis)
        val params = JSONObject().apply {
            val checkSumVal = createCheckSum(payUUPIBoltConfig, pgDetails, timestamp, txnId)
            put(PayUUpiConstant.PAYU_CHECKSUM, checkSumVal)
            put(PayUUpiConstant.PAYU_MERCHID, pgDetails.merchId)
            put(PayUUpiConstant.PAYU_MERCHANID, pgDetails.merchChanId)
            put(PayUUpiConstant.PAYU_SUB_MERCHNANID, pgDetails.merchId)
            put(PayUUpiConstant.PAYU_UNQ_CUSTID, InternalConfig.mobile)
            put(PayUUpiConstant.PAYU_MOBILE, InternalConfig.mobile)
            put(PayUUpiConstant.PAYU_UNQ_TXNID, txnId)
            put(PayUUpiConstant.PAYU_TIME_STAMP, timestamp)
            put(PayUUpiConstant.PAYU_MCC_CODE, pgDetails.mcc)
            put(PayUUpiConstant.PAYU_EMAIL_ID, payUUPIBoltConfig.email)
        }
        return params.toString()
    }

    private fun createCheckSum(
        payUUPIBoltConfig: PayUUPIBoltConfig?,
        pgDetails: PgDetails?,
        timeStamp: Timestamp,
        txnId: String
    ): String {
        val checksumInput = (payUUPIBoltConfig?.email ?: "") + (pgDetails?.merchId ?: "") + (pgDetails?.merchChanId ?: "") + (pgDetails?.merchId
            ?: "") + (pgDetails?.mcc ?: "") + (InternalConfig.mobile ?: "") + timeStamp.toString() + (InternalConfig.mobile
            ?: "") + (txnId ?: "")
        var checkSumval = ""
        try {
            val pluginKey = InternalConfig.pgDetails?.pluginKey
            val skey = if (pluginKey.isNullOrEmpty()) {
                if (payUUPIBoltConfig?.isProduction == true)
                    BuildConfig.AXIS_SECRET_KEY
                else
                    BuildConfig.AXIS_SECRET_KEY_UAT
            } else {
                pluginKey
            }
            val encData: ByteArray = Utils.encrypt(
                Utils.hexStringToByteArray(skey), getHash(
                    checksumInput, PayUUpiConstant.PAYU_SHA256
                )
            )
            checkSumval = Base64.encodeToString(encData, Base64.NO_WRAP)
        } catch (e: Exception) {
            Log.d("AuthTokenManager", "createCheckSum: exception " + e.message)
        }
        return checkSumval
    }

    private fun getHash(paramString: String, shaName: String): ByteArray {
        val localMessageDigest = MessageDigest.getInstance(shaName)
        localMessageDigest.update(paramString.toByteArray())
        return localMessageDigest.digest()
    }

    private fun prepareSdkHandShakeParams(
        authToken: String, txnId: String
    ): SDKHandshake {
        val sdkHandshake = SDKHandshake()
        sdkHandshake.appid = InternalConfig.appId
        sdkHandshake.merchId = pgDetails.merchId
        sdkHandshake.merchChanId = pgDetails.merchChanId
        sdkHandshake.submerchantid = pgDetails.merchId
        sdkHandshake.unqCustId = InternalConfig.mobile
        sdkHandshake.unqTxnId = txnId
        sdkHandshake.mobileNumber = InternalConfig.mobile
        sdkHandshake.subscriptionId = InternalConfig.subscriptionId
        sdkHandshake.deviceid = Utils.getDeviceId(context)
        sdkHandshake.merchanttoken = authToken
        Log.d("AuthTokenManager", "appId:" + sdkHandshake.appid
                + " merchId:" + sdkHandshake.merchId
                + " merchChanId:" + sdkHandshake.merchChanId
                + " submerchantid:" + sdkHandshake.submerchantid
                + " unqCustId:" + sdkHandshake.unqCustId
                + " unqTxnId:" + sdkHandshake.unqTxnId
                + " mobileNumber:" + sdkHandshake.mobileNumber
                + " subscriptionid:" + sdkHandshake.subscriptionId
                + " deviceid:" + sdkHandshake.deviceid
                + " merchanttoken:" + sdkHandshake.merchanttoken)
        return sdkHandshake
    }
}