package com.payu.upibolt.utils

import android.app.ActivityManager
import android.content.Context
import android.os.Build
import android.util.Log
import com.payu.commonmodelssdk.constants.CLConstant
import com.payu.commonmodelssdk.constants.KibanaEvents
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.DD_MM_YY_HH_MM_SS_Z_DATE_FORMAT
import com.payu.payuanalytics.analytics.factory.AnalyticsFactory
import com.payu.payuanalytics.analytics.model.AnalyticsConfig
import com.payu.payuanalytics.analytics.model.AnalyticsType
import com.payu.payuanalytics.analytics.model.ClevertapAnalytics
import com.payu.payuanalytics.analytics.model.PayUAnalytics
import com.payu.upibolt.BuildConfig
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale


/**
 * AnalyticsUtils class for kibana logging
 * */
object AnalyticsUtils {

    enum class EventSeverity{
        High,
        Low
    }
    enum class EventType{
        Info,
        Error
    }
    private var eventIndex = 0

    private fun getAnalyticsJsonWithBasicProperties(context: Context): JSONObject {
        val analyticsJson = JSONObject()

        analyticsJson.put(KibanaEvents.PAYU_TIMESTAMP, getCurrentDateInDDMMYYHHMMSSFormat())
        analyticsJson.put(
            KibanaEvents.PAYU_SDK_BOLT_APPLICATION_PACKAGE,
            context.applicationContext?.packageName
        )
        try {
            val versionName = context.packageManager
                .getPackageInfo(context.packageName, 0).versionName
            analyticsJson.put(
                KibanaEvents.PAYU_SDK_BOLT_APPLICATION_VERSION,
                versionName
            )
        } catch (e: Exception) {
            Log.d("AnalyticsUtils", "getAnalyticsJsonWithBasicProperties1 " + e.message)
        }
        analyticsJson.put(KibanaEvents.LOGGING_SDK,BuildConfig.LIBRARY_PACKAGE_NAME)
        analyticsJson.put(KibanaEvents.PAYU_SDK_VERSION, BuildConfig.VERSION_NAME)
        analyticsJson.put(
            KibanaEvents.PAYU_MERCHANT_KEY, InternalConfig.config?.merchantKey
        )
        analyticsJson.put(
            KibanaEvents.PAYU_SDK_BOLT_REF_ID,
            InternalConfig.config?.refId
        )
        val deviceName = Build.MANUFACTURER + " " + Build.MODEL
        val deviceOsVersion = Build.VERSION.SDK_INT
        val deviceTotalMemory = getTotalMemoryInfo(context)
        val availMemory = getAvailableMemoryInfo(context)
        var memoryInfo: Long = 0
        try {
            memoryInfo = availMemory * 100 / deviceTotalMemory
        } catch (ex: Exception) {
            Log.d("AnalyticsUtils", "getAnalyticsJsonWithBasicProperties2 " + ex.message)

        }
        analyticsJson.put(
            KibanaEvents.PAYU_DEVICE_DETAILS,
            deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
        )
        analyticsJson.put(KibanaEvents.PAYU_PLUGIN_TYPE, InternalConfig.pluginType)
        return analyticsJson
    }

    private fun getTotalMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return (memoryInfo.totalMem / (1024 * 1024))
    }

    private fun getAvailableMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return memoryInfo.availMem / (1024 * 1024)
    }


    fun logEventNameForKibana(
        context: Context,
        eventName: String,
        eventData: String? = null,
        txnId: String? = null,
        timeDiff: Long? = null,
        refType: String? = null,
        code: String? = null,
        status: String? = null,
        screenName: String? = null,
        eventType : EventType = EventType.Info,
        eventSeverity : EventSeverity = EventSeverity.Low,
        referenceId: String? = null,
        amount: Double? = null,
        sdkName: String = PayUUpiConstant.PAYU_BOLT_SDK_NAME,
        name: String? = null
    ) {

        try {
            val analyticsJson = getAnalyticsJsonWithBasicProperties(context)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_KEY, eventName)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_PLATFORM, PayUUpiConstant.ANDROID)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_TYPE, eventType.name)
            analyticsJson.put(KibanaEvents.PAYU_SDK_NAME, sdkName)
            analyticsJson.put(KibanaEvents.PAYU_EVENT_SEVERITY, eventSeverity.name)
            eventData?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_VALUE, eventData) }
            timeDiff?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_TIME, it) }
            analyticsJson.put(KibanaEvents.TXN_ID, txnId ?: InternalConfig.handshakeId)
            referenceId?.let { analyticsJson.put(KibanaEvents.PAYU_EVENT_REFERENCE_ID, it) }
            refType?.let { analyticsJson.put(KibanaEvents.PAYU_SDK_REF_TYPE, it) }
            code?.let { analyticsJson.put(KibanaEvents.PAYU_CODE, it) }
            status?.let { analyticsJson.put(KibanaEvents.PAYU_STATUS, it) }
            screenName?.let { analyticsJson.put(KibanaEvents.PAYU_SCREEN_NAME, it) }
            amount?.let { analyticsJson.put(KibanaEvents.PAYU_AMOUNT, it) }
            name?.let { analyticsJson.put(KibanaEvents.PAYU_NAME, it) }
            InternalConfig.config?.merchantKey?.let {
                analyticsJson.put(CLConstant.PAYU_EVENT_MERCHANT_KEY,it)
            }
            logKibanaData(context, analyticsJson)
        } catch (e: Exception) {
            Log.d("PAYU", e.message.toString())
        }
    }

    private fun logKibanaData(context: Context, analyticsJson: JSONObject) {
        eventIndex++
        analyticsJson.put(KibanaEvents.PAYU_EVENTS_INDEX, eventIndex)
        val payuAnalytics =
            AnalyticsFactory(context).getAnalyticsClass(AnalyticsType.PAYU_ANALYTICS) as PayUAnalytics
        Log.d("logKibanaData", analyticsJson.toString())
        payuAnalytics.log(analyticsJson.toString())
    }

    private fun logEventForCT(
        context: Context,
        eventName: String,
        eventData: HashMap<String, Any>,
        txnId: String? = null
    ) {
        try {

            //Prepare final event map
            val eventMap: HashMap<String, Any> = java.util.HashMap()
            eventMap[CLConstant.PAYU_EVENT_NAME] = eventName
            eventData[CLConstant.PAYU_TIMESTAMP] = getCurrentDateInDDMMYYHHMMSSFormat()
            eventData[CLConstant.PAYU_SDK_BOLT_APPLICATION_VERSION]= context.applicationContext?.packageName?:""
            eventData[CLConstant.PAYU_SDK_VERSION]= BuildConfig.VERSION_NAME
            eventData[CLConstant.PAYU_BOLT_LOGGING_SDK]=
                BuildConfig.LIBRARY_PACKAGE_NAME + " " + BuildConfig.VERSION_NAME

            eventData[CLConstant.PAYU_DEVICE_TYPE] = PayUUpiConstant.PLATFORM_VALUE
            val deviceName = Build.MANUFACTURER + " " + Build.MODEL
            val deviceOsVersion = Build.VERSION.SDK_INT
            val deviceTotalMemory = getTotalMemoryInfo(context)
            val availMemory = getAvailableMemoryInfo(context)
            var memoryInfo: Long = 0
            try {
                memoryInfo = availMemory * 100 / deviceTotalMemory
            } catch (ex: Exception) {
                Log.d("AnalyticsUtils", "getAnalyticsJsonWithBasicProperties " + ex.message)
            }
            eventData[CLConstant.PAYU_DEVICE_DETAILS]=
            deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
            (txnId ?: InternalConfig.handshakeId)?.let {
                eventData[CLConstant.PAYU_CT_TXN_ID] = it
            }
            InternalConfig.config?.merchantKey?.let {
                eventData[CLConstant.PAYU_EVENT_MERCHANT_KEY] = it
            }
            (InternalConfig.config?.refId)?.let {
                eventMap[CLConstant.PAYU_EVENT_IDENTITY] = it.plus(InternalConfig.config?.merchantKey)
                eventData[CLConstant.PAYU_EVENT_REF_ID] = it
            }
            eventMap[CLConstant.PAYU_EVENT_TYPE] = CLConstant.PAYU_CT_EVENT
            eventMap[CLConstant.PAYU_EVENT_DATA] = eventData

            logCleverTapData(context, eventMap)
        } catch (e: Exception) {
            Log.d("PAYU", e.message.toString())
        }
    }

    private fun logCleverTapData(context: Context, eventData: Map<String, Any>) {
        val cleverTapAnalytics =
            AnalyticsFactory(context, AnalyticsConfig().apply {
                initiatorIdentifier = "com.payu.upipluginui"
                ctAccountId = BuildConfig.ctAccountId
                ctPassCode = BuildConfig.ctPassCode
            }).getAnalyticsClass(AnalyticsType.CLEVERTAP) as ClevertapAnalytics
        val json = JSONObject(eventData).toString()
        Log.d("cleverTapAnalytics", json)
        cleverTapAnalytics.log(json)
    }

    private fun getCurrentDateInDDMMYYHHMMSSFormat(): String {
        val sdf = SimpleDateFormat(DD_MM_YY_HH_MM_SS_Z_DATE_FORMAT, Locale.ENGLISH)
        return sdf.format(Date())
    }

    private fun getIntegratedSdkPackageName(): String? {
        val stackTraceElements = Thread.currentThread().stackTrace
        var previousStack: StackTraceElement? = null
        for (i in stackTraceElements.indices) {
            val element = stackTraceElements[i]
            if (element.className.startsWith("com.payu")) {
                previousStack = element
            }
        }
        return previousStack?.className
    }

    private fun getTimeDifferenceInSeconds(apiCalledTime: Long, currentTime: Long) =
        (currentTime - apiCalledTime).toString()

    fun setCTLogsForUPIWithData(
        context: Context,
        eventName: String,
        eventData: HashMap<String, Any> = HashMap(),
        type: String? = null,
        status: String? = null,
        startTime: Long = 0,
        txnId: String? = null
    ) {
        type?.let { eventData[CLConstant.PAYU_EVENT_TYPE] = type }
        status?.let { eventData[CLConstant.PAYU_EVENT_STATUS] = status }
        if (startTime > 0) {
            val timeValue = getTimeDifferenceInSeconds(startTime, System.currentTimeMillis())
            eventData[CLConstant.PAYU_EVENT_TIME] = timeValue
            eventData[CLConstant.PAYU_EVENT_TIME_MILISECONDS] = timeValue.toInt() * 1000
        }
        logEventForCT(context, eventName, eventData, txnId)
    }

    fun prepareAuthTokenParamsJson(postRequest: String?): String {
        try {
            val authTokenJsonObject = JSONObject(postRequest)
            val jsonObject = JSONObject()
            jsonObject.put(PayUUpiConstant.PAYU_CHECKSUM, authTokenJsonObject.get(PayUUpiConstant.PAYU_CHECKSUM))
            jsonObject.put(PayUUpiConstant.PAYU_UNQ_TXNID, authTokenJsonObject.get(PayUUpiConstant.PAYU_UNQ_TXNID))
            jsonObject.put(PayUUpiConstant.PAYU_MOBILE, Utils.getMaskedMobileNumber(authTokenJsonObject.get(PayUUpiConstant.PAYU_MOBILE).toString()))
            jsonObject.put(PayUUpiConstant.PAYU_EMAIL_ID, authTokenJsonObject.get(PayUUpiConstant.PAYU_EMAIL_ID))
            return jsonObject.toString()
        }catch (e : Exception){
            return ""
        }

    }

    fun logClevertapCTAClickedEvent(
        context: Context,
        ctaName: String,
        type: String,
        map: HashMap<String, Any> = HashMap(),
        screenName: String? = null
    ) {
        map[CLConstant.PAYU_CT_NAME] = ctaName
        map[CLConstant.PAYU_EVENT_TYPE] = type
        screenName?.let { map[CLConstant.PAYU_CT_SCREEN] = it }
        setCTLogsForUPIWithData(context, CLConstant.PAYU_CT_CTA_CLICKED, map)
    }
}