package com.payu.upibolt.utils

import android.content.Context
import android.provider.Settings
import android.telephony.TelephonyManager
import android.text.TextUtils
import android.util.Log
import com.olive.upi.transport.model.Account
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.enums.PluginType
import com.payu.commonmodelssdk.model.response.PayUCustomerBankAccounts
import com.payu.commonmodelssdk.model.response.PayUAccountDetail
import com.payu.upibolt.utils.InternalConfig
import org.json.JSONArray
import org.json.JSONObject
import java.util.*
import com.payu.commonmodelssdk.constants.PayUUpiConstant.COMMAND
import com.payu.commonmodelssdk.constants.PayUUpiConstant.HASH
import com.payu.commonmodelssdk.constants.PayUUpiConstant.KEY
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_IMEI
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_UD_ID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_VERIFY_PAYMENT
import com.payu.commonmodelssdk.constants.PayUUpiConstant.VAR1
import com.payu.commonmodelssdk.model.request.VerifyApiRequest
import java.util.UUID
import javax.crypto.Cipher
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

internal object Utils {

    fun getDeviceId(context: Context): String? {
        var deviceId: String? = Settings.Secure.getString(
            context.contentResolver, Settings.Secure.ANDROID_ID
        )
        if (TextUtils.isEmpty(deviceId)) {
            val idOne = UUID.randomUUID().toString()
            idOne.replace("-", "")
            deviceId = idOne
        }
        return deviceId
    }

    internal fun hexStringToByteArray(hexString: String): ByteArray {
        val bytes = ByteArray(hexString.length / 2)
        for (i in bytes.indices) {
            val index = i * 2
            val value = hexString.substring(index, index + 2).toInt(16)
            bytes[i] = value.toByte()
        }
        return bytes
    }

    fun encrypt(key: ByteArray, data: ByteArray?): ByteArray {
        val secretKeySpec = SecretKeySpec(key, PayUUpiConstant.PAYU_AES_ALGO)
        val iv = ByteArray(16)
        val ivSpec = IvParameterSpec(iv)
        val acipher: Cipher = Cipher.getInstance(PayUUpiConstant.PAYU_AES_TRANSFORMATION)
        acipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, ivSpec)
        return acipher.doFinal(data)
    }

    internal fun isValidInitSDKParams(): Pair<Boolean, String?> {
        if (InternalConfig.appId.isNullOrEmpty()) {
            return Pair(false, InternalConfig.appId)
        }
        if (InternalConfig.config?.merchantKey.isNullOrEmpty()) {
            return Pair(false, InternalConfig.config?.merchantKey)
        }
        if (InternalConfig.subscriptionId.isNullOrEmpty()) {
            return Pair(false, InternalConfig.subscriptionId)
        }
        if (InternalConfig.mobile.isNullOrEmpty()) {
            return Pair(false, InternalConfig.mobile)
        }
        if (InternalConfig.pluginType == PluginType.AXIS && isAxisDependencyAvailable().not()) {
            return Pair(false, "${PayUUpiConstant.PAYU_RUNTIME_DEPENDENCY} ${PluginType.AXIS.name}")
        }
        if ((InternalConfig.pluginType == PluginType.HDFC || InternalConfig.pluginType == PluginType.BHIM)
            && isHDFCDependencyAvailable().not()) {
            return Pair(false, "${PayUUpiConstant.PAYU_RUNTIME_DEPENDENCY} ${PluginType.HDFC.name}")
        }
        if (InternalConfig.pluginType == PluginType.BHIM && InternalConfig.config?.issuingBanks.isNullOrEmpty()) {
            return Pair(false, PayUUpiConstant.PAYU_ISSUING_BANK)
        }
        return Pair(true, null)
    }

    private fun isSdkAvailable(className: String?): Boolean {
        try {
            if (!className.isNullOrEmpty()) {
                val wrapperClassLoader = Utils::class.java.classLoader
                wrapperClassLoader?.loadClass(className)
                return true
            }
        } catch (e: Exception) {
            Log.d("AnalyticsUtils", "isSdkAvailable " + e.message)
        }
        return false
    }

    fun isAxisDependencyAvailable() = isSdkAvailable(PayUUpiConstant.PAYU_OLIVE_SDK_PACKAGE)
            && isSdkAvailable(PayUUpiConstant.PAYU_AXIS_WRAPPER_SDK_PACKAGE)
            && isSdkAvailable(PayUUpiConstant.PAYU_NPCI_CL_SDK_PACKAGE)

    fun isHDFCDependencyAvailable() = isSdkAvailable(PayUUpiConstant.PAYU_HDFC_SDK_PACKAGE)
            && isSdkAvailable(PayUUpiConstant.PAYU_NPCI_CL_SDK_PACKAGE)

    fun generateVerifyPaymentPostData(txnId: String): String {
        return JSONObject().apply {
            put(PayUUpiConstant.PAYU_TXN_ID, JSONArray().apply {
                put(txnId)
            })
        }.toString()
    }
    internal fun generatePostDataForCancelApi(token: String): String {
        val postData = StringBuilder()
        postData.append(concatParams(PayUUpiConstant.CANCEL_API_TOKEN, token))
        postData.append(concatParams(PayUUpiConstant.CANCEL_API_ACTION, PayUUpiConstant.CANCEL_API_ACTION_VALUE))
        postData.append(concatParams(PayUUpiConstant.CANCEL_API_FAILURE_REASON, PayUUpiConstant.CANCEL_API_FAILURE_REASON_VALUE))
        return trimAmpersand(postData.toString())
    }

    private fun concatParams(key: String, value: String): String {
        return "$key=$value&"
    }

    private fun trimAmpersand(data: String): String {
        return if (data[data.length - 1] == '&') data.substring(0, data.length - 1) else data
    }

    /**
     * Get Phones IMEI number, will be required in every communication with payu server.
     *
     * @return iemei
     */
    //TODO: DD name variables properly, also this method can never return null please change
    private fun getImei(context: Context): String {
        val imei: String = try {
            val mTelephonyMgr = context.getSystemService(
                Context.TELEPHONY_SERVICE
            ) as TelephonyManager
            mTelephonyMgr.deviceId
        } catch (e: java.lang.Exception) {
            PayUUpiConstant.DEFAULT
        }
        return imei
    }

    /**
     * Phone's UDID, will be required in every communication with payu server.
     *
     * @return UDID.
     */
    private fun getUdId(context: Context): String? {
        val udId: String? = try {
            Settings.Secure.getString(
                context.getContentResolver(),
                Settings.Secure.ANDROID_ID
            )
        } catch (e: java.lang.Exception) {
            PayUUpiConstant.DEFAULT
        }
        return udId
    }

    fun getTimeDifferenceInMilliSeconds(apiCalledTime: Long): Long {
        return System.currentTimeMillis() - apiCalledTime
    }

    fun formatMobileNumber(s: String): String {
        val mobile = s.replace(" ", "").trim()
        if (mobile.length == 10) {
            return "91$mobile"
        } else if (mobile.startsWith("+")) {
            return mobile.substringAfter("+")
        }
        return mobile
    }

    fun generateUUID(): String {
        val id = UUID.randomUUID().toString()
        return id.replace("-", "")
    }

    fun getMaskedMobileNumber(number : String) : String{
        val regex = ".*\\d(?=\\d{4})".toRegex()
        return number.replace(regex,"*")
    }

    fun getMaskedString(input: String?): String? {
        input?.let {
            if (input.length < 5) return input
            val midpoint = input.length - 5
            return "**" + input.substring(midpoint)
        }
        return null

    }
}