package com.payu.upibolt.payUHDFCImpl

import androidx.fragment.app.FragmentActivity
import com.payu.commonmodelssdk.constants.PayUResponseCodes
import com.payu.commonmodelssdk.constants.PayUResponseMessages
import com.payu.commonmodelssdk.constants.PayUResponseMessages.PAYU_SUCCESS_MESSAGE
import com.payu.commonmodelssdk.constants.PayUResponseTypes
import com.payu.commonmodelssdk.constants.PayUUpiConstant
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_AMOUNT
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_QUERY_LIST_SIZE
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_STATUS
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_TXN_ID
import com.payu.commonmodelssdk.constants.PayUUpiConstant.PAYU_TXN_LIST_SIZE
import com.payu.commonmodelssdk.constants.payuResponseTypeMap
import com.payu.commonmodelssdk.listeners.OTPVerificationInterface
import com.payu.commonmodelssdk.listeners.PayUUPIBoltCallBack
import com.payu.commonmodelssdk.listeners.ApiFailedCallback
import com.payu.commonmodelssdk.model.PgDetailsRequest
import com.payu.commonmodelssdk.model.request.PayUUPIParams
import com.payu.upibolt.listeners.UpiInitiateCallback
import com.payu.commonmodelssdk.model.response.FetchAccountsIInResponse
import com.payu.commonmodelssdk.model.response.PayUAccountDetail
import com.payu.commonmodelssdk.model.response.PayUBankData
import com.payu.commonmodelssdk.model.response.PayUCustomerBankAccounts
import com.payu.commonmodelssdk.model.response.PayUDisputeData
import com.payu.commonmodelssdk.model.response.PayUDisputeType
import com.payu.commonmodelssdk.model.response.PayUTransactionHistory
import com.payu.commonmodelssdk.model.response.PayUUPIBoltResponse
import com.payu.upibolt.utils.CommonPluginImpl
import com.payu.upibolt.utils.InternalConfig
import com.payu.upibolt.utils.Utils
import com.payu.upiboltcore.PayUUPIPlugin
import com.payu.upiboltcore.models.CustDisputeList
import com.payu.upiboltcore.models.AccountInfo
import com.payu.upiboltcore.models.CheckTransactionStatusResponseResult
import com.payu.upiboltcore.models.DeRegisterVPAResponseResult
import com.payu.upiboltcore.models.DisputeType
import com.payu.upiboltcore.models.PayUPluginInitParams
import com.payu.upiboltcore.models.RaiseDisputeResponseResult
import com.payu.upiboltcore.models.RegisterVPAResponseResult
import com.payu.upiboltcore.models.RemoveAccountResponseResult
import com.payu.upiboltcore.models.TransactionHistoryListResponseResult
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.Locale
import kotlin.String
import kotlin.collections.ArrayList


/**
 * PayUPluginUtils class used as helper class for HDFCImpl class
 * */
open class PayUHDFCPluginManager(
    activity: FragmentActivity
) : CommonPluginImpl(activity), PayUUPIBoltCallBack {

    override fun getRegisteredMobile(): String? {
        var number: String? = null
        ifPluginCoreDependencyExist {
            number = PayUUPIPlugin.getRegisteredMobileNumber(activity)
            if(number.isNullOrEmpty().not()) {
                number = Utils.formatMobileNumber(number ?: "")
            }
        }
        return number
    }

    override fun initiateWrapperSDK(
        callback: UpiInitiateCallback,
        otpVerificationInterface: OTPVerificationInterface?,
        apiFailureCallback: ApiFailedCallback?
    ) {
        this.upiInitiateCallback = callback
        if (hasSmsAndPhonePermissions()) {
            val pairValue = Utils.isValidInitSDKParams()
            if (pairValue.first) {
                InternalConfig.pgDetails?.let {
                    val pgDetailsRequest = PgDetailsRequest(it.mKey, it.pgName)
                    InternalConfig.config?.let { params ->
                        PayuHDFCAuthTokenManager(activity, params, pgDetailsRequest,
                            onSuccess = { authToken, isRefresh ->
                                startTime = System.currentTimeMillis()
                                val txnId = PayUUpiConstant.PAYU_TXNID_PREFIX + System.currentTimeMillis().toString()
                                val initParams = PayUPluginInitParams(
                                    mKey = params.merchantKey,
                                    appId = InternalConfig.appId ?: "",
                                    subscriptionId = InternalConfig.subscriptionId ?: "",
                                    phone = InternalConfig.mobile ?: "",
                                    email = params.email ?: "",
                                    refId = params.refId,
                                    initTxnId = txnId,
                                    isProduction = params.isProduction,
                                    bankIds = params.pluginTypes,
                                    otpVerificationInterface = otpVerificationInterface,
                                    apiFailureCallback = apiFailureCallback
                                )
//                                    if (isRefresh) {
//                                        PayUUPIPlugin.saveAuthToken(authToken)
//                                    } else {
                                otpVerificationInterface?.let {
                                    PayUUPIPlugin.initPayUSDK(
                                        activity, initParams,
                                        authToken, this@PayUHDFCPluginManager
                                    )
                                }
//                                    }

                            }, onFailure = { errorCode, errorMessage ->
                                callback.onSdKError(errorCode, errorMessage)
                            }
                        )
                    }
                }
            } else {
                callback.onSdKError(
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    "${PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE} - ${pairValue.second ?: ""}"
                )
            }
        } else {
            callback.onSdKError(
                PayUResponseCodes.PERMISSION_MISSING_ERROR_CODE,
                PayUResponseMessages.PAYU_PERMISSION_ERROR_MESSAGE
            )
        }
    }

    protected fun clearSDKData() {
        PayUUPIPlugin.clearData(activity)
        clearPayUSdk()
    }

    override fun onPayUSuccess(response: PayUUPIBoltResponse) {
        val responseTypeString =
            payuResponseTypeMap[response.responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN
        val timeDiff = payuStartTimeMap[responseTypeString] ?: 0
        when (response.responseType) {
            PayUResponseTypes.REQUEST_SDK_HANDSHAKE -> {
                val registeredMobile = getRegisteredMobile()
                if (response.code == PayUResponseCodes.PAYU_SUCCESS_STATUS
                    && registeredMobile.isNullOrEmpty().not()) {
                    storeSimInfoInSharedPref(
                        InternalConfig.subscriptionId ?: "",
                        registeredMobile ?: "", getCarrierNameFromPhone(registeredMobile)
                    )
                    logSuccessResponse(
                        responseTypeString,
                        JSONObject().apply {
                            put(
                                "mobileNo",
                                Utils.getMaskedMobileNumber(registeredMobile.toString())
                            )
                        }.toString(),
                        timeDiff = timeDiff,
                        referenceId = referenceId
                    )
                    upiInitiateCallback?.onSDKSuccess()
                } else {
                    logFailureResponse(
                        responseTypeString,
                        PayUResponseCodes.PAYU_HANDSHAKE_FAILED,
                        response.message ?: PayUUpiConstant.PAYU_EVENT_FAILURE,
                        timeDiff,
                        referenceId = referenceId
                    )
                    upiInitiateCallback?.onSdKError(
                        PayUResponseCodes.PAYU_HANDSHAKE_FAILED,
                        response.message
                    )
                }
            }

            PayUResponseTypes.REQUEST_LIST_BANKS -> {
                getBankListSuccessResponse(response, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_FETCH_ACCOUNT_V3,
            PayUResponseTypes.REQUEST_ALL_ACCOUNTS_V3 -> {
                getAccountListSuccessResponse(response, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3 -> {
                if (response.result is DeRegisterVPAResponseResult) {
                    val payUUPIBoltResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            null, response.result
                        )
                    postSuccessCallBack(
                        payUUPIBoltResponse,
                        responseTypeString,
                        response.result?.toString() ?: PAYU_SUCCESS_MESSAGE,
                        timeDiff
                    )
                    clearSDKData()
                } else {
                    val errorResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_GET_CUSTOMER_deregister_V3,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                    )
                    postFailureCallBack(errorResponse, responseTypeString, timeDiff)
                }
            }

            PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3 -> {
                if (response.result is RemoveAccountResponseResult) {
                    val payUUPIBoltResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            null, response.result
                        )
                    postSuccessCallBack(
                        payUUPIBoltResponse,
                        responseTypeString,
                        response.result?.toString() ?: PAYU_SUCCESS_MESSAGE,
                        timeDiff
                    )
                } else {
                    val payUUPIBoltResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_GET_ACCOUNT_REMOVE_V3,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                    )
                    postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
                }
            }

            PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3 -> {
                (response.result as? RegisterVPAResponseResult)?.run {
                    val custBankList = ArrayList<PayUCustomerBankAccounts>()
                    custBankList.add(
                        PayUCustomerBankAccounts(
                            bankAccounts = PayUAccountDetail(
                                name = userInfo.name ?: "",
                                accRefNumber = userInfo.accountId ?: "",
                                maskedAccnumber = userInfo.accountNo ?: "",
                                vpa = userInfo.virtualAddress,
                                type = userInfo.accountType ?: "",
                                ifsc = userInfo.ifscCode ?: "",
                                iin = userInfo.iin ?: "", aeba = "",
                                mbeba = if (userInfo.mpinFlag == true) "Y" else "N",
                                formatType = userInfo.bankFormat ?: ""
                            )
                        )
                    )
                    val finalResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            null, custBankList
                        )
                    postSuccessCallBack(
                        finalResponse,
                        responseTypeString,
                        preparePluginCoreAccountJson(this).toString(),
                        timeDiff
                    )
                } ?: kotlin.run {
                    val payUUPIBoltResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_SAVE_ACCOUNT_V3,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                    )
                    postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
                }
            }

            PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3 -> {
                if (response.result is List<*>) {
                    val list = response.result as? List<TransactionHistoryListResponseResult>
                    val txnList = ArrayList<PayUTransactionHistory>()
                    if (list != null) {
                        for (txn in list) {
                            val transaction = PayUTransactionHistory(
                                tranid = txn.npciTxnId,
                                refid = txn.apiResp?.upiTransRefNo?.toString(),
                                dateTime = convertDateFormat(txn.addedOn),
                                amount = txn.amount,
                                debitAccount = txn.payerAccountNumber,
                                debitVpa = txn.payerVpa,
                                debitBankName = txn.payerBankName,
                                status = txn.txnStatus,
                                mobilenumber = txn.mobileNumber,
                                payUTxnId = txn.clientTxnId
                            )
                            txnList.add(transaction)
                        }
                    }
                    val payUUPIBoltResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3,
                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                        null, txnList
                    )
                    postSuccessCallBack(
                        payUUPIBoltResponse,
                        responseTypeString,
                        PAYU_TXN_LIST_SIZE + list?.size.toString(),
                        timeDiff
                    )
                    return
                }
                val payUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_GET_TRANSACTION_HISTORY_V3,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                )
                postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3 -> {
                (response.result as? RaiseDisputeResponseResult)?.run {
                    val disputeData = PayUDisputeData(
                        actionType = complaintInfo.actionType,
                        disputeType = complaintInfo.disputeType,
                        onBehalfOf = complaintInfo.onBehalfOf,
                        orgRrn = complaintInfo.orgRrn,
                        orgTxnDate = complaintInfo.orgTxnDate,
                        orgTxnId = complaintInfo.orgTxnId,
                        rrn = complaintInfo.rrn,
                        txnId = complaintInfo.txnId,
                        userMsg = complaintInfo.userMsg,
                        crn = complaintInfo.crn,
                        expectedResolDate = complaintInfo.expectedResolDate,
                        ticketNo = complaintInfo.ticketNo,
                    )
                    val payUUPIBoltResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            null, disputeData
                        )
                    postSuccessCallBack(
                        payUUPIBoltResponse,
                        responseTypeString,
                        complaintInfo.getJSONObject().toString(),
                        timeDiff
                    )
                } ?: run {
                    val payUUPIBoltResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_GET_RAISE_QUERY_V3,
                            PayUResponseCodes.PAYU_FAILED_STATUS,
                            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                        )
                    postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
                }
            }

            PayUResponseTypes.REQUEST_LIST_QUERIES_V3 -> {
                if (response.result is ArrayList<*>) {
                    val list = response.result as? ArrayList<CustDisputeList>
                    val disputeList = ArrayList<PayUTransactionHistory>()
                    if (list != null) {
                        for (txn in list) {
                            val transaction = PayUTransactionHistory(
                                tranid = txn.txnid,
                                queryStatus = txn.status,
                                amount = txn.amount,
                                queryid = txn.ticketNo,
                                querydate = txn.crtDate
                            )
                            disputeList.add(transaction)
                        }

                        val payUUPIBoltResponse =
                            PayUUPIBoltResponse(
                                PayUResponseTypes.REQUEST_LIST_QUERIES_V3,
                                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                null, disputeList
                            )
                        postSuccessCallBack(
                            payUUPIBoltResponse,
                            responseTypeString,
                            PAYU_QUERY_LIST_SIZE + list.size.toString(),
                            timeDiff
                        )
                        return
                    }
                }
                val payUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_LIST_QUERIES_V3,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                )
                postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_QUERY_TYPE_LIST -> {
                if (response.result is ArrayList<*>) {
                    val list = response.result as? List<DisputeType>
                    list?.let {
                        val disputeList = it.map { type ->
                            PayUDisputeType(type.reasonCode, type.reasonDesc)
                        }
                        val payUUPIBoltResponse =
                            PayUUPIBoltResponse(
                                PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                                null, disputeList
                            )
                        postSuccessCallBack(
                            payUUPIBoltResponse,
                            responseTypeString,
                            PayUUpiConstant.PAYU_QUERY_TYPE_LIST_SIZE + list.size.toString(),
                            timeDiff
                        )
                        return
                    }
                }
                val payUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_QUERY_TYPE_LIST,
                    PayUResponseCodes.PAYU_FAILED_STATUS,
                    PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                )
                postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_GET_BALANCE -> {
                handleBalanceCheckResponse(response, responseTypeString, timeDiff)
            }

            PayUResponseTypes.REQUEST_PAY -> {
                val payUUPIBoltResponse =
                    PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_PAY,
                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                        null, response.result
                    )
                postSuccessCallBack(
                    payUUPIBoltResponse,
                    responseTypeString,
                    response.result?.toString() ?: response.message ?: PAYU_SUCCESS_MESSAGE,
                    timeDiff
                )
            }

            PayUResponseTypes.REQUEST_GET_CHANGE_MPIN -> {
                val payUUPIBoltResponse =
                    PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_GET_CHANGE_MPIN,
                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                        null, response.result
                    )
                postSuccessCallBack(
                    payUUPIBoltResponse,
                    responseTypeString,
                    response.result?.toString() ?: response.message ?: PAYU_SUCCESS_MESSAGE,
                    timeDiff
                )
            }

            PayUResponseTypes.REQUEST_ADD_ACCOUNT -> {
                val payUUPIBoltResponse =
                    PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_ADD_ACCOUNT,
                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                        null, response.result
                    )
                postSuccessCallBack(
                    payUUPIBoltResponse,
                    responseTypeString,
                    response.result?.toString() ?: response.message ?: PAYU_SUCCESS_MESSAGE,
                    timeDiff
                )
            }

            PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG -> {
                val payUUPIBoltResponse =
                    PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_ACCOUNT_MOBILE_REG,
                        PayUResponseCodes.PAYU_SUCCESS_STATUS,
                        null, response.result
                    )
                postSuccessCallBack(
                    payUUPIBoltResponse, responseTypeString,
                    response.result?.toString() ?: response.message ?: PAYU_SUCCESS_MESSAGE,
                    timeDiff
                )
            }

            PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS -> {
                if (response.result is CheckTransactionStatusResponseResult) {
                    val res = response.result as CheckTransactionStatusResponseResult
                    val transaction = PayUTransactionHistory(
                        tranid = res.txnStatusDetails.txnId,
                        queryStatus = res.txnStatusDetails.result,
                        amount = res.amount
                    )
                    val payUUPIBoltResponse =
                        PayUUPIBoltResponse(
                            PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS,
                            PayUResponseCodes.PAYU_SUCCESS_STATUS,
                            null, transaction
                        )
                    val json = JSONObject().apply {
                        put(PAYU_TXN_ID, res.txnStatusDetails.txnId)
                        put(PAYU_STATUS, res.txnStatusDetails.result)
                        put(PAYU_AMOUNT, res.amount)
                    }
                    postSuccessCallBack(
                        payUUPIBoltResponse, responseTypeString, json.toString(), timeDiff
                    )
                } else {
                    val payUUPIBoltResponse = PayUUPIBoltResponse(
                        PayUResponseTypes.REQUEST_CHECK_DISPUTE_STATUS,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
                    )
                    postFailureCallBack(payUUPIBoltResponse, responseTypeString, timeDiff)
                }
            }
        }
    }

    override fun onPayUFailure(response: PayUUPIBoltResponse) {
        val responseTypeString =
            payuResponseTypeMap[response.responseType] ?: PayUUpiConstant.PAYU_REQUEST_UNKNOWN
        val timeDiff = payuStartTimeMap[responseTypeString] ?: 0
        if (response.responseType == PayUResponseTypes.REQUEST_SDK_HANDSHAKE) {
            logFailureResponse(
                responseTypeString, response.code,
                response.message ?: PayUUpiConstant.PAYU_EVENT_FAILURE,
                timeDiff,
                referenceId = referenceId
            )
            upiInitiateCallback?.onSdKError(
                response.code,
                response.message
            )
        } else {
            postFailureCallBack(response, responseTypeString, timeDiff)
        }
    }

    private fun getBankListSuccessResponse(
        response: PayUUPIBoltResponse,
        responseTypeString: String,
        timeDiff: Long
    ) {
        if (response.result is ArrayList<*>) {
            val bankDataList = response.result as ArrayList<PayUBankData>
            val payUUPIResponse = PayUUPIBoltResponse(
                PayUResponseTypes.REQUEST_LIST_BANKS,
                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                null, bankDataList
            )
            logSuccessResponse(
                responseTypeString,
                "count: ${bankDataList.size}",
                timeDiff,
                referenceId = referenceId
            )
            postSuccessCallBack(
                payUUPIResponse, responseTypeString, "count: ${bankDataList.size}", timeDiff
            )
            return
        }
        val payUUPIResponse = PayUUPIBoltResponse(
            PayUResponseTypes.REQUEST_LIST_BANKS,
            PayUResponseCodes.PAYU_FAILED_STATUS,
            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE,
            response
        )
        postFailureCallBack(payUUPIResponse, responseTypeString, timeDiff)
    }

    private fun getAccountListSuccessResponse(
        response: PayUUPIBoltResponse,
        responseTypeString: String,
        timeDiff: Long
    ) {
        if (response.result is List<*>) {
            val accountDataList = response.result as? ArrayList<AccountInfo>
            val accountList = accountDataList?.let {
                HDFCTransformerUtils.prepareAccountData(it).filter { acc ->
                    if (InternalConfig.config?.isProduction == true) {
                        acc.bankAccounts.type.isEmpty()
                                || acc.bankAccounts.type.uppercase() == PayUUpiConstant.PAYU_ACCOUNT_TYPE_SAVINGS
                                || acc.bankAccounts.type.uppercase() == PayUUpiConstant.PAYU_ACCOUNT_TYPE_CURRENT
                    } else true
                }
            }
            val result: Any? =
                if (response.responseType == PayUResponseTypes.REQUEST_FETCH_ACCOUNT_V3) {
                    FetchAccountsIInResponse(accountList, false)
                } else {
                    accountList
                }
            val payUUPIResponse = PayUUPIBoltResponse(
                response.responseType,
                PayUResponseCodes.PAYU_SUCCESS_STATUS,
                response.message,
                result
            )
            postSuccessCallBack(
                payUUPIResponse, responseTypeString,
                "count: ${accountDataList?.size ?: 0}", timeDiff
            )
            return
        }
        val payUUPIResponse = PayUUPIBoltResponse(
            response.responseType,
            PayUResponseCodes.PAYU_FAILED_STATUS,
            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
        )
        postFailureCallBack(payUUPIResponse, responseTypeString, timeDiff)
    }

    private fun handleBalanceCheckResponse(
        response: PayUUPIBoltResponse,
        responseTypeString: String,
        timeDiff: Long
    ) {
        if(response.result != null && response.result is String) {
            val balance = response.result.toString().toDoubleOrNull()
            if (balance != null) {
                val payUUPIBoltResponse = PayUUPIBoltResponse(
                    PayUResponseTypes.REQUEST_GET_BALANCE,
                    PayUResponseCodes.PAYU_SUCCESS_STATUS,
                    null, balance
                )
                postSuccessCallBack(
                    payUUPIBoltResponse,
                    responseTypeString,
                    payUUPIBoltResponse.message ?: "",
                    timeDiff
                )
                return
            }
        }
        val payUUPIResponse = PayUUPIBoltResponse(
            response.responseType,
            PayUResponseCodes.PAYU_FAILED_STATUS,
            PayUResponseMessages.PAYU_INVALID_RESPONSE_MESSAGE
        )
        postFailureCallBack(payUUPIResponse, responseTypeString, timeDiff)
    }

    private fun postSuccessCallBack(
        payUUPIResponse: PayUUPIBoltResponse,
        responseTypeString: String,
        data: String,
        timeDiff: Long
    ) {
        logSuccessResponse(
            responseTypeString,
            data,
            timeDiff,
            referenceId = referenceId
        )
        callBack[payUUPIResponse.responseType]?.onPayUSuccess(payUUPIResponse)
    }

    private fun postFailureCallBack(
        payUUPIResponse: PayUUPIBoltResponse,
        responseTypeString: String,
        timeDiff: Long
    ) {
        logFailureResponse(
            responseTypeString,
            payUUPIResponse.code,
            payUUPIResponse.message ?: "",
            timeDiff,
            referenceId = referenceId
        )
        callBack[payUUPIResponse.responseType]?.onPayUFailure(payUUPIResponse)
    }

    private fun convertDateFormat(
        inputDate: String
    ): String {
        val inputDateFormat = PayUUpiConstant.DD_MM_YY_HH_MM_SS_DATE_FORMAT
        val outputDateFormat = PayUUpiConstant.PAYU_TIME
        val inputFormat = SimpleDateFormat(inputDateFormat, Locale.getDefault())
        val outputFormat = SimpleDateFormat(outputDateFormat, Locale.getDefault())
        return try {
            val date = inputFormat.parse(inputDate)
            date?.let { outputFormat.format(date) } ?: ""
        } catch (e: Exception) {
            ""
        }
    }

    override fun isSessionValid(): Boolean {
        return System.currentTimeMillis() < InternalConfig.authTokenExpiryTime
    }

    override fun makePayment(payUUPIParams: PayUUPIParams) {
        val accountInfo = payUUPIParams.params?.accountDetail?.let { HDFCTransformerUtils.getAccount(it) }
        val merchId = InternalConfig.pgDetails?.merchId
        val mcc = InternalConfig.pgDetails?.mcc
        accountInfo?.let {
            PayUUPIPlugin.makePayment(
                payUUPIParams.params?.amount ?: "",
                accountInfo,
                initiatePayResponse?.merchantName ?: "",
                initiatePayResponse?.merchantVpa ?: "",
                "Payment",
                initiatePayResponse?.referenceId ?: "",
                mcc ?: "",
                merchId ?: "",
                payUUPIParams.params?.initMode ?: "10",
                payUUPIParams.params?.purpose ?: "00",
                payUUPIParams.params?.refUrl ?: "",
                payUUpiProCallBack = this@PayUHDFCPluginManager
            )
        }
    }

    protected fun ifPluginCoreDependencyExist(
        responseType: Int? = null,
        callBack: PayUUPIBoltCallBack? = null,
        ifExists: () -> Unit
    ) {
        if (Utils.isHDFCDependencyAvailable()) {
            referenceId = Utils.generateUUID()
            ifExists.invoke()
        } else {
            responseType?.let {
                callBack?.onPayUFailure(
                    PayUUPIBoltResponse(
                        responseType,
                        PayUResponseCodes.PAYU_FAILED_STATUS,
                        PayUResponseMessages.PAYU_MANDATORY_PARAM_MISSING_MESSAGE
                                + PayUUpiConstant.PAYU_RUNTIME_DEPENDENCY
                    )
                )
            }
        }
    }
}
