package com.payu.threeDS2.network

import android.os.Handler
import android.os.Looper
import com.payu.threeDS2.BuildConfig
import com.payu.threeDS2.constants.PayU3DS2Constants
import com.payu.threeDS2.interfaces.listeners.PayUAsyncTaskResponse
import com.payu.threeDS2.utils.Utils
import com.payu.threedsbase.constants.APIConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_RESPONSE_STATUS_CODE_1
import okhttp3.Call
import okhttp3.Callback
import okhttp3.MediaType
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import org.json.JSONObject
import java.io.IOException
import java.util.concurrent.TimeUnit


class PayU3DSAPICall {
    private var call: Call? = null

    fun makePostAPICall(
        httpMethod: String,
        payUNetworkData: PayUNetworkData,
        payUAsyncTaskResponse: PayUAsyncTaskResponse
    ) {
        val startTime = System.currentTimeMillis()
        if (httpMethod == APIConstants.HTTP_REQUEST_TYPE_POST && payUNetworkData.postRequest.isNullOrEmpty()) {
            payUAsyncTaskResponse.onFailure(
                SDK_RESPONSE_STATUS_CODE_1,
                PayU3DS2ErrorConstants.REQUEST_BODY_CANNOT_BE_NULL,
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )
            return
        }
        if (payUNetworkData.url.isNullOrEmpty()) {
            payUAsyncTaskResponse.onFailure(
                SDK_RESPONSE_STATUS_CODE_1,
                PayU3DS2ErrorConstants.REST_API_PATH_CANNOT_BE_NULL,
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )
            return
        }
        val okHttpClientBuilder: OkHttpClient.Builder = OkHttpClient.Builder()
        if (BuildConfig.DEBUG) {
            val logging = HttpLoggingInterceptor()
            logging.setLevel(HttpLoggingInterceptor.Level.BODY)
            okHttpClientBuilder.addInterceptor(logging)
        }
        val client = okHttpClientBuilder.connectTimeout(10, TimeUnit.SECONDS)
            .writeTimeout(10, TimeUnit.SECONDS)
            .readTimeout(10, TimeUnit.SECONDS).build()
        val mediaType: MediaType = payUNetworkData.contentType.toMediaType()

        val request: Request.Builder = Request.Builder()

        request.addHeader(APIConstants.HEADER_ACCEPT, APIConstants.APPLICATION_JSON)


        if (httpMethod == APIConstants.HTTP_REQUEST_TYPE_POST) {
            val body: RequestBody = payUNetworkData.postRequest!!.toRequestBody(mediaType)
            request.post(body)
            request.url(payUNetworkData.url!!)
        } else {
            var queryRequest = ""
            if (payUNetworkData.getRequest != null) {
                for ((key, value) in payUNetworkData.getRequest!!) {
                    queryRequest += "$key=$value&"
                }
            }
            request.url(payUNetworkData.url!! + "?${queryRequest.dropLast(1)}")
        }

        if (payUNetworkData.headerMap != null) {
            for ((key, value) in payUNetworkData.headerMap!!) {
                request.addHeader(key, value)

            }
        }
        call = client.newCall(request.build())
        call?.enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                val handler = Handler(Looper.getMainLooper())
                val runnable = Runnable {
                    payUAsyncTaskResponse.onFailure(
                        SDK_RESPONSE_STATUS_CODE_1,
                        PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE,
                        Utils.getTimeDifferenceInMilliSeconds(startTime)
                    )
                }
                handler.post(runnable)
            }

            override fun onResponse(call: Call, response: Response) {
                val apiResponse = response.body?.string()
                val handler = Handler(Looper.getMainLooper())
                val runnable = Runnable {
                    if (response.code == PayU3DS2ErrorConstants.GATEWAY_TIMEOUT_ERROR_CODE) {
                        payUAsyncTaskResponse.onFailure(
                            PayU3DS2ErrorConstants.GATEWAY_TIMEOUT_ERROR_CODE,
                            PayU3DS2ErrorConstants.GATEWAY_TIMEOUT_ERROR_MESSAGE,
                            Utils.getTimeDifferenceInMilliSeconds(startTime)
                        )
                    } else if (response.code >= PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_CODE) {
                        if (isValidJSON(apiResponse)) {
                            payUAsyncTaskResponse.onSuccess(
                                if (apiResponse.isNullOrEmpty()) PayU3DS2Constants.EMPTY_STRING else apiResponse,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                        } else {
                            payUAsyncTaskResponse.onFailure(
                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_CODE,
                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                        }
                    } else {
                        if (isValidJSON(apiResponse) && response.code == APIConstants.SUCCESS_API_CODE) {
                            payUAsyncTaskResponse.onSuccess(
                                if (apiResponse.isNullOrEmpty()) PayU3DS2Constants.EMPTY_STRING else apiResponse,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                        } else {
                            payUAsyncTaskResponse.onFailure(
                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_CODE,
                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                        }

                    }
                }
                handler.post(runnable)
            }
        })
    }

    private fun isValidJSON(response: String?): Boolean {
        try {
            JSONObject(response!!)
        } catch (e: Exception) {
            return false
        }
        return true
    }

}