package com.payu.threeDS2.utils

import android.app.Activity
import android.app.ActivityManager
import android.content.Context
import android.os.Build
import com.payu.payuanalytics.analytics.factory.AnalyticsFactory
import com.payu.payuanalytics.analytics.model.AnalyticsType
import com.payu.payuanalytics.analytics.model.PayUAnalytics
import com.payu.threeDS2.BuildConfig
import com.payu.threeDS2.config.InternalConfig
import com.payu.threeDS2.constants.LoggingConstants
import com.payu.threeDS2.constants.PayU3DS2Constants
import com.payu.threeDS2.constants.PayU3DS2Constants.Companion.PLATFORM
import com.payu.threeDS2.constants.PayU3DS2Constants.Companion.PLATFORM_VALUE
import com.payu.threeDS2.constants.PayU3DS2Constants.Companion.SDK_NAME_VALUE
import com.payu.threeDS2.enums.EventSeverity
import com.payu.threeDS2.enums.EventType
import com.payu.threeDS2.utils.Utils.getCurrentDateInDDMMYYHHMMSSFormat
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.lang.Exception

object LoggingUtils {
    private var payUAnalytics: PayUAnalytics? = null
    private var activity: Activity? = null

    fun logMessage(activity: Activity, key: String, msg: String) {
        if (payUAnalytics == null) {
            this.activity = activity
            payUAnalytics =
                AnalyticsFactory(activity).getAnalyticsClass(AnalyticsType.PAYU_ANALYTICS) as PayUAnalytics
        }

        val analyticsJson = getAnalyticsJsonWithBasicProperties(activity)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_KEY, key)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_VALUE, msg)
        payUAnalytics?.log(
            analyticsJson.toString()
        )
    }

    fun logMessage(
        activity: Activity,
        key: String,
        msg: String,
        eventSeverity: EventSeverity?,
        eventType: EventType,
        sdkTime: Long = 0
    ) {
        if (payUAnalytics == null) {
            this.activity = activity
            payUAnalytics =
                AnalyticsFactory(activity).getAnalyticsClass(AnalyticsType.PAYU_ANALYTICS) as PayUAnalytics
        }

        val analyticsJson = getAnalyticsJsonWithBasicProperties(activity)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_KEY, key)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_VALUE, msg)
        analyticsJson.put(PayU3DS2Constants.EVENT_SEVERITY, eventSeverity?.name)
        analyticsJson.put(PayU3DS2Constants.EVENT_TYPE, eventType.name)
        analyticsJson.put(PayU3DS2Constants.PAYU_SDK_TIME, sdkTime.toString())
        payUAnalytics?.log(
            analyticsJson.toString()
        )
    }


    fun logMessage(activity: Activity, key: String, msg: String, sdkTime: Long) {
        if (payUAnalytics == null) {
            this.activity = activity
            payUAnalytics =
                AnalyticsFactory(activity).getAnalyticsClass(AnalyticsType.PAYU_ANALYTICS) as PayUAnalytics
        }

        val analyticsJson = getAnalyticsJsonWithBasicProperties(activity)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_KEY, key)
        analyticsJson.put(PayU3DS2Constants.THREEDS_EVENT_VALUE, msg)
        analyticsJson.put(PayU3DS2Constants.PAYU_SDK_TIME, sdkTime.toString())
        payUAnalytics?.log(
            analyticsJson.toString()
        )
    }

    private fun getAnalyticsJsonWithBasicProperties(context: Context): JSONObject {
        val analyticsJson = JSONObject()
        analyticsJson.put(PayU3DS2Constants.THREEDS_TIMESTAMP, getCurrentDateInDDMMYYHHMMSSFormat())
        analyticsJson.put(PayU3DS2Constants.SDK_NAME, SDK_NAME_VALUE)
        analyticsJson.put(PayU3DS2Constants.SDK_INTEGRATION, getIntegratedSdkPackageName())
        analyticsJson.put(PayU3DS2Constants.THREEDS_SDK_VERSION, BuildConfig.VERSION_NAME)
        analyticsJson.put(PLATFORM, PLATFORM_VALUE)
        analyticsJson.put(
            PayU3DS2Constants.THREEDS_APPLICATION_VERSION,
            activity?.applicationContext?.packageName
        )
        analyticsJson.put(
            PayU3DS2Constants.THREEDS_LOGGING_SDK,
            BuildConfig.LIBRARY_PACKAGE_NAME + " " + BuildConfig.VERSION_NAME
        )
        analyticsJson.put(
            PayU3DS2Constants.THREEDS_TXN_ID,
            InternalConfig.requestId
        )
        analyticsJson.put(
            PayU3DS2Constants.THREEDS_MERCHANT_KEY,
            InternalConfig.key
        )
        val deviceName = Build.MANUFACTURER + "_" + Build.MODEL
        val deviceOsVersion = Build.VERSION.SDK_INT
        try {
            val deviceTotalMemory = getTotalMemoryInfo(context)
            val availMemory = getAvailableMemoryInfo(context)
            val memoryInfo = availMemory * 100 / deviceTotalMemory
            analyticsJson.put(
                PayU3DS2Constants.THREEDS_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion + "_" + memoryInfo + "%"
            )
        } catch (e: Exception) {
            analyticsJson.put(
                PayU3DS2Constants.THREEDS_DEVICE_DETAILS,
                deviceName + "_" + deviceOsVersion
            )
        }
        return analyticsJson
    }

    private fun getTotalMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return memoryInfo.totalMem / (1024 * 1024)
    }

    private fun getAvailableMemoryInfo(context: Context): Long {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        return memoryInfo.availMem / (1024 * 1024)
    }

    private fun getIntegratedSdkPackageName(): String? {
        val stackTraceElements = Thread.currentThread().stackTrace
        var previousStack: StackTraceElement? = null
        for (i in stackTraceElements.indices) {
            val element = stackTraceElements[i]
            if (element.className.startsWith("com.payu")) {
                previousStack = element
            }
        }
        return previousStack?.className
    }

    fun getAnalyticsString(sdkPlatformData: String?): String {
        var jsonArray = JSONArray()
        val jsonObject = JSONObject()
        try {
            jsonArray = if (null == sdkPlatformData) {
                JSONArray()
            } else {
                JSONArray(sdkPlatformData)
            }
            jsonObject.put(LoggingConstants.PLATFORM_KEY, LoggingConstants.PLATFORM_VALUE)
            jsonObject.put(LoggingConstants.NAME_KEY, LoggingConstants.NAME_VALUE)
            jsonObject.put(LoggingConstants.VERSION_KEY, BuildConfig.VERSION_NAME)
            jsonArray.put(jsonObject)
        } catch (e: JSONException) {
        }
        return jsonArray.toString()
    }
}