package com.payu.threeDS2.utils

import androidx.appcompat.app.AppCompatActivity
import com.payu.threeDS2.config.InternalConfig
import com.payu.threeDS2.config.PayU3DS2Config
import com.payu.threeDS2.constants.LoggingConstants.Companion.RBA_CHECK_DEVICE_STATUS
import com.payu.threeDS2.constants.LoggingConstants.Companion.RBA_INIT
import com.payu.threeDS2.constants.LoggingConstants.Companion.RBA_REGISTRATION
import com.payu.threeDS2.constants.LoggingConstants.Companion.RBA_TRANSACTION
import com.payu.threeDS2.enums.EventSeverity
import com.payu.threeDS2.enums.EventType
import com.payu.threeDS2.utils.Utils.mapUIThemeTridentity
import com.payu.threedsbase.constants.APIConstants
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_AUTH_TYPE
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_AUTH_TYPE_VALUE
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_BANK_LOGO
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_BIN_ID
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_CLIENT_ID
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_ENV
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_ENV_PROD_VALUE
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_ENV_UAT_VALUE
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_FALLBACK_REGISTRATION_TIMEOUT
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_FALLBACK_TRANSACTION_TIMEOUT
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_RETRY_COUNT
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_RETRY_COUNT_VALUE
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_THEME_CONFIG
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_TXN_ID
import com.payu.threedsbase.constants.APIConstants.Companion.RBA_UID
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_ALREADY_DEREGISTERED_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_CARD_OR_TEMP_CARD_ID_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_CHECK_REGISTRATION_FAILED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_CLIENT_ID_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_CONFIG_FAILED_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_DEVICE_DE_REGISTERED_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_DEVICE_DE_REGISTERED_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_MISSING_PARAMETERS
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_REGISTRATION_FAILED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_REGISTRATION_FAILED_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.RBA_TXN_ERROR
import com.payu.threedsbase.enums.TridentityFlow
import com.payu.threedsbase.interfaces.listeners.PayU3DS2BaseCallback
import com.payu.threedsui.SdkUiInitializer
import com.payu.threedsui.uiCustomisation.UICustomisation
import com.wibmo.tridentity.sdk.TridentitySDK

import com.wibmo.tridentity_headless.di.ConfigStatusCallback
import com.wibmo.tridentity_headless.di.RegistrationStatusCallBack
import com.wibmo.tridentity_headless.di.UpdateTransactionCallback
import org.json.JSONObject
import java.lang.ref.WeakReference

internal object TridentityHelper {
    private var appCompatActivity: WeakReference<AppCompatActivity>? = null
    private var clientId: String? = null
    private var tempCardId: String? = null
    private var cardId: String? = null
    private var callback: PayU3DS2BaseCallback? = null
    private var bankLogo: String? = null
    private var networkLogo: String? = null
    private var txnId: String? = null


    /**
     * Starts the Tridentity flow based on the provided flow type.
     *
     * This method initializes the necessary parameters and configures the SDK.
     * Depending on the flow type, it either initiates the registration process
     * or checks the registration status.
     *
     * @param appCompatActivity The current activity.
     * @param bin The BIN (Bank Identification Number) of the card.
     * @param clientId The client ID.
     * @param tempCardId The temporary card ID.
     * @param cardId The card ID.
     * @param flowType The type of Tridentity flow (registration or transaction).
     * @param callback The callback to handle success and error responses.
     */
    fun startTridentityFlow(
        appCompatActivity: AppCompatActivity,
        isProduction: Boolean,
        bin: String?,
        clientId: String?,
        tempCardId: String?,
        cardId: String?,
        bankLogo: String?,
        networkLogo: String?,
        registrationTimeout: Int?,
        transactionTimeout: Int?,
        txnId: String?,
        flowType: TridentityFlow,
        threedsUiCustomization: PayU3DS2Config?,
        callback: PayU3DS2BaseCallback
    ) {
        if (clientId.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                appCompatActivity,
                RBA_INIT,
                RBA_CLIENT_ID_NULL,
                EventSeverity.HIGH, EventType.ERROR
            )
            callback.onError(RBA_MISSING_PARAMETERS, RBA_CLIENT_ID_NULL)
        } else if (tempCardId.isNullOrEmpty() && cardId.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                appCompatActivity,
                RBA_INIT,
                RBA_CARD_OR_TEMP_CARD_ID_NULL,
                EventSeverity.HIGH, EventType.ERROR
            )
            callback.onError(RBA_MISSING_PARAMETERS, RBA_CARD_OR_TEMP_CARD_ID_NULL)
        }


        this.appCompatActivity = WeakReference(appCompatActivity)
        this.clientId = clientId
        this.tempCardId = tempCardId
        this.cardId = cardId
        this.bankLogo = bankLogo
        this.networkLogo = networkLogo
        this.callback = callback
        this.txnId = txnId

        val configObject = JSONObject().apply {
            put(RBA_ENV, if (isProduction) RBA_ENV_PROD_VALUE.uppercase() else RBA_ENV_UAT_VALUE.uppercase()) // optional
            put(RBA_CLIENT_ID, clientId)
            put(RBA_AUTH_TYPE, RBA_AUTH_TYPE_VALUE)
            put(RBA_RETRY_COUNT, RBA_RETRY_COUNT_VALUE)
            put(RBA_BIN_ID, bin)
            put(RBA_BANK_LOGO, bankLogo)
            put(RBA_FALLBACK_REGISTRATION_TIMEOUT, registrationTimeout)
            put(RBA_FALLBACK_TRANSACTION_TIMEOUT, transactionTimeout)
        }

        var themeConfig = com.wibmo.tridentity.sdk.theme.ThemeConfig()
        if (threedsUiCustomization?.uiCustomisation != null) {
            val mapUIThemeTridentity = mapUIThemeTridentity(threedsUiCustomization)
            mapUIThemeTridentity?.let { themeConfig = it }
        }

        configObject.put(RBA_THEME_CONFIG, themeConfig)

        LoggingUtils.logMessage(
            appCompatActivity,
            RBA_INIT,
            "$RBA_INIT $configObject",
            null, EventType.INFO
        )

        TridentitySDK.getInstance()
            .configSdk(appCompatActivity, configObject, object : ConfigStatusCallback {
                override fun onSuccess(event: JSONObject) {
                    when (flowType) {
                        TridentityFlow.REGISTRATION -> {
                            startRegistration()
                        }
                        TridentityFlow.TRANSACTION -> {
                            checkRegistrationStatus(threedsUiCustomization?.uiCustomisation)
                        }
                        TridentityFlow.OOB_INITIATED -> {
                            startTransaction()
                        }
                    }
                }

                override fun onError(code: Int, error: String) {
                    LoggingUtils.logMessage(
                        appCompatActivity,
                        RBA_INIT,
                        "$code $error",
                        EventSeverity.HIGH, EventType.ERROR
                    )
                    InternalConfig.deviceDeregistered = true
                    callback.onError(RBA_CONFIG_FAILED_CODE, error)
                }
            })
    }


    /**
     * Initiates the registration process using the Tridentity SDK.
     *
     * This method creates a JSON object with the temporary card ID and calls the
     * `initiateRegistration` method of the Tridentity SDK. It handles the success
     * and error responses through the `RegistrationStatusCallBack`. Depending on the
     * registration status, it either calls the success callback or logs an error message.
     */
    private fun startRegistration() {
        val regObject = JSONObject()
        regObject.put(RBA_UID, if (tempCardId.isNullOrEmpty().not()) tempCardId else cardId)
        LoggingUtils.logMessage(
            appCompatActivity?.get()!!,
            RBA_REGISTRATION,
            "$RBA_REGISTRATION $regObject",
            null, EventType.INFO
        )
        TridentitySDK.getInstance().initiateRegistration(
            appCompatActivity?.get()!!, regObject,
            object : RegistrationStatusCallBack {
                override fun onSuccess(event: JSONObject) {
                    getRegistrationStatusFromResponse(event)?.let { registrationStatus ->
                        if (registrationStatus.equals(
                                APIConstants.REG_SUCCESS_STATUS,
                                ignoreCase = true
                            )
                        ) {
                            callback?.onSuccess(registrationStatus)
                        } else {
                            LoggingUtils.logMessage(
                                appCompatActivity?.get()!!,
                                RBA_REGISTRATION,
                                RBA_REGISTRATION_FAILED,
                                EventSeverity.HIGH, EventType.ERROR
                            )
                            callback?.onError(
                                RBA_REGISTRATION_FAILED_ERROR_CODE,
                                RBA_REGISTRATION_FAILED
                            )
                        }
                    } ?: run {
                        LoggingUtils.logMessage(
                            appCompatActivity?.get()!!,
                            RBA_REGISTRATION,
                            RBA_REGISTRATION_FAILED,
                            EventSeverity.HIGH, EventType.ERROR
                        )
                        callback?.onError(
                            RBA_REGISTRATION_FAILED_ERROR_CODE,
                            RBA_REGISTRATION_FAILED
                        )
                    }
                }

                override fun onError(code: Int, error: String) {
                    LoggingUtils.logMessage(
                        appCompatActivity?.get()!!,
                        RBA_REGISTRATION,
                        "$code $error",
                        EventSeverity.HIGH, EventType.ERROR
                    )
                    callback?.onError(RBA_REGISTRATION_FAILED_ERROR_CODE, error)
                }
            })
    }

    /**
     * Checks the registration status of the user and proceeds accordingly.
     *
     * This method creates a JSON object with the temporary card ID and calls the
     * `checkRegistrationStatus` method of the Tridentity SDK. It handles the success
     * and error responses through the `RegistrationStatusCallBack`. Depending on the
     * registration status, it either shows the authentication mode selection screen
     * or the de-registration screen.
     */
    private fun checkRegistrationStatus(themeConfig: UICustomisation?) {
        val regStatusObject = JSONObject()
        regStatusObject.put(RBA_UID, cardId)
        regStatusObject.put(RBA_CLIENT_ID, clientId)
        LoggingUtils.logMessage(
            appCompatActivity?.get()!!,
            RBA_CHECK_DEVICE_STATUS,
            "$RBA_CHECK_DEVICE_STATUS $regStatusObject",
            null, EventType.INFO
        )

        TridentitySDK.getInstance()
            .checkRegistrationStatus(
                appCompatActivity?.get()!!,
                regStatusObject,
                object : RegistrationStatusCallBack {
                    override fun onSuccess(event: JSONObject) {

                        InternalConfig.deviceDeregistered = false
                        val registrationStatusFromResponse =
                            getRegistrationStatusFromResponse(event)
                        if (registrationStatusFromResponse != null &&
                            registrationStatusFromResponse.equals(
                                APIConstants.REG_SUCCESS_STATUS,
                                ignoreCase = true
                            )
                        ) {
                            callback?.onSuccess(registrationStatusFromResponse)
                        } else {
                            LoggingUtils.logMessage(
                                appCompatActivity?.get()!!,
                                RBA_CHECK_DEVICE_STATUS,
                                "$RBA_DEVICE_DE_REGISTERED_ERROR_CODE $RBA_DEVICE_DE_REGISTERED_ERROR_MESSAGE",
                                EventSeverity.HIGH, EventType.INFO
                            )

                            callback?.onError(
                                RBA_DEVICE_DE_REGISTERED_ERROR_CODE,
                                RBA_DEVICE_DE_REGISTERED_ERROR_MESSAGE
                            )

                        }
                    }

                    override fun onError(code: Int, error: String) {

                        InternalConfig.deviceDeregistered = true
                        LoggingUtils.logMessage(
                            appCompatActivity?.get()!!,
                            RBA_CHECK_DEVICE_STATUS,
                            "$code $error",
                            EventSeverity.HIGH, EventType.ERROR
                        )

                        if (RBA_ALREADY_DEREGISTERED_CODE.contains(code)) {
                            SdkUiInitializer.showDeRegisterScreen(
                                appCompatActivity?.get()!!,
                                themeConfig,
                                bankLogo,
                                error
                            ) {
                                callback?.onError(
                                    RBA_DEVICE_DE_REGISTERED_ERROR_CODE,
                                    RBA_DEVICE_DE_REGISTERED_ERROR_MESSAGE
                                )
                            }
                        } else {
                            callback?.onError(RBA_CHECK_REGISTRATION_FAILED, error)
                        }
                    }
                })
    }

    private fun getRegistrationStatusFromResponse(json: JSONObject): String? {
        val subParam = json.optJSONObject(APIConstants.SUB_PARAM)
        return subParam?.optString(APIConstants.CUSTOMER_STATUS)
    }

    /**
     * Initiates a transaction process using the Tridentity SDK.
     *
     * This method creates a JSON object with transaction details and
     * calls the `processTransaction` method of the Tridentity SDK.
     * It handles the success and error responses through the `UpdateTransactionCallback`.
     */
    private fun startTransaction(
    ) {
        val transactionObject = JSONObject()
        transactionObject.put(RBA_TXN_ID, txnId)
        transactionObject.put(RBA_UID, cardId)
        transactionObject.put(RBA_CLIENT_ID, clientId)

        LoggingUtils.logMessage(
            appCompatActivity?.get()!!,
            RBA_CHECK_DEVICE_STATUS,
            "$RBA_TRANSACTION $transactionObject",
            null, EventType.INFO
        )
        TridentitySDK.getInstance()
            .processTransaction(
                appCompatActivity?.get()!!,
                transactionObject,
                object : UpdateTransactionCallback {

                    override fun onSuccess(event: JSONObject) {
                        callback?.onSuccess(event.optString(APIConstants.STATUS))
                    }

                    override fun onError(code: Int, error: String) {
                        LoggingUtils.logMessage(
                            appCompatActivity?.get()!!,
                            RBA_TRANSACTION,
                            "$code $error",
                            EventSeverity.HIGH, EventType.ERROR
                        )
                        callback?.onError(RBA_TXN_ERROR, error)
                    }
                })
    }
}