package com.payu.threeDS2

import android.app.Activity
import androidx.appcompat.app.AppCompatActivity
import com.payu.paymentparamhelper.PaymentParams
import com.payu.threeDS2.config.InternalConfig
import com.payu.threeDS2.config.PayU3DS2Config
import com.payu.threeDS2.network.GetBinInfoAPITask
import com.payu.threeDS2.network.PayU3DSAPILayer
import com.payu.threeDS2.utils.ThreeDSHelper
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.MERCHANT_KEY_CANNOT_BE_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.REQUEST_ID_CANNOT_BE_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_NOT_INITIALIZED_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_RESPONSE_STATUS_CODE_1
import com.payu.threedsbase.data.ACSActionParams
import com.payu.threedsbase.data.CardData
import com.payu.threedsbase.data.ChallengeParameter
import com.payu.threedsbase.data.PayU3DS2Response
import com.payu.threedsbase.data.apiRequest.CardBinInfoRequest
import com.payu.threedsbase.enums.ACSActionType
import com.payu.threedsbase.interfaces.listeners.PayU3DS2BaseCallback
import com.payu.threedsbase.interfaces.listeners.PayU3DS2Callback
import com.payu.threedsui.SdkUiInitializer
import com.payu.threedsui.interfaces.listeners.PayU3DS2PaymentBaseCallback
import com.payu.threedsui.interfaces.listeners.PayU3DS2PaymentCallback
import com.payu.threedsui.uiCustomisation.UICustomisation

object PayU3DS2 {
    private var config: PayU3DS2Config = PayU3DS2Config()
    private var apiLayer: PayU3DSAPILayer? = null

    @JvmStatic
    fun initialise(
        key: String,
        requestId: String,
        activity: Activity,
        config: PayU3DS2Config
    ): PayU3DS2Response {
        if (key.isEmpty())
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                MERCHANT_KEY_CANNOT_BE_NULL,
                null
            )

        if (requestId.isEmpty())
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                REQUEST_ID_CANNOT_BE_NULL,
                null
            )
        apiLayer = PayU3DSAPILayer(config)
        apiLayer?.setContext(activity)
        InternalConfig.key = key
        InternalConfig.requestId = requestId
        this.config = config
        return ThreeDSHelper.initialize(activity, config)
    }

    @JvmStatic
    fun extractDeviceDetails(cardData: CardData): PayU3DS2Response {
        if (!ThreeDSHelper.isInitialized()) {
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                SDK_NOT_INITIALIZED_ERROR_MESSAGE,
                null
            )
        }
        return ThreeDSHelper.extractDeviceDetails(cardData)
    }

    @JvmStatic
    fun initiateChallenge(
        activity: Activity,
        challengeParameter: ChallengeParameter,
        listener: PayU3DS2BaseCallback
    ) {
        if (!ThreeDSHelper.isInitialized()) {
            listener.onError(SDK_RESPONSE_STATUS_CODE_1, SDK_NOT_INITIALIZED_ERROR_MESSAGE)
            return
        }
        ThreeDSHelper.initiateChallenge(
            activity,
            challengeParameter,
            listener
        )
    }

    @JvmStatic
    fun cardBinInfo(
        cardBinInfoRequest: CardBinInfoRequest,
        callback: PayU3DS2Callback
    ) {
        if (!ThreeDSHelper.isInitialized()) {
            callback.onError(SDK_RESPONSE_STATUS_CODE_1, SDK_NOT_INITIALIZED_ERROR_MESSAGE)
            return
        }
        apiLayer?.let {
            GetBinInfoAPITask(it).callAPI(cardBinInfoRequest, callback)
        }
    }

    @JvmStatic
    fun initiatePayment(
        activity: AppCompatActivity,
        config: PayU3DS2Config,
        paymentParams: PaymentParams,
        callback: PayU3DS2PaymentCallback
    ) {
        InternalConfig.key = paymentParams.key
        InternalConfig.requestId = paymentParams.txnId
        InternalConfig.authenticateOnly = config.authenticateOnly
        callUi(activity, config, paymentParams, callback)
    }

    private fun callUi(
        activity: AppCompatActivity,
        config: PayU3DS2Config,
        paymentParams: PaymentParams,
        callback: PayU3DS2PaymentCallback
    ) {
        SdkUiInitializer.startPayment(
            activity,
            PayU3DSAPILayer(config, paymentParams),
            callback
        )
    }


    @JvmStatic
    fun action(
        acsActionType: ACSActionType,
        acsActionParams: ACSActionParams,
        callback: PayU3DS2BaseCallback
    ) {
        if (!ThreeDSHelper.isInitialized()) {
            return callback.onError(
                SDK_RESPONSE_STATUS_CODE_1,
                SDK_NOT_INITIALIZED_ERROR_MESSAGE
            )
        }
        ThreeDSHelper.acsAction(acsActionType, acsActionParams, callback)
    }

    fun startRedirectionFlow(
        activity: Activity,
        params: Map<String, Any>,
        uiCustomisation: UICustomisation,
        callback: PayU3DS2PaymentBaseCallback
    ) {
        this.config.uiCustomisation = uiCustomisation
        apiLayer = PayU3DSAPILayer(this.config)
        apiLayer?.let {
            it.setContext(activity)
            SdkUiInitializer.openWebFlow(
                activity,
                params,
                it,
                callback
            )
        }
    }
}