package com.payu.threeDS2.utils

import android.app.Activity
import com.payu.threeDS2.BuildConfig
import com.payu.threeDS2.config.InternalConfig
import com.payu.threeDS2.config.PayU3DS2Config
import com.payu.threeDS2.constants.LoggingConstants
import com.payu.threeDS2.constants.LoggingConstants.Companion.FAILURE
import com.payu.threeDS2.constants.LoggingConstants.Companion.PARQ_EXCEPTION
import com.payu.threeDS2.constants.LoggingConstants.Companion.RUN_TIME_EXCEPTION
import com.payu.threeDS2.constants.LoggingConstants.Companion.SUCCESS
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_ACS_INITIATED
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_ACTION_ERROR
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_ALREADY_INITIALIZED
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_CHALLENGE_FLOW
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_CHALLENGE_FLOW_INITIATED
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_DEVICE_COLLECTION
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_HEADLESS
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_INITIALIZATION_FAILED
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_INITIALIZATION_STARTED
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_INITIALIZATION_SUCCESS
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_SDK
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_SDK_CHALLENGE_RESPONSE
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_SDK_CHALLENGE_RESPONSE_PROTOCOL_ERROR
import com.payu.threeDS2.constants.LoggingConstants.Companion.THREE_DS_SDK_CHALLENGE_RESPONSE_RUNTIME_ERROR
import com.payu.threeDS2.constants.PayU3DS2Constants
import com.payu.threeDS2.utils.Utils.mapPayUActionTypeToWibmoActionType
import com.payu.threeDS2.utils.Utils.mapUICustomisationToWibmo
import com.payu.threedsbase.constants.APIConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_ACTION_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_ACTION_PARAMS_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_ACTION_PARAMS_NULL_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_INVALID_OTP_SUBMIT
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_OTP_DATA_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_OTP_RESEND_LIMIT_EXCEEDED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_OTP_SUBMISSION_FAILED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_RENDERING_TYPE_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_TXN_CANCELLED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACS_TXN_ID_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACTION_API_TIMEOUT
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACTION_TYPE_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.ACTION_TYPE_NULL_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.CARD_SCHEME_OR_THREE_DS_VERSION_NULL
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.CHALLENGE_TIMEDOUT_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.OTP_RESEND_LIMIT_EXCEEDED
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.OTP_RESENT_SUCCESSFULLY
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_ALREADY_INITIALIZED_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_RESPONSE_STATUS_CODE_0
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_RESPONSE_STATUS_CODE_1
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SOMETHING_WENT_WRONG_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_ACS_ERROR_CODE_001
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_ACS_ERROR_CODE_002
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_ACS_ERROR_CODE_003
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_ACS_ERROR_CODE_004
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_TRANSACTION_NOT_CREATED_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_TRANSACTION_NOT_CREATED_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.WIBMO_TRANSACTION_NULL
import com.payu.threedsbase.data.ACSActionParams
import com.payu.threedsbase.data.CardData
import com.payu.threedsbase.data.ChallengeParameter
import com.payu.threedsbase.data.HeadlessData
import com.payu.threedsbase.data.ImageDetails
import com.payu.threedsbase.data.PArqResponse
import com.payu.threedsbase.data.PayU3DS2ACSResponse
import com.payu.threedsbase.data.PayU3DS2Response
import com.payu.threedsbase.enums.ACSActionType
import com.payu.threedsbase.enums.CardScheme
import com.payu.threedsbase.interfaces.listeners.PayU3DS2BaseCallback
import com.wibmo.threeds2.sdk.ChallengeUIStatusReceiver
import com.wibmo.threeds2.sdk.StatusCallBack
import com.wibmo.threeds2.sdk.ThreeDS2RequestorHelper
import com.wibmo.threeds2.sdk.ThreeDS2Service
import com.wibmo.threeds2.sdk.Transaction
import com.wibmo.threeds2.sdk.cfg.ChallengeParameters
import com.wibmo.threeds2.sdk.cfg.ConfigParameters
import com.wibmo.threeds2.sdk.error.InvalidInputException
import com.wibmo.threeds2.sdk.error.SDKAlreadyInitializedException
import com.wibmo.threeds2.sdk.error.SDKRuntimeException
import com.wibmo.threeds2.sdk.event.ACSPageEvents
import com.wibmo.threeds2.sdk.event.ActionType
import com.wibmo.threeds2.sdk.event.CompletionEvent
import com.wibmo.threeds2.sdk.event.ProtocolErrorEvent
import com.wibmo.threeds2.sdk.event.RuntimeErrorEvent
import com.wibmo.threeds2.sdk.event.UIData
import com.wibmo.threeds2.sdk.impl.WibmoThreeDS2ServiceImpl
import java.util.Date


object ThreeDSHelper {
    private var threeDS2Service: ThreeDS2Service? = null
    private var transaction: Transaction? = null
    private var activity: Activity? = null
    private var isSeamless: Boolean? = null
    private var acsTransactionId: String? = null
    private var resendCount: Int = 0

    internal fun clean() {
        threeDS2Service = null
        transaction = null
    }

    internal fun isInitialized(): Boolean {
        return threeDS2Service != null
    }

    /**
     * Method to initialise Wibmo SDK if success will return warnings else will
     * return error
     */
    internal fun initialize(
        activity: Activity,
        config: PayU3DS2Config?
    ): PayU3DS2Response {
        clean()
        resendCount = 0
        this.activity = activity
        this.isSeamless = !config?.supportedUIMode.isNullOrEmpty()

        LoggingUtils.logMessage(
            activity,
            THREE_DS_SDK,
            THREE_DS_INITIALIZATION_STARTED
        )
        threeDS2Service = WibmoThreeDS2ServiceImpl()
        val wibmoUICustomisation = mapUICustomisationToWibmo(config!!.uiCustomisation)
        try {

            threeDS2Service!!.initialize(
                activity, generateConfigParam(config), APIConstants.LOCALE, wibmoUICustomisation
            )
            LoggingUtils.logMessage(
                activity,
                THREE_DS_SDK,
                THREE_DS_INITIALIZATION_SUCCESS
            )
            LoggingUtils.logMessage(
                activity,
                THREE_DS_SDK,
                config.toString()
            )

            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_0,
                null,
                Utils.mapWibmoWarningsToDeviceWarning(threeDS2Service!!.warnings)
            )


        } catch (e: InvalidInputException) {
            LoggingUtils.logMessage(
                activity,
                THREE_DS_SDK,
                THREE_DS_INITIALIZATION_FAILED + e.message
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                e.message!!,
                null
            )
        } catch (e: SDKAlreadyInitializedException) {
            LoggingUtils.logMessage(
                activity,
                THREE_DS_SDK,
                THREE_DS_ALREADY_INITIALIZED + e.message
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                SDK_ALREADY_INITIALIZED_ERROR_MESSAGE,
                null
            )
        } catch (e: SDKRuntimeException) {
            LoggingUtils.logMessage(
                activity,
                THREE_DS_SDK,
                RUN_TIME_EXCEPTION + e.message!!
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                e.message!!,
                null
            )
        }
    }

    /**
     * Method to create transaction at wibmo end using SDK
     */
    private fun createTransaction(cardScheme: CardScheme, threeDSVersion: String): Transaction? {
        if (cardScheme == CardScheme.MASTERCARD) {
            return threeDS2Service!!.createTransaction(
                BuildConfig.mastercardDSServerId,
                threeDSVersion
            )
        } else if (cardScheme == CardScheme.VISA) {
            return threeDS2Service!!.createTransaction(
                BuildConfig.visaDSServerId,
                threeDSVersion
            )
        }
        return null
    }

    /**
     * Method to extract device details(pArs), it will return device details if success else
     * respective error
     */
    internal fun extractDeviceDetails(cardData: CardData): PayU3DS2Response {
        val startTime = System.currentTimeMillis()
        if (cardData.cardScheme != null && cardData.threeDSVersion != null)
            transaction = createTransaction(cardData.cardScheme!!, cardData.threeDSVersion!!)
        else {
            LoggingUtils.logMessage(
                activity!!,
                THREE_DS_SDK,
                CARD_SCHEME_OR_THREE_DS_VERSION_NULL,
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                CARD_SCHEME_OR_THREE_DS_VERSION_NULL,
                null
            )
        }
        if (transaction == null) {
            LoggingUtils.logMessage(
                activity!!,
                THREE_DS_SDK,
                WIBMO_TRANSACTION_NOT_CREATED_ERROR_MESSAGE,
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                PayU3DS2ErrorConstants.SCHEME_NOT_RECOGNISED_ERROR_MESSAGE,
                null
            )
        }
        try {
            val pArq = ThreeDS2RequestorHelper.makePArq(
                transaction!!.authenticationRequestParameters, Date()
            )
            val payuPArq = PArqResponse(
                pArq.sdkAppID,
                pArq.sdkEncData,
                pArq.sdkEphemPubKey.crv,
                pArq.sdkEphemPubKey.kty,
                pArq.sdkEphemPubKey.x,
                pArq.sdkEphemPubKey.y,
                pArq.sdkTransID,
                pArq.sdkReferenceNumber
            )

            LoggingUtils.logMessage(
                activity!!,
                THREE_DS_SDK, payuPArq.toString(),
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )
            CleverTapLoggingUtils.logAnalytics(
                activity!!,
                THREE_DS_DEVICE_COLLECTION,
                SUCCESS,
                isSeamless
            )
            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_0,
                null,
                payuPArq
            )
        } catch (e: SDKRuntimeException) {
            LoggingUtils.logMessage(
                activity!!,
                THREE_DS_SDK,
                PARQ_EXCEPTION + e.message!!,
                Utils.getTimeDifferenceInMilliSeconds(startTime)
            )

            CleverTapLoggingUtils.logAnalytics(
                activity!!,
                THREE_DS_DEVICE_COLLECTION,
                FAILURE,
                isSeamless
            )

            return PayU3DS2Response(
                SDK_RESPONSE_STATUS_CODE_1,
                e.message!!,
                null
            )
        }
    }

    /**
     * Method to invoke challenge i.e.; to open screen where user will enter OTP to validate
     * transaction. It will return below values it success else error code and message.
     *  String sdkTransactionID;
     *  String transactionStatus;
     *  String data;
     *  String threeDSServerTransID;
     *  String acsTransID;
     *  String dsTransID;
     */
    internal fun initiateChallenge(
        activity: Activity, challengeParameter: ChallengeParameter, listener: PayU3DS2BaseCallback
    ) {
        if (transaction == null) {
            LoggingUtils.logMessage(
                activity, THREE_DS_SDK, WIBMO_TRANSACTION_NOT_CREATED_ERROR_MESSAGE
            )

            listener.onError(
                WIBMO_TRANSACTION_NOT_CREATED_ERROR_CODE,
                WIBMO_TRANSACTION_NOT_CREATED_ERROR_MESSAGE
            )
        } else {
            val challengeParameters = ChallengeParameters()
            challengeParameters.acsRefNumber = challengeParameter.acsRefNumber
            challengeParameters.acsSignedContent = challengeParameter.acsSignedContent
            challengeParameters.acsTransactionID = challengeParameter.acsTransactionID
            challengeParameters.set3DSServerTransactionID(challengeParameter.threeDSServerTransactionID)
            LoggingUtils.logMessage(
                activity,
                THREE_DS_CHALLENGE_FLOW,
                "$THREE_DS_CHALLENGE_FLOW_INITIATED $challengeParameter"
            )

            CleverTapLoggingUtils.logAnalytics(
                activity, THREE_DS_ACS_INITIATED, isSeamless = isSeamless
            )
            var startTime = System.currentTimeMillis()
            transaction!!.doChallenge(
                activity, challengeParameters, object : ChallengeUIStatusReceiver {
                    override fun completed(completionEvent: CompletionEvent) {
                        if (!activity.isDestroyed && !activity.isFinishing) LoggingUtils.logMessage(
                            activity,
                            THREE_DS_CHALLENGE_FLOW,
                            THREE_DS_SDK_CHALLENGE_RESPONSE + completionEvent.transactionStatus,
                            Utils.getTimeDifferenceInMilliSeconds(startTime)
                        )
                        listener.onSuccess(completionEvent.transactionStatus)
                    }

                    override fun cancelled(completionEvent: CompletionEvent) {
                        if (!activity.isDestroyed && !activity.isFinishing) LoggingUtils.logMessage(
                            activity,
                            THREE_DS_CHALLENGE_FLOW,
                            THREE_DS_SDK_CHALLENGE_RESPONSE + completionEvent.error,
                            Utils.getTimeDifferenceInMilliSeconds(startTime)
                        )
                        listener.onError(
                            PayU3DS2ErrorConstants.CHALLENGE_CANCEL_ERROR_CODE,
                            completionEvent.error
                        )
                    }


                    override fun timedout() {
                        if (!activity.isDestroyed && !activity.isFinishing) LoggingUtils.logMessage(
                            activity,
                            THREE_DS_CHALLENGE_FLOW,
                            THREE_DS_SDK_CHALLENGE_RESPONSE + PayU3DS2ErrorConstants.CHALLENGE_TIMEDOUT_ERROR_MESSAGE,
                            Utils.getTimeDifferenceInMilliSeconds(startTime)
                        )
                        listener.onError(
                            CHALLENGE_TIMEDOUT_ERROR_CODE,
                            PayU3DS2ErrorConstants.CHALLENGE_TIMEDOUT_ERROR_MESSAGE
                        )
                    }

                    override fun protocolError(protocolErrorEvent: ProtocolErrorEvent) {
                        transaction?.close()
                        if (protocolErrorEvent.errorMessage != null && !protocolErrorEvent.errorMessage.errorDescription.isNullOrEmpty()) {
                            LoggingUtils.logMessage(
                                activity,
                                THREE_DS_CHALLENGE_FLOW,
                                THREE_DS_SDK_CHALLENGE_RESPONSE + protocolErrorEvent.errorMessage.errorDescription,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                            listener.onError(
                                PayU3DS2ErrorConstants.CHALLENGE_PROTOCOL_ERROR_CODE,
                                protocolErrorEvent.errorMessage.errorDescription
                            )
                        } else {
                            LoggingUtils.logMessage(
                                activity,
                                THREE_DS_CHALLENGE_FLOW,
                                THREE_DS_SDK_CHALLENGE_RESPONSE + THREE_DS_SDK_CHALLENGE_RESPONSE_PROTOCOL_ERROR,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                            listener.onError(
                                PayU3DS2ErrorConstants.CHALLENGE_PROTOCOL_ERROR_CODE,
                                SOMETHING_WENT_WRONG_ERROR_MESSAGE
                            )
                        }

                    }

                    override fun runtimeError(runtimeErrorEvent: RuntimeErrorEvent) {
                        if (!runtimeErrorEvent.errorMessage.isNullOrEmpty()) {
                            LoggingUtils.logMessage(
                                activity,
                                THREE_DS_CHALLENGE_FLOW,
                                THREE_DS_SDK_CHALLENGE_RESPONSE + runtimeErrorEvent.errorMessage,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                            listener.onError(
                                PayU3DS2ErrorConstants.CHALLENGE_RUNTIME_ERROR_CODE,
                                runtimeErrorEvent.errorMessage
                            )
                        } else {
                            LoggingUtils.logMessage(
                                activity,
                                THREE_DS_CHALLENGE_FLOW,
                                THREE_DS_SDK_CHALLENGE_RESPONSE + THREE_DS_SDK_CHALLENGE_RESPONSE_RUNTIME_ERROR,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                            listener.onError(
                                PayU3DS2ErrorConstants.CHALLENGE_RUNTIME_ERROR_CODE,
                                SOMETHING_WENT_WRONG_ERROR_MESSAGE
                            )
                        }
                    }

                    override fun acsPageActionTaken(acsPageEvents: ACSPageEvents) {
                        if (acsPageEvents == ACSPageEvents.THREEDS_ACS_LOADED && !activity.isDestroyed && !activity.isFinishing) {
                            LoggingUtils.logMessage(
                                activity,
                                LoggingConstants.PAYU_THREE_DS_RENDER_TIME,
                                acsPageEvents.name,
                                Utils.getTimeDifferenceInMilliSeconds(startTime)
                            )
                        } else if (acsPageEvents == (ACSPageEvents.OTP_AUTO_SUBMIT) || acsPageEvents == ACSPageEvents.SUBMIT_OTP) startTime =
                            System.currentTimeMillis()
                        CleverTapLoggingUtils.logAnalytics(
                            activity, acsPageEvents.name, isSeamless = isSeamless
                        )
                    }

                    override fun uiResponseData(uiData: UIData) {
                        CleverTapLoggingUtils.logAnalytics(
                            activity, ACSPageEvents.THREEDS_ACS_LOADED.name, isSeamless = isSeamless
                        )
                        acsTransactionId = uiData.acsTransactionID
                        listener.onSuccess(
                            HeadlessData(
                                uiData.acsTransactionID,
                                uiData.acsRenderingType,
                                ImageDetails(
                                    uiData.issuerImage.medium,
                                    uiData.issuerImage.high,
                                    uiData.issuerImage.extraHigh
                                ),
                                ImageDetails(
                                    uiData.issuerImage.medium,
                                    uiData.issuerImage.high,
                                    uiData.issuerImage.extraHigh
                                )
                            )
                        )
                    }
                }, PayU3DS2Constants.SDK_CHALLENGE_TIMEOUT
            )
        }
    }

    /**
     * Method to generate config parameters required for initialisation of Wibmo SDK
     */
    private fun generateConfigParam(config: PayU3DS2Config): ConfigParameters {
        val configParameters = ConfigParameters()
        configParameters.licenseKey = BuildConfig.licenseKey
        configParameters.addParam(
            BuildConfig.visaPublicKey, BuildConfig.visaDSServerId
        )
        configParameters.addParam(
            BuildConfig.mastercardPublicKey, BuildConfig.mastercardDSServerId
        )
        configParameters.txnID = InternalConfig.requestId
        configParameters.isAutoReadEnabled = config.autoRead
        configParameters.isAutoSubmitEnable = config.autoSubmit
        configParameters.setCustomDialogLoader(
            config.showDefaultLoader, config.defaultProgressLoaderColor
        )
        if (!config.supportedUIMode.isNullOrEmpty()) configParameters.supportedUiMode =
            config.supportedUIMode?.toTypedArray()

        if (config.enableCustomizedOtpUIFlow) {

            Utils.validateCustomACSContent(config.amount, config.merchantName)
            configParameters.isEnableNoDetailUI = config.enableCustomizedOtpUIFlow
            configParameters.isShowTxnTimer = config.enableTxnTimeoutTimer
            configParameters.merchantName = config.merchantName
            var formattedSplitAmount = config.amount?.split(".")
            var formattedAmount = formattedSplitAmount!![0]
            if (formattedSplitAmount.size > 1 && formattedSplitAmount!![1] != "")
                formattedAmount = String.format("%.2f", config.amount?.toDouble())
            configParameters.txnAmount = "${config.currency.value}${formattedAmount}"
            configParameters.otpContent =
                if (config.acsContentConfig?.otpContent?.trim()
                        .isNullOrEmpty()
                ) PayU3DS2Constants.OTP_SENT_DEFAULT_MESSAGE else config.acsContentConfig?.otpContent
            if (!config.acsContentConfig?.submitButtonTitle?.trim().isNullOrEmpty())
                configParameters.submitButtonTitle = config.acsContentConfig?.submitButtonTitle
            if (!config.acsContentConfig?.resendButtonTitle?.trim().isNullOrEmpty())
                configParameters.resendButtonTitle = config.acsContentConfig?.resendButtonTitle

            if (!config.acsContentConfig?.resendInfoContent?.trim().isNullOrEmpty())
                configParameters.resendInfoString = config.acsContentConfig?.resendInfoContent
            if (!config.acsContentConfig?.maxResendInfoContent?.trim().isNullOrEmpty())
                configParameters.maxResendInfoString = config.acsContentConfig?.maxResendInfoContent
        }
        return configParameters
    }

    fun acsAction(
        acsActionType: ACSActionType?,
        acsActionParams: ACSActionParams?,
        callback: PayU3DS2BaseCallback
    ) {
        if (acsActionType == null) return callback.onError(
            ACTION_TYPE_NULL_ERROR_CODE, ACTION_TYPE_NULL
        )
        if (acsActionParams == null) return callback.onError(
            ACS_ACTION_PARAMS_NULL_ERROR_CODE, ACS_ACTION_PARAMS_NULL
        )
        if (acsActionParams.acsTransactionID.isNullOrEmpty()) return callback.onError(
            ACS_ACTION_PARAMS_NULL_ERROR_CODE, ACS_TXN_ID_NULL
        )
        if (acsActionParams.acsRenderingType.isNullOrEmpty()) return callback.onError(
            ACS_ACTION_PARAMS_NULL_ERROR_CODE, ACS_RENDERING_TYPE_NULL
        )
        if (acsActionType == ACSActionType.SUBMIT && acsActionParams.challengeData.isNullOrEmpty()) return callback.onError(
            ACS_ACTION_PARAMS_NULL_ERROR_CODE, ACS_OTP_DATA_NULL
        )
        if (transaction == null) return callback.onError(
            WIBMO_TRANSACTION_NOT_CREATED_ERROR_CODE, WIBMO_TRANSACTION_NULL
        )


        if (resendCount == 3 && acsTransactionId.equals(acsActionParams.acsTransactionID) &&
            acsActionType == ACSActionType.RESEND
        ) {
            return callback.onError(
                ACS_OTP_RESEND_LIMIT_EXCEEDED,
                OTP_RESEND_LIMIT_EXCEEDED
            )
        }

        if (acsActionType == ACSActionType.RESEND)
            resendCount++

        CleverTapLoggingUtils.logAnalytics(
            activity!!,
            if (acsActionType == ACSActionType.SUBMIT) ACSPageEvents.SUBMIT_OTP.name else ACSPageEvents.RESEND_OTP.name,
            isSeamless = isSeamless
        )

        val request = HashMap<String, Any>()
        request[APIConstants.ACS_TXN_ID] = acsActionParams.acsTransactionID.toString()
        request[APIConstants.ACS_RENDERING_TYPE] = acsActionParams.acsRenderingType.toString()

        if (!acsActionParams.challengeData.isNullOrEmpty()) request[APIConstants.OTP_CHALLENGE_DATA] =
            acsActionParams.challengeData.toString()
        transaction?.submitAction(
            mapPayUActionTypeToWibmoActionType(acsActionType),
            request,
            object : StatusCallBack {
                override fun onSuccess(
                    message: String?,
                    actionType: ActionType,
                    uiData: UIData?
                ) {
                    val payU3DS2ACSResponse = PayU3DS2ACSResponse(
                        if (actionType == ActionType.RESEND) OTP_RESENT_SUCCESSFULLY else message,
                        Utils.mapWibmoActionTypeToPayUActionType(actionType),
                        HeadlessData(
                            uiData?.acsTransactionID,
                            uiData?.acsRenderingType,

                            ImageDetails(
                                uiData?.issuerImage?.medium,
                                uiData?.issuerImage?.high,
                                uiData?.issuerImage?.extraHigh
                            ),
                            ImageDetails(
                                uiData?.issuerImage?.medium,
                                uiData?.issuerImage?.high,
                                uiData?.issuerImage?.extraHigh
                            )
                        )
                    )

                    callback.onSuccess(
                        payU3DS2ACSResponse
                    )
                }

                override fun onError(errorCode: String?, errorMessage: String?, p2: UIData?) {
                    val errorCodeInt = when (errorCode) {
                        WIBMO_ACS_ERROR_CODE_001 -> ACS_INVALID_OTP_SUBMIT
                        WIBMO_ACS_ERROR_CODE_002 -> ACS_TXN_CANCELLED
                        WIBMO_ACS_ERROR_CODE_003 -> ACTION_API_TIMEOUT
                        WIBMO_ACS_ERROR_CODE_004 -> ACS_OTP_SUBMISSION_FAILED
                        else -> {
                            ACS_ACTION_ERROR_CODE
                        }

                    }

                    LoggingUtils.logMessage(
                        activity!!,
                        THREE_DS_HEADLESS,
                        "$THREE_DS_ACTION_ERROR  ${acsActionType.name} $errorCode $errorMessage"
                    )
                    callback.onError(
                        errorCodeInt, if (!errorMessage.isNullOrEmpty()) errorMessage
                        else SOMETHING_WENT_WRONG_ERROR_MESSAGE
                    )
                }
            })
    }
}