package com.payu.threeDS2.network

import com.payu.paymentparamhelper.HashCommand
import com.payu.paymentparamhelper.PaymentParams
import com.payu.threeDS2.config.InternalConfig
import com.payu.threeDS2.constants.LoggingConstants.Companion.BIN_INFO_TAG
import com.payu.threeDS2.constants.LoggingConstants.Companion.BIN_INFO_VALIDATION_TAG
import com.payu.threeDS2.constants.LoggingConstants.Companion.LOGGING_ERROR_KEY
import com.payu.threeDS2.interfaces.listeners.PayU3DS2APIService
import com.payu.threeDS2.interfaces.listeners.PayUAsyncTaskResponse
import com.payu.threeDS2.utils.HashGenerationUtils
import com.payu.threeDS2.utils.LoggingUtils
import com.payu.threeDS2.utils.PostDataGeneratorUtils
import com.payu.threedsbase.constants.APIConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.BIN_INFO_API_CARD_BIN_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.HASH_NULL_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.HASH_NULL_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.MERCHANT_KEY_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.SDK_RESPONSE_STATUS_CODE_1
import com.payu.threedsbase.data.ErrorResponse
import com.payu.threedsbase.data.apiRequest.CardBinInfoRequest
import com.payu.threedsbase.data.apiResponse.BinInfoResponse
import com.payu.threedsbase.interfaces.listeners.PayU3DS2Callback
import com.payu.threedsbase.interfaces.listeners.PayUHashGeneratedListener
import org.json.JSONObject

class GetBinInfoAPITask(private val apiLayer: PayU3DSAPILayer) : PayU3DS2APIService {
    override fun callAPI(request: Any, callback: PayU3DS2Callback) {
        val params = PaymentParams()
        params.cardBin = (request as CardBinInfoRequest).cardDetails
        params.key = InternalConfig.key

        val error = validate(params)
        if (error.errorCode != null && error.errorMessage != null) {
            callback.onError(error.errorCode!!, error.errorMessage!!)
        } else {
            generateHash(params, callback, object : PayUHashGeneratedListener {
                override fun onHashGenerated(map: HashMap<String, String>) {
                    val hash = map[HashCommand.BinInfoHash.hashName]
                    if (hash.isNullOrEmpty()) {
                        callback.onError(HASH_NULL_ERROR_CODE, HASH_NULL_ERROR_MESSAGE)
                    } else {
                        val postData = generatePostData(params, hash)

                        val networkData = PayUNetworkData()
                        networkData.url =
                            if (apiLayer.config.isProduction) APIConstants.INFO_URL else
                                APIConstants.UAT_INFO_URL
                        networkData.postRequest = postData.toString()
                        PayU3DSAPICall().makePostAPICall(
                            APIConstants.HTTP_REQUEST_TYPE_POST,
                            networkData,
                            object : PayUAsyncTaskResponse {
                                override fun onSuccess(response: Any, executionTime: Long) {
                                    val jsonObject = JSONObject(response.toString())
                                    if (jsonObject.getInt(APIConstants.STATUS) == 1) {
                                        val parsedData = parseData(jsonObject)
                                        if (parsedData != null) {
                                            LoggingUtils.logMessage(
                                                apiLayer.activity!!,
                                                BIN_INFO_TAG,
                                                "",
                                                executionTime
                                            )
                                            callback.onSuccess(parsedData)
                                        } else {
                                            LoggingUtils.logMessage(
                                                apiLayer.activity!!,
                                                BIN_INFO_TAG,
                                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE,
                                                executionTime
                                            )
                                            callback.onError(
                                                SDK_RESPONSE_STATUS_CODE_1,
                                                PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE
                                            )
                                        }
                                    } else {
                                        LoggingUtils.logMessage(
                                            apiLayer.activity!!,
                                            BIN_INFO_TAG, LOGGING_ERROR_KEY +
                                                    if (!jsonObject.optString(APIConstants.BIN_INFO_ERROR_MESSAGE)
                                                            .isNullOrEmpty()
                                                    ) jsonObject.optString(
                                                        APIConstants.BIN_INFO_ERROR_MESSAGE
                                                    ) else PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE,
                                            executionTime
                                        )
                                        callback.onError(
                                            SDK_RESPONSE_STATUS_CODE_1,
                                            if (!jsonObject.optString(APIConstants.BIN_INFO_ERROR_MESSAGE)
                                                    .isNullOrEmpty()
                                            ) jsonObject.optString(
                                                APIConstants.BIN_INFO_ERROR_MESSAGE
                                            ) else PayU3DS2ErrorConstants.SOMETHING_WENT_WRONG_ERROR_MESSAGE
                                        )
                                    }
                                }

                                override fun onFailure(errorCode: Int, errorMessage: String, executionTime: Long) {
                                    LoggingUtils.logMessage(
                                        apiLayer.activity!!,
                                        BIN_INFO_TAG,
                                        LOGGING_ERROR_KEY + errorMessage,
                                        executionTime
                                    )
                                    callback.onError(errorCode, errorMessage)
                                }
                            }
                        )
                    }
                }
            })
        }
    }

    override fun validate(paymentParams: PaymentParams): ErrorResponse {
        val errorResponse = ErrorResponse()
        if (paymentParams.cardBin.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                BIN_INFO_TAG,
                BIN_INFO_VALIDATION_TAG + PayU3DS2ErrorConstants.CARD_BIN_NULL_ERROR_MESSAGE
            )

            errorResponse.errorCode =
                BIN_INFO_API_CARD_BIN_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.CARD_BIN_NULL_ERROR_MESSAGE
        } else if (paymentParams.key.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                BIN_INFO_TAG,
                BIN_INFO_VALIDATION_TAG + PayU3DS2ErrorConstants.MERCHANT_KEY_CANNOT_BE_NULL
            )
            errorResponse.errorCode =
                MERCHANT_KEY_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.MERCHANT_KEY_CANNOT_BE_NULL
        }
        return errorResponse
    }

    override fun generateHash(
        param: Any,
        callback: PayU3DS2Callback,
        payUHashGeneratedListener: PayUHashGeneratedListener
    ) {
        HashGenerationUtils.getHash(
            (param as PaymentParams).key,
            HashCommand.BinInfoHash,
            callback, payUHashGeneratedListener
        )
    }

    override fun generatePostData(request: PaymentParams, hash: String): Any? {
        return PostDataGeneratorUtils.generatePostData(
            HashCommand.BinInfoHash.hashName,
            hash,
            request
        )
    }

    private fun parseData(jsonObject: JSONObject): BinInfoResponse? {
        if (jsonObject.has(APIConstants.BIN_INFO_API_DATA)) {
            val binsData = jsonObject.getJSONObject(APIConstants.BIN_INFO_API_DATA)
                .getJSONObject(APIConstants.BIN_INFO_BINS_DATA)
            val binInfoResponse = BinInfoResponse()
            binInfoResponse.cardType =
                binsData.getString(APIConstants.BIN_INFO_CARD_TYPE)
            binInfoResponse.isDomestic =
                binsData.getString(APIConstants.BIN_INFO_IS_DOMESTIC)
            binInfoResponse.isAtmPinCard =
                binsData.optString(APIConstants.BIN_INFO_IS_ATMPIN_CARD)
            binInfoResponse.issuingBank =
                binsData.getString(APIConstants.BIN_INFO_ISSUING_BANK)
            binInfoResponse.bin = binsData.getString(APIConstants.BIN_INFO_BIN)
            binInfoResponse.category = binsData.getString(APIConstants.BIN_INFO_CATEGORY)
            binInfoResponse.isOtpOnTheFly =
                binsData.optInt(APIConstants.BIN_INFO_IS_OTP_ON_THE_FLY)
            binInfoResponse.isSiSupported =
                binsData.optInt(APIConstants.BIN_INFO_IS_SI_SUPPORTED)
            binInfoResponse.messageVersion =
                binsData.optString(APIConstants.BIN_INFO_MESSAGE_VERSION)

            return binInfoResponse
        }
        return null
    }
}
