package com.payu.threeDS2.network

import com.payu.paymentparamhelper.HashCommand
import com.payu.paymentparamhelper.PaymentParams
import com.payu.threeDS2.constants.LoggingConstants.Companion.AUTHORIZE_PAYMENT_TAG
import com.payu.threeDS2.constants.LoggingConstants.Companion.AUTHORIZE_PAYMENT_VALIDATION_TAG
import com.payu.threeDS2.constants.LoggingConstants.Companion.LOGGING_ERROR_KEY
import com.payu.threeDS2.interfaces.listeners.PayU3DS2APIService
import com.payu.threeDS2.interfaces.listeners.PayUAsyncTaskResponse
import com.payu.threeDS2.utils.HashGenerationUtils
import com.payu.threeDS2.utils.LoggingUtils
import com.payu.threeDS2.utils.PostDataGeneratorUtils
import com.payu.threedsbase.constants.APIConstants
import com.payu.threedsbase.constants.APIConstants.Companion.HTTP_REQUEST_TYPE_POST
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.HASH_NULL_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.HASH_NULL_ERROR_MESSAGE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.MERCHANT_KEY_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.TXN_AMOUNT_ERROR_CODE
import com.payu.threedsbase.constants.PayU3DS2ErrorConstants.Companion.TXN_ID_ERROR_CODE
import com.payu.threedsbase.data.ErrorResponse
import com.payu.threedsbase.interfaces.listeners.PayU3DS2Callback
import com.payu.threedsbase.interfaces.listeners.PayUHashGeneratedListener

class AuthorizeTransactionAPITask(private val apiLayer: PayU3DSAPILayer) : PayU3DS2APIService {
    override fun callAPI(request: Any, callback: PayU3DS2Callback) {
        request as PaymentParams
        val error = validate(request)

        if (error.errorCode != null) {
            callback.onError(error.errorCode!!, error.errorMessage!!)
        } else {
            generateHash(request, callback, object : PayUHashGeneratedListener {
                override fun onHashGenerated(map: HashMap<String, String>) {
                    val hash = map[HashCommand.AuthorizePaymentHash.hashName]
                    if (hash.isNullOrEmpty()) {
                        callback.onError(HASH_NULL_ERROR_CODE, HASH_NULL_ERROR_MESSAGE)
                    } else {
                        val postData = generatePostData(
                            request, hash
                        )
                        val networkData = PayUNetworkData()
                        networkData.url =
                            if (apiLayer.config.isProduction) APIConstants.AUTHORIZATION_PAYMENT_URL else
                                APIConstants.UAT_AUTHORIZATION_PAYMENT_URL
                        networkData.postRequest = postData.toString()
                        PayU3DSAPICall().makePostAPICall(
                            HTTP_REQUEST_TYPE_POST,
                            networkData,
                            object : PayUAsyncTaskResponse {
                                override fun onSuccess(response: Any, executionTime: Long) {
                                    LoggingUtils.logMessage(apiLayer.activity!!,
                                        AUTHORIZE_PAYMENT_TAG,
                                        "",
                                        executionTime
                                    )
                                    callback.onSuccess(response)
                                }

                                override fun onFailure(errorCode: Int, errorMessage: String, executionTime: Long) {
                                    LoggingUtils.logMessage(apiLayer.activity!!,
                                        AUTHORIZE_PAYMENT_TAG,
                                        LOGGING_ERROR_KEY + errorMessage,
                                        executionTime
                                    )
                                    callback.onError(errorCode, errorMessage)
                                }

                            }
                        )
                    }
                }
            })

        }
    }

    override fun validate(paymentParams: PaymentParams): ErrorResponse {
        val errorResponse = ErrorResponse()
        try {
            if (!paymentParams.amount.isNullOrEmpty()) paymentParams.amount.toDoubleOrNull()
        } catch (e: NumberFormatException) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                AUTHORIZE_PAYMENT_TAG,
                AUTHORIZE_PAYMENT_VALIDATION_TAG + PayU3DS2ErrorConstants.AMOUNT_ERROR_MESSAGE
            )
            errorResponse.errorCode =
                TXN_AMOUNT_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.AMOUNT_ERROR_MESSAGE
        } catch (e: NullPointerException) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                AUTHORIZE_PAYMENT_TAG,
                AUTHORIZE_PAYMENT_VALIDATION_TAG + PayU3DS2ErrorConstants.AMOUNT_ERROR_MESSAGE
            )
            errorResponse.errorCode =
                TXN_AMOUNT_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.AMOUNT_ERROR_MESSAGE
        }

        if (paymentParams.key.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                AUTHORIZE_PAYMENT_TAG,
                AUTHORIZE_PAYMENT_VALIDATION_TAG + PayU3DS2ErrorConstants.MERCHANT_KEY_CANNOT_BE_NULL
            )
            errorResponse.errorCode =
                MERCHANT_KEY_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.MERCHANT_KEY_CANNOT_BE_NULL
        } else if (paymentParams.txnId.isNullOrEmpty()) {
            LoggingUtils.logMessage(
                apiLayer.activity!!,
                AUTHORIZE_PAYMENT_TAG,
                AUTHORIZE_PAYMENT_VALIDATION_TAG + PayU3DS2ErrorConstants.TXN_ID_CANNOT_BE_NULL
            )
            errorResponse.errorCode =
                TXN_ID_ERROR_CODE
            errorResponse.errorMessage =
                PayU3DS2ErrorConstants.TXN_ID_CANNOT_BE_NULL
        }
        return errorResponse
    }

    override fun generateHash(
        param: Any,
        callback: PayU3DS2Callback,
        payUHashGeneratedListener: PayUHashGeneratedListener
    ) {
        return HashGenerationUtils.getHash(
            HashCommand.AuthorizePaymentHash,
            param as PaymentParams,
            callback,
            payUHashGeneratedListener
        )
    }

    override fun generatePostData(request: PaymentParams, hash: String, isSi: Boolean): Any? {
        return PostDataGeneratorUtils.generatePostData(
            HashCommand.AuthorizePaymentHash.hashName,
            hash,
            request
        )
    }
}