package com.payufin.mobilemsg.network

import android.content.Context
import android.util.Log
import com.orhanobut.logger.Logger
import com.payufin.mobilemsg.configurables.SdkConfig
import com.payufin.mobilemsg.helper.Callback
import com.payufin.mobilemsg.helper.CallbackV2
import com.payufin.mobilemsg.models.AuthRequestData
import com.payufin.mobilemsg.models.AuthResponseData
import com.payufin.mobilemsg.models.SmsUploadRequestData
import com.payufin.mobilemsg.sms.UploadSms
import com.payufin.mobilemsg.sms.checkpoint
import kotlinx.coroutines.delay
import okhttp3.ResponseBody
import retrofit2.Call
import kotlin.math.pow


class SmsDataSource ( private val apiService: SmsAPIService, private val config: SdkConfig, private val context: Context)  {


    private val maxRetries = config.retryCount
    private val initialBackoffTime = 5000 // Initial backoff time in milliseconds

    companion object {
        private val authRetryCounter = mutableMapOf<String, Int>()
    }



    private fun shouldRetry(responseCode: Int): Boolean {
        return responseCode == 404 || (responseCode in 500..599) || responseCode == 408
    }

    private fun shouldRetryWithRefreshToken(responseCode: Int): Boolean {
        return responseCode == 403
    }

    private suspend fun <T> executeApiCall(call: Call<T>, callback: CallbackV2<T>
                                            ,toCheckForRefreshTokenError: Int = 0
                                            ,retryCount: Int = 0 )  {
        try {
            Logger.d("readSMSSDK : Executing... call - %s", call, "Attempt - %s", retryCount)
            val response = call.execute()

            Logger.d("readSMSSDK : Executed... response %s", response.isSuccessful,"|", response.code())
            if (response.isSuccessful) {
                callback.onSuccess( response.body() !! )
            } else {
                if (shouldRetry(response.code()) && retryCount < maxRetries) {
                    delay(initialBackoffTime * 2.0.pow(retryCount.toDouble()).toLong())
                    Logger.d("readSMSSDK : Retrying... Attempt %s", retryCount)
                    return executeApiCall(call.clone(), callback, toCheckForRefreshTokenError, retryCount + 1)
                } else if(  shouldRetryWithRefreshToken( response.code() ) ) {
                    Log.w("readSMSSDK", "Auth token expired")
                    callback.onAuthFailure( response.errorBody()?.string(), response.code() )
                } else {
                    callback.onFailure( response.errorBody()?.string(), response.code() )
                }
            }
        }catch ( e : Exception ){
            e.printStackTrace()
            callback.onFailure( e.message )
        }
    }



    suspend fun getAuthToken(authRequestDataRequest: AuthRequestData, callback: Callback<AuthResponseData> )  {
        Logger.d("readSMSSDK : maxRetires - %s", maxRetries)
        val call: Call<AuthResponseData> = apiService.clientAuth(authRequestDataRequest)
        executeApiCall( call, object : CallbackV2<AuthResponseData>{
            override suspend fun onSuccess(t: AuthResponseData) {
                callback.onSuccess( t )
            }

            override suspend fun onFailure(message: String?, code: Int) {
                callback.onFailure( message, code )
            }

            override suspend fun onAuthFailure( message : String?, code: Int ) {
                callback.onFailure( message, code )
            }

        })
    }

    suspend fun uploadSms(smsUploadRequestData: SmsUploadRequestData, callback: Callback<ResponseBody>){



        Logger.d("readSMSSDK : maxRetires - %s", maxRetries)
        try{
        val call: Call<ResponseBody> = apiService.uploadSMS( smsUploadRequestData.smsBodyData.android_id, smsUploadRequestData.uploadTimeStamp,
                                                                smsUploadRequestData.checksum, smsUploadRequestData.authToken, smsUploadRequestData.smsBodyData )
        executeApiCall( call,  object : CallbackV2<ResponseBody>{
            override suspend fun onSuccess(t: ResponseBody) {
                callback.onSuccess( t )
            }

            override suspend fun onFailure(message: String?, code: Int) {
                callback.onFailure( message, code )
            }

            override suspend fun onAuthFailure( message : String?, code: Int ) {
                Log.w("readSMSSDK", "uploadSms. onAuthFailure")

                val smsBody = smsUploadRequestData.smsBodyData

                val apiMsgList = smsBody.messages

                val packetId = if(apiMsgList.isNotEmpty()){
                    "${apiMsgList.first().get("date")} - ${apiMsgList.last().get("date")}"}
                else{
                    "emptyList"
                }
                Log.w("readSMSSDK", "packetID : $packetId")

                val authRetries = authRetryCounter.getOrDefault(packetId, 0)
                Log.w("readSMSSDK", "retryCount : ${authRetries + 1}")


                if(authRetries < 1) {

                    authRetryCounter[packetId] = authRetries+1
                    // TODO: again populate this
                    val clientEmail = config.clientEmail
                    val accessId = config.accessId
                    val authDataRequest = AuthRequestData(clientEmail, accessId)

                    // Try to refresh the token
                    getAuthToken(authDataRequest, object : Callback<AuthResponseData> {
                        override suspend fun onSuccess(t: AuthResponseData) {
                            // TODO: save the refreshed token

                            // TODO: save refreshedAuthToken
                            val refreshedAuthToken = "Bearer " + t.access
                            Logger.d(
                                "readSMSSDK DataSource: New token received. Ensure we save & reuse it - %s", refreshedAuthToken
                            )
                            checkpoint.setAuthToken(context, refreshedAuthToken)

                            val newSmsUploadRequestData = UploadSms().getSmsUploadRequest(
                                smsUploadRequestData.smsBodyData.android_id,
                                smsUploadRequestData.smsBodyData.messages,
                                smsUploadRequestData.smsBodyData.sms_permission_available,
                                context
                            )
                            // TODO: create new smsUploadRequestData with refreshed token & old callback
                            uploadSms(newSmsUploadRequestData, callback)
                        }

                        override suspend fun onFailure(message: String?, code: Int) {
                            // No retry after refresh auth token fails.
                            callback.onFailure(message, code)
                        }


                    })
                }
                else
                    callback.onFailure( message, code )
            }

        }, 1  )}
        catch ( e: Exception ){
            e.printStackTrace()
            callback.onFailure( e.message )
        }
    }


}