package com.payufin.mobilemsg.sms

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.database.Cursor
import android.net.Uri
import android.provider.Telephony
import androidx.core.content.ContextCompat
import com.google.gson.JsonObject
import com.orhanobut.logger.Logger
import com.payufin.mobilemsg.configurables.SdkConfig
import com.payufin.mobilemsg.helper.Callback
import com.payufin.mobilemsg.helper.Utils
import com.payufin.mobilemsg.network.SmsRepository
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.withContext
import okhttp3.ResponseBody

/* Check the SMS permission if it is already granted or not :
      a. if permission is granted
        a.1 proceed with reading SMS from the device
        a.2 convert it into json packets of 50 (transformation)
      b. else do not proceed
*/
class ReadSms(private val repository: SmsRepository ) {

    private var smsFailureCount             = 0
    private var smsSuccessCount             = 0
    private var nonPersonalMessageFound     = false

        // Check SMS permission
        suspend fun checkSmsPermission( context: Context, authToken: String?,
                                        config: SdkConfig ) {
            // Check for SMS permission
            //AuthToken = authToken
            Logger.d("readSMSSDK : Checking SMS permission")

            if (ContextCompat.checkSelfPermission(
                    context,
                    Manifest.permission.READ_SMS
                ) != PackageManager.PERMISSION_GRANTED
            ) {
                // Permission is not granted
                // You can request the permission here or handle the lack of permission appropriately
                Logger.d("readSMSSDK : No SMS permission")

                val androidId = Utils.getAndroidId(context)

                val uploadRequest = UploadSms().getSmsUploadRequest( androidId, emptyList<JsonObject>(),
                    false, context )

                repository.uploadSms( uploadRequest, object : Callback<ResponseBody>{
                    override suspend fun onSuccess(t: ResponseBody) {
                        withContext(Dispatchers.Main){
                            config.sdkCallback.permissionNotAvailable(  "No SMS permission" , Utils.getAndroidId(context))
                        }

                    }

                    override suspend fun onFailure(message: String?, code: Int) {
                        withContext(Dispatchers.Main){
                            config.sdkCallback.permissionNotAvailable(  "No SMS permission", Utils.getAndroidId(context) )
                            config.sdkCallback.onUploadFailure(code, message?: "", Utils.getAndroidId(context))
                        }
                    }

                })

            } else {
                readSmsAndConvertToJson(context, authToken, config  )
            }
        }

        private fun getCheckpointsSharedPreferences(context: Context): Pair<Long, Long> {
            return checkpoint.getLastSyncedIndex(context)
        }

        private fun getDatabaseCursor(
            context: Context,
            lastSyncedLatestTimestamp: Long,
            lastSyncedEarliestTimestamp: Long,
            config: SdkConfig
        )
                : Cursor? {

            val uri = Uri.parse("content://sms/inbox")
            val sortOrder = "date DESC"


            // Calculate the timestamp for 190 days ago
            Logger.d("readSMSSDK : Upper Limit - %s", config.upperLimit)
            val daysInMillis = config.upperLimit * 24 * 60 * 60 * 1000L // Days to milliseconds conversion
            val timestamp190DaysAgo = System.currentTimeMillis() - daysInMillis

            var selectionArgs: Array<String>? = null
            var selection: String? = null

            if (lastSyncedLatestTimestamp.toInt() != -1 && lastSyncedEarliestTimestamp.toInt() != -1) {
                // Combine conditions to check within the synced timestamps and also limit to the last 190 days
                selection = "(date > ? AND date > ?) OR (date < ? AND date > ?)"
                selectionArgs = arrayOf(
                    lastSyncedLatestTimestamp.toString(),
                    timestamp190DaysAgo.toString(),
                    lastSyncedEarliestTimestamp.toString(),
                    timestamp190DaysAgo.toString()
                )
            } else {
                // Fallback to just checking the last 190 days if no valid timestamps are provided
                selection = "date > ?"
                selectionArgs = arrayOf(timestamp190DaysAgo.toString())
            }

            val cursor: Cursor? =
                context.contentResolver.query(uri, null, selection, selectionArgs, sortOrder)
            Logger.d(
                "readSMSSDK : URI - %s", uri, "Selection - %s", selection, "SelectionArgs - %s", selectionArgs?.joinToString(), "SortOrder - %s", sortOrder
            )

            return cursor

        }


        private suspend fun readSmsAndConvertToJson(context: Context, authToken: String?,
                                                    config: SdkConfig ) {


            // Getting the last synced latest and last synced earliest message index
            val (lastSyncedLatestTimestamp, lastSyncedEarliestTimestamp) = getCheckpointsSharedPreferences(context)
            Logger.d(
                "readSMSSDK : Last Synced Latest Message Checkpoint Epoch Timestamp - %s", lastSyncedLatestTimestamp
            )
            Logger.d(
                "readSMSSDK : Last Synced Earliest Message Checkpoint Epoch Timestamp - %s", lastSyncedEarliestTimestamp
            )

            // Getting the android id
            val androidId = Utils.getAndroidId(context)

            // creating database cursor
            val cursor = getDatabaseCursor(context, lastSyncedLatestTimestamp, lastSyncedEarliestTimestamp, config)

            cursor?.use {
                val indexBody       = it.getColumnIndex(Telephony.Sms.BODY)
                val indexAddress    = it.getColumnIndex(Telephony.Sms.ADDRESS)
                val indexDate       = it.getColumnIndex(Telephony.Sms.DATE)
                val indexId         = it.getColumnIndex(Telephony.Sms._ID)
                val pattern         = Regex(config.regexPattern)

                Logger.d("readSMSSDK : Initial iterator position: ${it.position}")

                /*
                    Message reading mechanism :
                    1. Check if there are any messages in the device
                        1.1 if there are messages then get the latest message id
                        1.2 iterate through the whole messages and check if there are any messages whose id greater than lastSyncedLatestIndex
                            and id less than lastSyncedEarliestIndex are there, this indicates that there are messages that are left to be synced
                        1.3 If the current sms id is less than earliestMessageId, set the earliestMessageId to current ID, this indicates
                            that earliest Message Id that is synced.
                        1.4 if there are no messages left to be synced, break the loop
                        1.5 set the checkpoints for lastSyncedLatestIndex and lastSyncedEarliestIndex
                    2. Send an empty list to the api response

                    */
                // Capture the ID of the latest message before entering the loop


                if (it.moveToFirst()) {
                    Logger.d("readSMSSDK : Current iterator position - %s", it.position)

                    uploadNextBatchOfSms( cursor,indexDate, indexAddress,
                                            indexId, indexBody, pattern,
                                            config.packetSize, androidId, authToken, config.sdkCallback, context )
                    Logger.d("readSMSSDK : End of all batches.")
                    withContext(Dispatchers.Main){
                        config.sdkCallback.smsSyncSuccess("All batches Done !!! Successful messages : $smsSuccessCount and Failure messages : $smsFailureCount",
                                                                        smsSuccessCount, smsFailureCount, Utils.getAndroidId(context) )
                    }
                }
                if(!(it.moveToFirst()) || !(nonPersonalMessageFound)) {
                    // Case when there are no messages in the device
                    val emptyUploadRequest = UploadSms().getSmsUploadRequest(
                                                        androidId,
                                                        emptyList(),
                                                        true,
                                                        context )

                    repository.uploadSms( emptyUploadRequest, object : Callback<ResponseBody> {
                        override suspend fun onSuccess(t: ResponseBody) {
                            withContext(Dispatchers.Main){
                                config.sdkCallback.noMessagesToSync("No messages to sync", Utils.getAndroidId(context))
                            }

                        }

                        override suspend fun onFailure(message: String?, code: Int) {
                            withContext(Dispatchers.Main){
                                config.sdkCallback.noMessagesToSync("No messages to sync", Utils.getAndroidId(context))
                                config.sdkCallback.onUploadFailure( code, message ?: "" , Utils.getAndroidId(context))
                            }

                        }
                    } )

                    Logger.d("readSMSSDK : No messages to be synced")
                }
            }
        }



        private suspend fun uploadNextBatchOfSms(
            cursor: Cursor,
            indexDate: Int,
            indexAddress: Int,
            indexId : Int,
            indexBody : Int,
            pattern : Regex,
            packetSize: Int,
            androidId: String,
            authToken: String?,
            sdkCallback: SdkCallback,
            context: Context
        ){
            cursor.let {

                // Creating empty sms list
                val smsList = mutableListOf<JsonObject>()
                if( !(it.isAfterLast) ) {

                    do {
                        // Getting the date and address for fetching non-personal messages in the given timestamp between two checkpoints
                        val date = it.getLong(indexDate)
                        val address = it.getString(indexAddress)

                        // checking the regex pattern to fetch non-personal messages
                        if (pattern.containsMatchIn(address)) {
                            nonPersonalMessageFound = true

                            // creating message body
                            val smsObj = JsonObject()
                            // adding the sms properties
                            smsObj.addProperty("sms_id", it.getLong(indexId))
                            smsObj.addProperty("address", address)
                            smsObj.addProperty("body", it.getString(indexBody))
                            smsObj.addProperty("date", date)

                            smsList.add(smsObj)
                            Logger.d("readSMSSDK : sms content %s", smsObj)
                            if (smsList.size == packetSize) {

                                // TODO: check the condition
                                it.moveToNext()

                                val uploadRequest = UploadSms().getSmsUploadRequest(
                                    androidId,
                                    smsList,
                                    true,
                                    context
                                )

                                repository.uploadSms(
                                    uploadRequest,
                                    object : Callback<ResponseBody> {
                                        override suspend fun onSuccess(t: ResponseBody) {

                                            smsSuccessCount += smsList.size
                                            // TODO: remove after testing
                                            delay(1000)
                                            Logger.d("readSMSSDK : Upload Next batch")
                                            smsList.clear()
                                            uploadNextBatchOfSms(
                                                cursor, indexDate, indexAddress,
                                                indexId, indexBody, pattern,
                                                packetSize, androidId, authToken, sdkCallback, context
                                            )

                                        }

                                        override suspend fun onFailure(
                                            message: String?,
                                            code: Int
                                        ) {
                                            smsFailureCount += smsList.size
                                            withContext(Dispatchers.Main) {
                                                sdkCallback.onUploadFailure(code, message ?: "", Utils.getAndroidId(context))
                                            }
                                            smsList.clear()
                                            uploadNextBatchOfSms(
                                                cursor, indexDate, indexAddress,
                                                indexId, indexBody, pattern,
                                                packetSize, androidId, authToken, sdkCallback, context
                                            )
                                        }

                                    })

                                break
                            } else if (it.isLast) {
                                uploadLastBatch(androidId, smsList, sdkCallback, context)
                            }
                        }

                        Logger.d("readSMSSDK : Loop - %s", smsList, "| size - %s", smsList.size)
                    } while (it.moveToNext())
                    Logger.d("readSMSSDKSDK - %s", smsList.size)
                    if (smsList.isNotEmpty()) {
                        uploadLastBatch(androidId, smsList, sdkCallback, context)

                    }
                }

            }
        }


        private suspend fun uploadLastBatch(
            androidId: String,
            smsList: MutableList<JsonObject>,
            sdkCallback: SdkCallback,
            context: Context
        ){
            val lastBatchOfUploadRequest = UploadSms().getSmsUploadRequest(
                androidId,
                smsList,
                true,
                context
            )
            repository.uploadSms(
                lastBatchOfUploadRequest,
                object : Callback<ResponseBody> {
                    override suspend fun onSuccess(t: ResponseBody) {
                        smsSuccessCount += smsList.size
                        smsList.clear()

                    }

                    override suspend fun onFailure(
                        message: String?,
                        code: Int
                    ) {
                        smsFailureCount += smsList.size
                        smsList.clear()
                        withContext(Dispatchers.Main) {
                            sdkCallback.onUploadFailure(code, message ?: "", Utils.getAndroidId(context))

                        }
                    }

                })
        }

}
