package `in`.trainman.book.trainmanbookingsdk

import `in`.trainman.book.trainmanbookingsdk.callback.TrainmanBookingListener
import android.app.Activity
import android.content.Intent
import okhttp3.*
import org.json.JSONObject
import java.io.IOException
import java.lang.ref.WeakReference


var TOKEN_RESPONSE = "TOKEN_RESPONSE"

class TrainmanBookingManager {

    private var canStartBookingFlow: Boolean = false
    private var isAuthInProgress: Boolean = false
    private var username: String = ""
    private var password: String = ""
    private var client_token: String = ""

    private lateinit var ctx: WeakReference<Activity>

    var authCall: Call? = null

    fun init(
        username: String,
        password: String,
        client_token: String,
        app_type: APP_TYPE
    ): TrainmanBookingManager {
        this.username = username
        this.password = password
        this.client_token = client_token
        Utils.appType = app_type
        isAuthInProgress = false
        Utils.initBaseURLs()
        ConfigManager.fetchData {
            canStartBookingFlow = it
        }
        return this
    }

    fun startBookingFlow(activity: Activity, requestCode: Int, listener: TrainmanBookingListener?) {
        ctx = WeakReference(activity)
        if (!this::ctx.isInitialized) {
            throw RuntimeException(Utils.INITIALIZATION_ERROR_CTX)
        }
        if (username.isEmpty()) {
            throw RuntimeException(Utils.INITIALIZATION_ERROR)
        }
        if (password.isEmpty()) {
            throw RuntimeException(Utils.INITIALIZATION_ERROR)
        }
        if (client_token.isEmpty()) {
            throw RuntimeException(Utils.INITIALIZATION_ERROR)
        }
        if(!canStartBookingFlow){
            listener?.onFailure(Utils.ERR_GENERIC)
            return
        }
        if (!isAuthInProgress) {
            validateClient(
                username,
                password,
                client_token,
                "password",
                requestCode,
                listener
            )
        }

    }

    private fun validateClient(
        username: String, password: String,
        client_token: String, grant_type: String,
        requestCode: Int, listener: TrainmanBookingListener?
    ) {
        val client = Network.client

        val formBody = getFormBody(username, password, client_token, grant_type)
        println(formBody)

        val request = buildRequest(formBody)

        println("Data upload request : $request")
        authCall?.cancel()
        authCall = client.newCall(request)
        isAuthInProgress = true
        authCall?.enqueue(object : Callback {
            override fun onResponse(call: Call, response: Response) {
                if (response.isSuccessful) {
                    response.body()?.string()?.let {
                        try {
                            val json = JSONObject(it)
                            val tokenResponse = getTokenResponse(json)
                            tokenResponse?.let {
                                onSuccessCallback(tokenResponse, requestCode, listener)
                            }?: run {
                                onFailureCallback(json, listener)
                            }
                        } catch (e: Exception) {
                            onFailureCallback(null, listener)
                        }
                    } ?: run {
                        onFailureCallback(null, listener)
                    }
                } else {
                    onFailureCallback(null, listener)
                }
            }

            override fun onFailure(call: Call, e: IOException) {
                onFailureCallback(null, listener)
            }

        })

    }

    private fun getTokenResponse(json: JSONObject?): String? {
        try {
            if (json?.has("success") == true) {
                val success = json.getBoolean("success")
                if (success) {
                    if (json.has("data")) {
                        val data = json.getJSONObject("data")
                        if(data.has("token_response")) {
                            val token_response = data.getString("token_response")
                            return token_response
                        }
                    }
                }
            }
        } catch (e : java.lang.Exception){}
        return null
    }

    private fun buildRequest(formBody: RequestBody): Request {
        return Request.Builder()
            .addHeader("Content-Type", formBody.contentType().toString())
            .post(formBody)
            .url(Utils.APP_BASE_URL + Utils.AUTH_API_END_POINT)
            .build()
    }

    private fun getFormBody(
        username: String,
        password: String,
        clientToken: String,
        grantType: String
    ): RequestBody {
        return MultipartBody.Builder()
            .setType(MultipartBody.FORM)
            .addFormDataPart("client_id", username)
            .addFormDataPart("client_secret", password)
            .addFormDataPart("client_token", clientToken)
            .addFormDataPart("grant_type", grantType)
            .build()
    }

    private fun onSuccessCallback(
        tokenResponse: String,
        requestCode: Int,
        listener: TrainmanBookingListener?
    ) {
        isAuthInProgress = false
        listener?.onSuccess()
        startWebFlow(tokenResponse, requestCode)
    }

    private fun onFailureCallback(json: JSONObject?, listener: TrainmanBookingListener?) {
        isAuthInProgress = false
        json?.let {
            if (json.has("error")) {
                val error = json.getString("error")
                listener?.onFailure(error)
            } else {
                listener?.onFailure(Utils.ERR_GENERIC)
            }
        } ?: run {
            listener?.onFailure(Utils.ERR_GENERIC)
        }

    }

    private fun startWebFlow(response: String, requestCode: Int) {
        ctx.get()?.startActivityForResult(Intent(ctx.get(), WebActivity::class.java).apply {
            putExtra(
                TOKEN_RESPONSE, response
            )
        }, requestCode)
    }

}