package umun.entity.controller;

import java.util.HashMap;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import umun.core.constants.RestConstants;
import umun.core.constants.ValidationException;
import umun.entity.model.EntityPref;
import umun.entity.service.EntityPrefService;
import umun.iam.model.User;
import umun.iam.service.TokenService;

@RestController
@CrossOrigin("*")
@RequestMapping(RestConstants.V1 + "/pref")
public class EntityPrefController {

	@Autowired
	private TokenService tokenService;

	private static Map<String, EntityPrefService<?, ?, ?>> serviceMap = new HashMap<>();

	public static void addServiceReference(String entityName, EntityPrefService<?, ?, ?> service) {
		serviceMap.put(entityName, service);
	}

	@GetMapping("/{entityName}/{entityId}")
	public ResponseEntity<?> read(@RequestHeader(RestConstants.HEADER_AUTHORIZATION) String authHeader,
			@PathVariable("entityName") String entityName, @PathVariable("entityId") long entityId) {
		try {
			return ResponseEntity
					.ok(getServiceOrFail(entityName).read(entityId, tokenService.getToken(authHeader).getUser()));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}
	
	@GetMapping("/{entityName}/{entityId}/object")
	public ResponseEntity<?> readEntity(@RequestHeader(RestConstants.HEADER_AUTHORIZATION) String authHeader,
			@PathVariable("entityName") String entityName, @PathVariable("entityId") long entityId) {
		try {
			return ResponseEntity
					.ok(getServiceOrFail(entityName).findOne(entityId, tokenService.getToken(authHeader).getUser()));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}
	
	@GetMapping("/template/{entityName}")
	public ResponseEntity<?> template(@PathVariable("entityName") String entityName,
			@RequestParam Map<String,String> allRequestParams) {
		try {
			return ResponseEntity
					.ok(getServiceOrFail(entityName).getCreationTemplate(allRequestParams));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}
	
	@GetMapping("/{entityName}/{entityId}/category/")
	public ResponseEntity<?> readAllPrefs(@RequestHeader(name = RestConstants.HEADER_AUTHORIZATION, required =  false) String authHeader,
			@PathVariable("entityName") String entityName, 
			@PathVariable("entityId") long entityId) {
		try {
			User requestingUser = null;
			
			if(authHeader != null && !authHeader.equals("")) {
				requestingUser =tokenService.getToken(authHeader).getUser();
			}
			
			return ResponseEntity
					.ok(getServiceOrFail(entityName).readAllPrefs(requestingUser, entityId));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	@GetMapping("/{entityName}/{entityId}/category/{categoryId}")
	public ResponseEntity<?> read(@RequestHeader(RestConstants.HEADER_AUTHORIZATION) String authHeader,
			@PathVariable("entityName") String entityName, 
			@PathVariable("entityId") long entityId,
			@PathVariable("categoryId") long categoryId) {
		try {
			return ResponseEntity
					.ok(getServiceOrFail(entityName).read(tokenService.getToken(authHeader).getUser(), categoryId, entityId));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	@PutMapping("/template/{entityName}/{key}")
	public ResponseEntity<?> validate(@PathVariable("entityName") String entityName,
			 @PathVariable("key") String key, 
			@RequestBody(required = true) EntityPref value) {
		try {
			return ResponseEntity.ok(getServiceOrFail(entityName).validate(key, value));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	@PutMapping("/{entityName}/{entityId}/{categoryId}/{key}")
	public ResponseEntity<?> update(@RequestHeader(RestConstants.HEADER_AUTHORIZATION) String authHeader,
			@PathVariable("entityName") String entityName, @PathVariable("entityId") long entityId,
			@PathVariable("categoryId") long categoryId, @PathVariable("key") String key, 
			@RequestBody(required = false) String value) {
		try {
			if(value == null ) {
				value = "";
			}
			return ResponseEntity.ok(getServiceOrFail(entityName).update(entityId, categoryId, key, value,
					tokenService.getToken(authHeader).getUser()));
		} catch (ValidationException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	public static EntityPrefService<?, ?, ?> getServiceOrFail(String entityName) throws ValidationException {
		if (!serviceMap.containsKey(entityName)) {
			throw new ValidationException("Entity name not found", HttpStatus.NOT_FOUND);
		}
		return serviceMap.get(entityName);
	}
}
