package umun.entity.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.http.HttpStatus;

import umun.core.constants.ValidationException;
import umun.core.service.SystemPrefService;
import umun.entity.model.CategoryPref;
import umun.entity.model.EntityPref;
import umun.entity.model.EntityPrefCategory;
import umun.entity.model.EntityPrefId;
import umun.entity.model.PrefDataType;
import umun.iam.model.Meta;

/*
 * MAPPED_ENTITY refers to entity's relation with another entity. (Like. User ->
 * Contract/Script) The 2nd entity will be referred ad MAPPED_ENTITY
 * 
 * @author abhinav
 *
 * @param <ENTITY>
 * @param <PREF>
 */
public abstract class EntityPrefCategoryInterface<ENTITY extends Meta, PREF extends EntityPref, ID extends EntityPrefId, MAPPED_ENTITY> {

	public abstract String getRemoteHideKey();

	protected abstract Boolean getRemoteHideDefault();

	public abstract PrefDataType getDataType();

	public abstract Map<String, String> getOptions();
	
	public abstract EntityPrefValidationInterface getCustomValidation();
	
	public abstract String getGroup();

	protected abstract String getCategoryName();

	protected abstract String getCategoryDisplayName();

	protected abstract String getEntityName();

	protected abstract PREF createNewPref(CategoryPref<ENTITY> pref, EntityPrefCategory category, ENTITY entity);

	protected abstract List<MAPPED_ENTITY> getAllEntitiesToMap(ENTITY entity);

	protected abstract String getKey(MAPPED_ENTITY mappedEntity);

	protected abstract String getDisplayName(MAPPED_ENTITY mappedEntity);

	protected abstract String getNotFoundDefaultValue(ENTITY entity, String key, MAPPED_ENTITY mapped_ENTITY);

	@Autowired
	private EntityPrefCategoryService categoryService;

	@Autowired
	protected EntityPrefService<ENTITY, PREF, ID> entityPrefService;

	@Autowired
	private SystemPrefService systemPrefService;

	protected EntityPrefCategory category;

	@EventListener
	public void startUp(ContextRefreshedEvent event) {
		category = categoryService.readWithoutFail(getCategoryName(), getEntityName(), getRemoteHideKey(),
				getCategoryDisplayName());
		entityPrefService.addServiceReference(getCategoryName(), this);

		systemPrefService.getValue(getRemoteHideKey(), Boolean.toString(getRemoteHideDefault()));
	}

	public List<PREF> read(ENTITY entity) {
		EntityPrefCategory category = categoryService.readWithoutFail(getCategoryName(), getEntityName(),
				getRemoteHideKey(), getCategoryDisplayName());
		List<PREF> prefs = new ArrayList<>();
		for (CategoryPref<ENTITY> pref : getCategoryPrefs(entity)) {
			prefs.add(createNewPref(pref, category, entity));
		}
		return prefs;
	}

	public PREF readWithoutFail(ENTITY entity, MAPPED_ENTITY mappedEntity) throws ValidationException {
		return entityPrefService.readWithoutFail(entity, category, getKey(mappedEntity),
				getNotFoundDefaultValue(entity, getKey(mappedEntity), mappedEntity));
	}

	public Object getMultiInputValue(ENTITY entity, MAPPED_ENTITY mapped_ENTITY, String optionKey,
			Object defaultValue) {
		PREF pref;
		try {
			pref = entityPrefService.readWithoutFail(entity, category, getKey(mapped_ENTITY),
					getNotFoundDefaultValue(entity, getKey(mapped_ENTITY), mapped_ENTITY));

			if (!getOptions().containsKey(optionKey)) {
				throw new ValidationException(String.format("Option key %s not found", optionKey),
						HttpStatus.NOT_FOUND);
			}
			switch (getDataType()) {
			case MultiInput_Boolean:
				return new JSONObject(pref.getValue()).optBoolean(optionKey);
			case MultiInput_Double:
				return new JSONObject(pref.getValue()).optDouble(optionKey);
			case MultiInput_Long:
				return new JSONObject(pref.getValue()).optLong(optionKey);
			case MultiInput_String:
				return new JSONObject(pref.getValue()).optString(optionKey);

			default:
				return defaultValue;
			}
		} catch (ValidationException | JSONException e) {
			e.printStackTrace();
			return defaultValue;
		}
	}

	private List<CategoryPref<ENTITY>> getCategoryPrefs(ENTITY entity) {
		List<CategoryPref<ENTITY>> prefs = findByEntity(entity);

		List<CategoryPref<ENTITY>> allPrefs = new ArrayList<CategoryPref<ENTITY>>();

		CategoryPref<ENTITY> tempPref;
		for (MAPPED_ENTITY mappedEntity : getAllEntitiesToMap(entity)) {
			tempPref = new CategoryPref<ENTITY>(entity, getKey(mappedEntity),
					getOverride(prefs, getKey(mappedEntity), entity, mappedEntity), getOptions(), getDataType(),
					getDisplayName(mappedEntity));

			allPrefs.add(tempPref);
		}
		return allPrefs;
	}

	private String getOverride(List<CategoryPref<ENTITY>> prefs, String key, ENTITY entity,
			MAPPED_ENTITY mapped_ENTITY) {
		for (CategoryPref<ENTITY> categoryPref : prefs) {
			if (key.equals(categoryPref.getKey())) {
				return categoryPref.getValue();
			}
		}
		// TODO: this mehtod should pass more information to the sub class
		return getNotFoundDefaultValue(entity, key, mapped_ENTITY);
	}

	private List<CategoryPref<ENTITY>> findByEntity(ENTITY entity) {
		List<PREF> prefs = entityPrefService.read(entity, categoryService.readWithoutFail(getCategoryName(),
				getEntityName(), getRemoteHideKey(), getCategoryDisplayName()));
		List<CategoryPref<ENTITY>> categoryPrefs = new ArrayList<>();
		for (PREF pref : prefs) {
			categoryPrefs.add(new CategoryPref<ENTITY>(entity, pref.getKey(), pref.getValue(), getOptions(),
					getDataType(), pref.getDisplayName()));
		}
		return categoryPrefs;
	}

}
