package umun.entity.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.http.HttpStatus;

import umun.core.constants.ValidationException;
import umun.core.service.SystemPrefService;
import umun.entity.controller.EntityPrefController;
import umun.entity.model.EntityPref;
import umun.entity.model.EntityPrefCategory;
import umun.entity.model.EntityPrefId;
import umun.entity.model.EntityPrefProp;
import umun.entity.model.EntityPrefValue;
import umun.entity.model.PrefDataType;
import umun.entity.model.impl.TemplatePref;
import umun.entity.repository.EntityPrefRepository;
import umun.entity.templates.service.EntityCRUDService;
import umun.entity.util.EntityPrefValidationUtil;
import umun.iam.model.Meta;
import umun.iam.model.User;
import umun.log.service.SystemLogService;

public abstract class EntityPrefService<ENTITY extends Meta, PREF extends EntityPref, ID extends EntityPrefId> {

	protected abstract EntityPrefRepository<PREF, ID> getRepository();

	protected abstract Map<String, EntityPrefCategoryInterface<ENTITY, PREF, ID, ?>> getCategoryServicesMap();

	protected abstract Map<String, EntityPrefValue<ENTITY, PREF>> getRootKeys();

	protected abstract List<PREF> findByEntityAndCategoryOrderByKey(ENTITY entity, EntityPrefCategory rootCategory);

	protected abstract PREF createMissingRootPref(ENTITY entity, EntityPrefCategory rootCategory, String key,
			EntityPrefValue<ENTITY, PREF> value, Map<String, String> options, String group);

	protected abstract PREF createCategoryAsPref(ENTITY entity, EntityPrefCategory category) throws ValidationException;

	protected abstract ID createEntityPrefId(ENTITY entity, EntityPrefCategory category, String key);

	protected abstract PREF createPref(ENTITY entity, EntityPrefCategory category, String key, String value,
			String remoteHideKey) throws ValidationException;

	protected abstract ENTITY findEntityOrFail(long entityId, User requestedBy) throws ValidationException;

	protected abstract String getEntityName();

	protected abstract EntityCRUDService<ENTITY> getCrudService();

	public EntityPrefCategory rootCategory;

	@Autowired
	protected EntityPrefCategoryService categoryService;

	@Autowired
	protected SystemPrefService systemPrefService;

	@Autowired
	private SystemLogService systemLogService;

	@EventListener
	public void startUp(ContextRefreshedEvent event) {
		rootCategory = categoryService.findRootCategory(getEntityName());
		EntityPrefController.addServiceReference(getEntityName(), this);
		
		systemPrefService.setValue("Entity Version", "101");
	}

	public void addRootKeys(String rootKey, String defaultValue, String remoteHideKey, PrefDataType dataType,
			EntityPrefOptionsInterface optionsInterface, String displayName,
			EntityPrefValidationInterface<ENTITY, PREF> entityPrefValidationInterface, String group,
			boolean defaultRemoteHide, EntityPrefProp<ENTITY, PREF> properties) {

		systemPrefService.getValue(remoteHideKey, String.valueOf(defaultRemoteHide));
		getRootKeys().put(rootKey, new EntityPrefValue<ENTITY, PREF>(defaultValue, remoteHideKey, dataType,
				optionsInterface, displayName, entityPrefValidationInterface, group, properties));
	}

	public void addServiceReference(String categoryName, EntityPrefCategoryInterface<ENTITY, PREF, ID, ?> instance) {
		getCategoryServicesMap().put(categoryName, instance);
	}

	public PREF createOrUpdate(ENTITY entity, EntityPrefCategory category, String key, String value, User requestedBy)
			throws ValidationException {
		PREF pref = readWithoutFail(entity, category, key, value);
		String oldValue = pref.getValue();
		validateCreateOrUpdate(entity, category, key, value, requestedBy, pref);
		pref = doPostCreationOperations(entity, category, pref, key, value, requestedBy);
		saveAndFlush(entity, category, pref, key, value, requestedBy, oldValue);
		return pref;
	}

	protected PREF saveAndFlush(ENTITY entity, EntityPrefCategory category, PREF pref, String key, String value,
			User requestedBy, String oldValue) {
		pref = getRepository().saveAndFlush(pref);

		systemLogService.create(requestedBy,
				"Value for %s (%d %s) Changed in category [%s] & key [%s]. Old Value: %s. New Value: %s",
				category.getEntityName(), entity.getId(), entity.getName(), category.getDisplayName(), key, oldValue,
				pref.getValue());
		return pref;
	}

	protected void validateCreateOrUpdate(ENTITY entity, EntityPrefCategory category, String key, String value,
			User requestedBy, PREF pref) throws ValidationException {
		// override this method in subclasses
	}

	protected PREF doPostCreationOperations(ENTITY entity, EntityPrefCategory category, PREF pref, String key,
			String value, User requestedBy) throws ValidationException {
		pref.setValue(value);
		pref.setOptions(getOptions(category, key, entity));

		EntityPrefValidationUtil.validate(pref);
		doCustomValidation(entity, pref);

		mapPrefToField(entity, pref, requestedBy);

		return pref;
	}

	public List<PREF> read(long entityId, User requestedBy) throws ValidationException {
		return read(findEntityOrFail(entityId, requestedBy));
	}

	public List<PREF> read(User requestedBy, long categoryId, long entityId) throws ValidationException {
		EntityPrefCategory category = categoryService.readOrFail(categoryId);
		ENTITY entity = findEntityOrFail(entityId, requestedBy);
		if (getCategoryServicesMap().containsKey(category.getName())
				&& getCategoryServicesMap().get(category.getName()) != null) {
			try {
				return getCategoryServicesMap().get(category.getName()).read(entity);
			} catch (Exception e) {
				// do nothing here
			}
		}
		return read(entity, category);
	}

	public List<PREF> read(ENTITY entity, EntityPrefCategory category) {
		return findByEntityAndCategoryOrderByKey(entity, category);
	}

	public Map<String, List<PREF>> readAllPrefs(User requestedBy, long entityId) throws ValidationException {
		ENTITY entity = findEntityOrFail(entityId, requestedBy);
		Map<String, List<PREF>> allPrefs = new HashMap<>();
		List<PREF> prefs;
		for (EntityPrefCategory category : categoryService.read(false, getEntityName())) {
			if (category.getName().equals("root")) {
				prefs = findByEntityAndCategoryOrderByKey(entity, category);
				prefs.addAll(getMissingRootPrefs(prefs, entity, category));
			} else {
				prefs = getCategoryServicesMap().get(category.getName()).read(entity);
			}
			allPrefs.put(category.getName(), prefs);
		}

		return allPrefs;
	}

	public List<PREF> read(ENTITY entity) {

		List<PREF> prefs = findByEntityAndCategoryOrderByKey(entity, rootCategory);

		prefs.addAll(getMissingRootPrefs(prefs, entity, rootCategory));
		prefs.addAll(getCategoriesAsPrefs(entity));

		prefs.sort((o1, o2) -> {
			if (o1.getDisplayName() != null && o2.getDisplayName() != null) {
				return o1.getDisplayName().compareTo(o2.getDisplayName());
			} else if (o1.getCategory() != null && o1.getCategory().getDisplayName() != null && o2.getCategory() != null
					&& o2.getCategory().getDisplayName() != null) {
				return o2.getCategory().getDisplayName().compareTo(o1.getCategory().getDisplayName());
			} else {
				return 0;
			}
		});

		prefs = removeHidden(prefs);
		return prefs;
	}

	public PREF readWithoutFail(ENTITY entity, EntityPrefCategory category, String key, String value)
			throws ValidationException {
		PREF pref = read(createEntityPrefId(entity, category, key));
		if (pref == null) {
			pref = createPref(entity, category, key, value, getRemoteHideKey(category, key));
		}
		return pref;
	}

	public PREF readOrFail(ENTITY entity, EntityPrefCategory category, String key) throws ValidationException {
		PREF pref = read(createEntityPrefId(entity, category, key));
		if (pref == null) {
			throw new ValidationException(String.format("Key %s not found", key), HttpStatus.NOT_FOUND);
		}
		return pref;
	}

	public List<TemplatePref> getCreationTemplate(Map<String, String> allRequestParams) {
		List<TemplatePref> prefs = new ArrayList<>();
		this.getRootKeys().forEach((key, value) -> {
			if (systemPrefService.getValue(value.remoteHideKey, "false").equals("true")
					&& value.dataType != PrefDataType.Table) {
				TemplatePref templatePref = new TemplatePref(this.getEntityName(), key, rootCategory.getId(),
						rootCategory, value.value, value.dataType, value.remoteHideKey, value.displayName);
				templatePref.setGroup(value.group);
				templatePref.setOptions(value.getOptions(null, allRequestParams));

				if (value.getAdditionaProperties() != null) {
					templatePref.setRequired(value.getAdditionaProperties().isRequiredInCreateTemplate());
					templatePref.setNumberOfOptionsAsList(value.getAdditionaProperties().getNumberOfOptionsAsList());
				}

				if (value.getAdditionaProperties() == null
						|| !value.getAdditionaProperties().isHiddenInCreateTemplate()) {
					prefs.add(templatePref);
				}
				

			}
		});
		return prefs;
	}

	public String validate(String key, EntityPref value) throws ValidationException {
		if (!this.getRootKeys().containsKey(key)) {
			throw new ValidationException("Key not found.", HttpStatus.NOT_FOUND);
		}
		EntityPrefValue<?, PREF> entityPrefValue = this.getRootKeys().get(key);
		if (entityPrefValue.entityPrefValidationInterface != null) {
			entityPrefValue.entityPrefValidationInterface.validate(null, value);
		}

		return "Valid";
	}

	public ENTITY findOne(long entityId, User requestedBy) throws ValidationException {
		return findEntityOrFail(entityId, requestedBy);
	}

	protected PREF read(ID entityPrefId) {
		return getRepository().findOne(entityPrefId);
	}

	public PREF update(long entityId, long categoryId, String key, String value, User requestedBy)
			throws ValidationException {

		return createOrUpdate(findEntityOrFail(entityId, requestedBy), categoryService.readOrFail(categoryId), key,
				value, requestedBy);
	}

	protected String getRemoteHideKey(EntityPrefCategory category, String key) throws ValidationException {
		if (category.getName().equalsIgnoreCase("root")) {
			if (!getRootKeys().containsKey(key)) {
				throw new ValidationException("Remote hide key not found in root " + key, HttpStatus.NOT_FOUND);
			}
			return getRootKeys().get(key).remoteHideKey;
		}

		if (!getCategoryServicesMap().containsKey(category.getName())) {
			throw new ValidationException("Remote hide key not found in category " + key, HttpStatus.NOT_FOUND);
		}
		return getCategoryServicesMap().get(category.getName()).getRemoteHideKey();
	}

	protected PrefDataType getDataType(EntityPrefCategory category, String key) throws ValidationException {
		if (category.getName().equalsIgnoreCase("root")) {
			if (!getRootKeys().containsKey(key)) {
				throw new ValidationException("Data type not found in root " + key, HttpStatus.NOT_FOUND);
			}
			return getRootKeys().get(key).dataType;
		}

		if (!getCategoryServicesMap().containsKey(category.getName())) {
			throw new ValidationException("Data type key not found in category " + key, HttpStatus.NOT_FOUND);
		}
		return getCategoryServicesMap().get(category.getName()).getDataType();
	}

	protected Map<String, String> getOptions(EntityPrefCategory category, String key, ENTITY entity)
			throws ValidationException {
		if (category.getName().equalsIgnoreCase("root")) {
			if (!getRootKeys().containsKey(key)) {
				throw new ValidationException("Data type not found in root " + key, HttpStatus.NOT_FOUND);
			}
			return getRootKeys().get(key).getOptions(entity);
		}

		if (!getCategoryServicesMap().containsKey(category.getName())) {
			throw new ValidationException("Data type key not found in category " + key, HttpStatus.NOT_FOUND);
		}
		return getCategoryServicesMap().get(category.getName()).getOptions();
	}

	protected String getGroup(EntityPrefCategory category, String key) throws ValidationException {
		if (category.getName().equalsIgnoreCase("root")) {
			if (!getRootKeys().containsKey(key)) {
				throw new ValidationException("Group not found in root " + key, HttpStatus.NOT_FOUND);
			}
			return getRootKeys().get(key).group;
		}

		if (!getCategoryServicesMap().containsKey(category.getName())) {
			throw new ValidationException("Group not found in category " + key, HttpStatus.NOT_FOUND);
		}
		return getCategoryServicesMap().get(category.getName()).getGroup();
	}

	protected String getDisplayname(String key) {
		if (!getRootKeys().containsKey(key)) {
			return key;
		}
		return getRootKeys().get(key).displayName;
	}

	private void doCustomValidation(ENTITY entity, PREF pref) throws ValidationException {
		if (pref.getCategoryName().equals("root")) {
			if (!getRootKeys().containsKey(pref.getKey())) {
				throw new ValidationException("Custom Validation not found in root " + pref.getKey(),
						HttpStatus.NOT_FOUND);
			}
			if (getRootKeys().get(pref.getKey()).entityPrefValidationInterface != null) {
				getRootKeys().get(pref.getKey()).entityPrefValidationInterface.validate(entity, pref);
			}
		} else {
			if (!getCategoryServicesMap().containsKey(pref.getCategoryName())) {
				throw new ValidationException("Custom Validation key not found in category " + pref.getKey(),
						HttpStatus.NOT_FOUND);
			}
			if (getCategoryServicesMap().get(pref.getCategoryName()).getCustomValidation() != null) {
				getCategoryServicesMap().get(pref.getCategoryName()).getCustomValidation().validate(entity, pref);
			}
		}
	}

	private void mapPrefToField(ENTITY entity, PREF pref, User modifiedBy) {
		if (pref.getCategoryName().equals("root")) {
			EntityPrefValue<ENTITY, PREF> prefValue = getRootKeys().get(pref.getKey());
			if (prefValue != null && prefValue.getAdditionaProperties() != null
					&& prefValue.getAdditionaProperties().getMappedField() != null) {
				if (getCrudService() == null) {
					System.err.println("getCrudService() not overridden in EntityPrefService " + entity.getName());
					return;
				}
				prefValue.getAdditionaProperties().getMappedField().map(entity, pref);

			}

		}

	}

	protected List<PREF> removeHidden(List<PREF> prefs) {
		List<PREF> visiblePrefs = new ArrayList<>();
		for (PREF _pref : prefs) {
			if (_pref.getRemoteHideKey() == null || _pref.getRemoteHideKey().equals("")) {
				continue;
			}
			if (Boolean.parseBoolean(systemPrefService.getValue(_pref.getRemoteHideKey(), "true"))) {
				visiblePrefs.add(_pref);
			}
		}
		return visiblePrefs;
	}

	private List<? extends PREF> getMissingRootPrefs(List<PREF> prefs, ENTITY entity, EntityPrefCategory rootCategory) {
		List<PREF> missingPrefs = new ArrayList<>();

		PREF missingRootPref = null;
		boolean found = false;
		for (Map.Entry<String, EntityPrefValue<ENTITY, PREF>> entry : getRootKeys().entrySet()) {
			found = false;
			for (PREF pref : prefs) {
				if (pref.getKey().equalsIgnoreCase(entry.getKey())) {
					found = true;
					doPostFoundOperations(pref, entry.getValue(), entity);
				}
			}
			if (!found & !isDoNotAddIfMissing(entry.getKey(), entity)) {
				missingRootPref = createMissingRootPref(entity, rootCategory, entry.getKey(), entry.getValue(),
						entry.getValue().getOptions(entity), entry.getValue().group);
				if (entry.getValue().additionaProperties != null && entry.getValue().additionaProperties.getFieldPrefMapping() != null) {
					missingRootPref.setValue(entry.getValue().additionaProperties.getFieldPrefMapping().getValue(entity,
							missingRootPref));
				}
				missingPrefs.add(missingRootPref);
			}
			found = false;
		}

		return missingPrefs;
	}

	protected PREF doPostFoundOperations(PREF pref, EntityPrefValue<ENTITY, PREF> value, ENTITY entity) {
		pref.setOptions(value.getOptions(entity));
		pref.setGroup(value.group);
		pref.setDisplayName(value.displayName);
		return pref;
	}

	protected boolean isDoNotAddIfMissing(String key, ENTITY entity) {
		// Override this method in a subclass for a more relevant logic to not add a
		// missing root pref or category as pref
		return false;
	}

	private List<? extends PREF> getCategoriesAsPrefs(ENTITY entity) {
		List<PREF> prefs = new ArrayList<PREF>();
		categoryService.read(false, getEntityName()).forEach(category -> {
			if (!category.getName().equals("root")) {
				try {
					prefs.add(createCategoryAsPref(entity, category));
				} catch (ValidationException e) {
					System.err.println("Error in creating category as pref " + category.getName());
					e.printStackTrace();
				}
			}
		});
		return prefs;
	}

}
