package umun.entity.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Slice;
import org.springframework.data.domain.SliceImpl;
import org.springframework.http.HttpStatus;

import umun.core.constants.ValidationException;
import umun.entity.controller.TablePrefController;
import umun.entity.model.EntityPref;
import umun.entity.model.EntityPrefCategory;
import umun.entity.model.EntityPrefProp;
import umun.entity.model.EntityPrefValue;
import umun.entity.model.PrefDataType;
import umun.entity.model.TableEntityPref;
import umun.entity.model.TableEntityPrefId;
import umun.entity.model.TablePref;
import umun.entity.model.TablePrefSlotItem;
import umun.entity.model.TablePrefValidationValue;
import umun.entity.util.EntityPrefValidationUtil;
import umun.iam.model.Meta;
import umun.iam.model.User;

public abstract class TablePrefService<ENTITY extends Meta, SUB_ENTITY extends Meta, PREF extends TableEntityPref, ID extends TableEntityPrefId>
		extends EntityPrefOverrideService<ENTITY, PREF, ID> {

	public abstract String getTableName();

	protected abstract Slice<SUB_ENTITY> readSubEntities(ENTITY entity, int page, String searchTerm);

	protected abstract PREF createMissingRootPref(ENTITY entity, EntityPrefCategory rootCategory, String key,
			EntityPrefValue<ENTITY, PREF> value, Map<String, String> options, String group, SUB_ENTITY subEntity);

	protected abstract SUB_ENTITY findSubEntityOrFail(long subEntityId, User requestedBy) throws ValidationException;

	protected abstract PREF createPref(ENTITY entity, EntityPrefCategory category, String key, String value,
			String remoteHideKey, SUB_ENTITY subEntity) throws ValidationException;

	protected abstract List<PREF> readSavedPrefs(ENTITY entity, Slice<SUB_ENTITY> subEntities,
			EntityPrefCategory category);

	protected abstract ID createEntityPrefId(ENTITY entity, EntityPrefCategory category, String key,
			SUB_ENTITY subEntity);

	protected abstract void createSubEntities(ENTITY entity, String tableName,
			List<TablePref<EntityPref>> doubleListOfTablePref, User requestedBy);

	protected abstract Map<String, EntityPrefValue<ENTITY, PREF>> getRootKeys();
	
	@Override
	public void startUp(ContextRefreshedEvent event) {
		rootCategory = categoryService.findRootCategory(getEntityName());
		TablePrefController.addServiceReference(getEntityName(), getTableName(), this);
	}

	@Override
	public void addRootKeys(String rootKey, String defaultValue, String remoteHideKey, PrefDataType dataType,
			EntityPrefOptionsInterface optionsInterface, String displayName,
			EntityPrefValidationInterface<ENTITY, PREF> entityPrefValidationInterface, String group,
			boolean defaultRemoteHide, EntityPrefProp<ENTITY, PREF> properties) {

		systemPrefService.getValue(remoteHideKey, String.valueOf(defaultRemoteHide));
		getRootKeys().put(rootKey, new EntityPrefValue<ENTITY, PREF>(defaultValue, remoteHideKey, dataType,
				optionsInterface, displayName, entityPrefValidationInterface, group, properties, getTableName()));

	}

	public PREF createOrUpdate(ENTITY entity, EntityPrefCategory category, String key, String value, User requestedBy,
			SUB_ENTITY subEntity) throws ValidationException {
		PREF pref = readWithoutFail(entity, category, key, value, subEntity);
		String oldValue = pref.getValue();
		pref = doPostCreationOperations(entity, category, pref, key, value, requestedBy);
		mapSubEntityPrefToField(subEntity, pref, requestedBy);
		saveAndFlush(entity, category, pref, key, value, requestedBy, oldValue);
		return pref;
	}

	public Slice<TablePref<PREF>> read(String tableName, long entityId, int page, String searchTerm, User user)
			throws ValidationException {
		ENTITY entity = findEntityOrFail(entityId, user);
		Slice<SUB_ENTITY> subEntities = readSubEntities(entity, page, searchTerm);
		List<PREF> savedPrefs = readSavedPrefs(entity, subEntities, rootCategory);

		Map<String, PREF> savedPrefsMap = new HashMap<>();
		savedPrefs.forEach(pref -> {
			savedPrefsMap.put(createKeyForSubEntityMap(pref.getKey(), pref.getSubEntityId()), pref);
		});

		return mergeMissingPrefs(entity, subEntities, page, savedPrefsMap);
	}

	public PREF readWithoutFail(ENTITY entity, EntityPrefCategory category, String key, String value,
			SUB_ENTITY subEntity) throws ValidationException {
		PREF pref = read(createEntityPrefId(entity, category, key, subEntity));
		if (pref == null) {
			pref = createPref(entity, category, key, value, getRemoteHideKey(category, key), subEntity);
		}
		return pref;
	}

	public PREF update(long entityId, long categoryId, String key, String value, User requestedBy, String tableName,
			long subEntityId, List<TablePref<EntityPref>> doubleListOfTablePref) throws ValidationException {
		ENTITY entity = findEntityOrFail(entityId, requestedBy);
		if (subEntityId < 0) {
			this.createSubEntities(entity, tableName, doubleListOfTablePref, requestedBy);
		}

		return createOrUpdate(entity, categoryService.readOrFail(categoryId), key, value, requestedBy,
				findSubEntityOrFail(subEntityId, requestedBy));
	}

	private void mapSubEntityPrefToField(SUB_ENTITY entity, PREF pref, User modifiedBy) {
		if (pref.getCategoryName().equals("root")) {
			EntityPrefValue<ENTITY, PREF> prefValue = getRootKeys().get(pref.getKey());
			if (prefValue != null && prefValue.getAdditionaProperties() != null
					&& prefValue.getAdditionaProperties().getSubEntityMappedField() != null) {
				if (getCrudService() == null) {
					System.err.println("getCrudService() not overridden in EntityPrefService " + entity.getName());
					return;
				}
				prefValue.getAdditionaProperties().getSubEntityMappedField().map(entity, pref);

			}
		}
	}

	private Slice<TablePref<PREF>> mergeMissingPrefs(ENTITY entity, Slice<SUB_ENTITY> subEntities, int page,
			Map<String, PREF> savedPrefsMap) {
		List<TablePref<PREF>> missingPrefs = new ArrayList<>();

		subEntities.forEach(sub -> {
			List<PREF> prefs = new ArrayList<PREF>();
			getRootKeys().forEach((key, value) -> {
				if (savedPrefsMap.containsKey(createKeyForSubEntityMap(key, sub.getId()))) {
					prefs.add(doPostFoundOperations(savedPrefsMap.get(createKeyForSubEntityMap(key, sub.getId())),
							value, entity));
				} else {
					prefs.add(createMissingRootPref(entity, rootCategory, key, value, value.getOptions(entity),
							value.group, sub));
				}

			});
			missingPrefs.add(new TablePref<PREF>(sub, removeHidden(prefs), getSlotItems(sub)));
		});
		return new SliceImpl<>(missingPrefs, new PageRequest(page, subEntities.getSize()), subEntities.hasNext());
	}

	private String createKeyForSubEntityMap(String key, long subEntityId) {
		return String.format("%d::::%s", subEntityId, key);
	}

	public Slice<TablePref<EntityPref>> readForCreation(String tableName, User user,
			Map<String, String> allRequestParams) {
		// Override in sub class if that is of type table_crud
		throw new UnsupportedOperationException("Do not call this method: readForCreation - Override in sub class");
	}
	
	protected TablePrefSlotItem[] getSlotItems(SUB_ENTITY sub) {
		//Override in sub class if that is of type table_crud
		return null;
	}

	@Override
	public String validate(String key, EntityPref pref) throws ValidationException {
		EntityPrefValidationUtil.validate(pref);
		System.out.println(pref.getKey());
		return super.validate(key, pref);
	}

	public String validateAll(String key, TablePrefValidationValue value) throws ValidationException {
		double totalCredits = 0, totalDebits = 0;
		for (TablePref<EntityPref> tablePref : value.tablePrefs) {
			for (EntityPref pref : tablePref.prefs) {
				if (pref.getKey().equals("--jt-transaction-credits")) {
					totalCredits += Double.parseDouble(pref.getValue());
				}
				if (pref.getKey().equals("--jt-transaction-debits")) {
					totalDebits += Double.parseDouble(pref.getValue());
				}
			}
		}
		if(totalCredits != totalDebits) {
			throw new ValidationException("Total Credits != Total Debits", HttpStatus.NOT_ACCEPTABLE);
		}
		value.tablePrefs.forEach(p -> {
			p.prefs.forEach(pref -> {
				System.out.println(pref.getKey());
				System.out.println(pref.getValue());
				System.out.println("TODO: validation");
//TODO: validation required here
			});
		});
		return "Valid";
	}

}

abstract class EntityPrefOverrideService<ENTITY extends Meta, PREF extends EntityPref, ID extends TableEntityPrefId>
		extends EntityPrefService<ENTITY, PREF, ID> {
	/**
	 * Overridden by new concrete methods -v
	 */

	@Deprecated
	@Override
	public PREF createOrUpdate(ENTITY entity, EntityPrefCategory category, String key, String value, User requestedBy)
			throws ValidationException {
		throw new UnsupportedOperationException("Do not call this method: createOrUpdate");
	}

	@Deprecated
	@Override
	public PREF readWithoutFail(ENTITY entity, EntityPrefCategory category, String key, String value)
			throws ValidationException {
		throw new UnsupportedOperationException("Do not call this method: readWithoutFail");
	}

	@Deprecated
	@Override
	public PREF update(long entityId, long categoryId, String key, String value, User requestedBy)
			throws ValidationException {
		throw new UnsupportedOperationException("Do not call this method: update");
	}

	/**
	 * Overridden by new concrete methods -^
	 */

	/**
	 * Overridden by new abstract methods -v
	 */

	@Deprecated
	@Override
	protected PREF createMissingRootPref(ENTITY entity, EntityPrefCategory rootCategory, String key,
			EntityPrefValue<ENTITY, PREF> value, Map<String, String> options, String group) {
		throw new UnsupportedOperationException("Do not call this method: createMissingRootPref");
	}

	@Deprecated
	@Override
	protected PREF createPref(ENTITY entity, EntityPrefCategory category, String key, String value,
			String remoteHideKey) throws ValidationException {
		throw new UnsupportedOperationException("Do not call this method: createPref");
	}

	@Deprecated
	@Override
	protected ID createEntityPrefId(ENTITY entity, EntityPrefCategory category, String key) {
		throw new UnsupportedOperationException("Do not call this method: createEntityPrefId");
	}

	/**
	 * Overridden by new abstract methods -^
	 */

	@Deprecated
	@Override
	protected PREF createCategoryAsPref(ENTITY entity, EntityPrefCategory category) throws ValidationException {
		// do nothing here - as of now - tables do not support categories
		throw new UnsupportedOperationException("Do not call this method");
	}
}
