package umun.entity.service.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import umun.core.constants.ValidationException;
import umun.entity.model.EntityPrefCategory;
import umun.entity.model.EntityPrefProp;
import umun.entity.model.EntityPrefValue;
import umun.entity.model.PrefDataType;
import umun.entity.model.impl.NamedEntity;
import umun.entity.model.impl.NamedEntityPref;
import umun.entity.model.impl.NamedEntityPrefId;
import umun.entity.repository.EntityPrefRepository;
import umun.entity.repository.impl.NamedEntityPrefRepository;
import umun.entity.service.EntityPrefCategoryInterface;
import umun.entity.service.EntityPrefOptionsInterface;
import umun.entity.service.EntityPrefService;
import umun.entity.service.EntityPrefValidationInterface;
import umun.iam.model.User;

@Service
public abstract class NamedEntityPrefService extends EntityPrefService<NamedEntity, NamedEntityPref, NamedEntityPrefId>  {

	@Autowired
	private NamedEntityPrefRepository repository;
	
	@Autowired
	protected NamedEntityCRUDService namedEntityCRUDService;
	
	private static Map<String, EntityPrefCategoryInterface<NamedEntity, NamedEntityPref, NamedEntityPrefId, ?>> categoryServicesMap = new HashMap<>();

	private static Map<String, EntityPrefValue<NamedEntity, NamedEntityPref>> rootKeys = new HashMap<>();

	private static Map<String, String> keysAndEntityNames = new HashMap<>();
	
	protected abstract void addRootKeys();
	protected abstract void onStartup();
	
	@EventListener
	public void startUp(ContextRefreshedEvent event) {
		super.startUp(event);
		onStartup();
		addRootKeys();
	}
	
	@Override
	public void addRootKeys(String rootKey, String defaultValue, String remoteHideKey, PrefDataType dataType,
			EntityPrefOptionsInterface optionsInterface, String displayName, 
			EntityPrefValidationInterface<NamedEntity, NamedEntityPref> entityPrefValidationInterface,
			String group, boolean defaultRemoteHide, EntityPrefProp<NamedEntity, NamedEntityPref> properties) {
		System.err.println("addRootKeys method is not allowed for NamedEntityPrefService subclasses. Please use addRootKeys()  with entityName only.");
	}

	public void addRootKeys(String entityName, String rootKey, String defaultValue, String remoteHideKey, PrefDataType dataType,
			EntityPrefOptionsInterface optionsInterface, String displayName, 
			EntityPrefValidationInterface<NamedEntity, NamedEntityPref> entityPrefValidationInterface,
			String group, boolean defaultRemoteHide, EntityPrefProp<NamedEntity, NamedEntityPref> properties) {
		keysAndEntityNames.put(rootKey, entityName);
		super.addRootKeys(rootKey, defaultValue, remoteHideKey, dataType, optionsInterface,
				displayName, entityPrefValidationInterface, group, defaultRemoteHide, properties);
	}
	
	@Override
	public void addServiceReference(String categoryName, EntityPrefCategoryInterface<NamedEntity, NamedEntityPref, NamedEntityPrefId, ?> instance) {
		System.err.println("addServiceReference method is not allowed for NamedEntityPrefService subclasses. "
				+ "Named Entity Prefs do not support categories");
	}

	@Override
	protected EntityPrefRepository<NamedEntityPref, NamedEntityPrefId> getRepository() {
		return repository;
	}

	@Override
	protected Map<String, EntityPrefCategoryInterface<NamedEntity, NamedEntityPref, NamedEntityPrefId, ?>> getCategoryServicesMap() {
		return categoryServicesMap;
	}

	@Override
	protected Map<String, EntityPrefValue<NamedEntity, NamedEntityPref>> getRootKeys() {
		return rootKeys;
	}

	@Override
	protected List<NamedEntityPref> findByEntityAndCategoryOrderByKey(NamedEntity namedEntity,
			EntityPrefCategory rootCategory) {
		return repository.findByNamedEntityAndCategoryOrderByKey(namedEntity, rootCategory);
	}

	@Override
	protected NamedEntityPref createMissingRootPref(NamedEntity namedEntity, EntityPrefCategory category, String key,
			EntityPrefValue<NamedEntity, NamedEntityPref> entityPrefValue, Map<String, String> options, String group) {
		return new NamedEntityPref(key, category.getId(), category, entityPrefValue.value, entityPrefValue.dataType,
				entityPrefValue.remoteHideKey, namedEntity.getId(), namedEntity, options, entityPrefValue.displayName,
				group);
	}
	
	protected boolean isDoNotAddIfMissing(String key, NamedEntity namedEntity) {
		return !keysAndEntityNames.get(key).equals(namedEntity.getEntityName());
	}

	@Override
	protected NamedEntityPref createCategoryAsPref(NamedEntity namedEntity, EntityPrefCategory category)
			throws ValidationException {
		return new NamedEntityPref(category.getName(), category.getId(), category, null, PrefDataType.category,
				getRemoteHideKey(category, category.getName()), namedEntity.getId(), namedEntity,
				getOptions(category, category.getName(), namedEntity), category.getDisplayName(),
				getGroup(category, category.getName()));
	}

	@Override
	protected NamedEntityPrefId createEntityPrefId(NamedEntity namedEntity, EntityPrefCategory category, String key) {
		return new NamedEntityPrefId(namedEntity.getId(), key, category.getId());
	}

	@Override
	protected NamedEntityPref createPref(NamedEntity namedEntity, EntityPrefCategory category, String key, String value,
			String remoteHideKey) throws ValidationException {
		return new NamedEntityPref(key, category.getId(), category, value, getDataType(category, key), remoteHideKey,
				namedEntity.getId(), namedEntity, getOptions(category, key, namedEntity), getDisplayname(key),
				getGroup(category, key));
	}

	@Override
	protected NamedEntity findEntityOrFail(long entityId, User requestedBy) throws ValidationException {
		return namedEntityCRUDService.readOrFail(entityId, requestedBy);
	}

}
