package umun.entity.templates.service;

import java.util.List;
import java.util.Set;

import org.springframework.http.HttpStatus;

import umun.core.constants.ValidationException;
import umun.core.util.DateUtil;
import umun.entity.model.EntityCreateOption;
import umun.entity.model.impl.NamedEntity;
import umun.entity.templates.repository.EntityCRUDRepository;
import umun.iam.model.Meta;
import umun.iam.model.User;

public abstract class EntityCRUDService<ENTITY extends Meta> {
	
	protected abstract EntityService<ENTITY, ?, ?> getEntityService();
	
	protected abstract EntityCRUDRepository<ENTITY> getRepository();

	protected abstract String getEntityName();

	protected abstract void validateCreate(ENTITY entity, User createdBy) throws ValidationException;

	protected abstract void validateUpdate(ENTITY savedEntity, ENTITY entity, User modifiedBy)
			throws ValidationException;

	protected abstract void validateDelete(ENTITY savedEntity, User modifiedBy) throws ValidationException;

	protected abstract ENTITY setValuesOnUpdate(ENTITY savedEntity, ENTITY entity, User modifiedBy)
			throws ValidationException;

	protected abstract ENTITY setVariablesOnCreate(ENTITY entity) throws ValidationException;
	
	protected abstract EntityFilterService<ENTITY> getFilterService();

	public ENTITY create(ENTITY entity, User createdBy) throws ValidationException {
		if (entity == null) {
			throw new ValidationException("Provide " + getEntityName(), HttpStatus.METHOD_NOT_ALLOWED);
		}

		validateCreate(entity, createdBy);

		if (entity.getId() != null && getRepository().findOne(entity.getId()) != null) {
			throw new ValidationException(getEntityName() + " ID already exists", HttpStatus.METHOD_NOT_ALLOWED);
		}

		entity.setCreateTime(DateUtil.getCurrentTimeInIST());
		entity.setCreatedBy(createdBy);

		//saving twice so that while setting additional variable ID is available 
		entity = getRepository().saveAndFlush(
				setVariablesOnCreate(
						getRepository().saveAndFlush(entity)
						) 
				);
		
		afterCreate(entity);
		
		return entity;
	}

	public List<ENTITY> read() throws ValidationException {
		return addPrefs(getRepository().findByDeletedOrderByNameDesc(false));
	}

	public List<ENTITY> read(boolean allowDeleted) {
		if (allowDeleted) {
			return getRepository().findAll();
		}
		return getRepository().findByDeletedOrderByNameDesc(false);
	}
	
	public List<ENTITY> read(NamedEntity filter) throws ValidationException {
		if(getFilterService() == null) {
			return read();
		}
		return addPrefs(getFilterService().read(filter));
	}
	
	public List<ENTITY> read(Set<Long> ids) throws ValidationException {
		return addPrefs(getRepository().findByDeletedAndIdInOrderByIdDesc(false, ids));
	}

	public ENTITY readOrFail(Long entityId, User requestedBy) throws ValidationException {
		ENTITY entity = getRepository().findOne(entityId);
		if (entity == null) {
			throw new ValidationException(String.format("%s not found", getEntityName()), HttpStatus.NOT_FOUND);
		}
		return entity;
	}

	public ENTITY update(long entityId, ENTITY entity, User modifiedBy) throws ValidationException {
		ENTITY savedEntity = readOrFail(entityId, modifiedBy);

		validateUpdate(savedEntity, entity, modifiedBy);
		savedEntity = setValuesOnUpdate(savedEntity, entity, modifiedBy);

		savedEntity.setName(entity.getName());
		savedEntity.setModificationTime(DateUtil.getCurrentTimeInIST());
		savedEntity.setModifiedBy(modifiedBy);
		return getRepository().saveAndFlush(savedEntity);
	}

	public ENTITY delete(long entityId, User modifiedBy) throws ValidationException {
		ENTITY savedEntity = readOrFail(entityId, modifiedBy);

		validateDelete(savedEntity, modifiedBy);

		savedEntity.setDeleted(true);
		savedEntity.setModificationTime(DateUtil.getCurrentTimeInIST());
		savedEntity.setModifiedBy(modifiedBy);
		return getRepository().saveAndFlush(savedEntity);
	}

	public List<ENTITY> addPrefs(List<ENTITY> entites) throws ValidationException {
		return getEntityService().addPrefs(entites);
	}
	
	protected void afterCreate(ENTITY entity) {
		//override this method to do something after an entity is created
	}
	
	public List<EntityCreateOption> getCreateOptions() throws ValidationException{
		return null;
	}
}
