package umun.entity.templates.service;

import java.util.ArrayList;
import java.util.List;

import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;

import umun.core.constants.ValidationException;
import umun.entity.model.EntityPref;
import umun.entity.model.EntityPrefId;
import umun.entity.model.EntityPrefProp;
import umun.entity.model.PrefDataType;
import umun.entity.service.EntityPrefOptionsInterface;
import umun.entity.service.EntityPrefService;
import umun.entity.service.EntityPrefValidationInterface;
import umun.iam.model.Meta;

public abstract class EntityService<ENTITY extends Meta, PREF extends EntityPref, ID extends EntityPrefId> {

	/*
	 * Add transient prefs to the entity, this method is called while reading the
	 * entity.
	 * 
	 * For this function to work please call addPrefs(List<ENTITY>) from the
	 * crudService of your entity.
	 * 
	 * @param entity
	 * @return
	 * @throws ValidationException
	 */
	protected abstract ENTITY addPref(ENTITY entity) throws ValidationException;

	/*
	 * Add root preferences, these are the fields of the entity's form (prefs)
	 * 
	 * call function addRootKeys(...) from here
	 * 
	 */
	protected abstract void addRootsPref();

	/*
	 * This is a hook for application startup event
	 */
	protected abstract void onStartUp();

	protected abstract EntityPrefService<ENTITY, PREF, ID> getPrefService();

	@EventListener
	public void onApplicationEvent(ContextRefreshedEvent event) {
		onStartUp();
		addRootsPref();
	}

	/*
	 * This function should be called from the read method of the CRUD service where
	 * prefs are required
	 * 
	 * @param entities
	 * @return
	 * @throws ValidationException
	 */
	public List<ENTITY> addPrefs(List<ENTITY> entities) throws ValidationException {
		if (entities == null) {
			System.err.println("EntityService: Null entites list sent to addPrefs");
			return new ArrayList<>();
		}
		for (ENTITY entity : entities) {
			entity = addPref(entity);
		}
		return entities;
	}
	
	

	public void addRootKeys(String rootKey, String defaultValue, String remoteHideKey, PrefDataType dataType,
			EntityPrefOptionsInterface optionsInterface, String displayName,
			EntityPrefValidationInterface<ENTITY, PREF> entityPrefValidationInterface, String group,
			boolean defaultRemoteHide, EntityPrefProp properties) {
		getPrefService().addRootKeys(rootKey, defaultValue, remoteHideKey, dataType, optionsInterface, displayName,
				entityPrefValidationInterface, group, defaultRemoteHide, properties);
	}
}
