package umun.entity.util;

import java.util.Map;
import java.util.regex.Pattern;

import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.http.HttpStatus;

import umun.core.constants.ValidationException;
import umun.entity.model.EntityPref;

public class EntityPrefValidationUtil {

	public static void validate(EntityPref pref) throws ValidationException {
		switch (pref.getDatatype()) {
		case String:
			validateString(pref.getValue());
			break;

		case Long:
			validateLong(pref.getValue());
			break;

		case Double:
			validateDouble(pref.getValue());
			break;

		case Boolean:
			validateBoolean(pref.getValue());
			break;

		case Select:
			validateSelect(pref.getValue(), pref.getOptions());
			break;

		case MultiSelect:
			validateMultiSelect(pref.getValue(), pref.getOptions());
			break;

		case Long_Positive:
			validateLongPositive(pref.getValue());
			break;

		case Long_Negative:
			validateLongNegative(pref.getValue());
			break;

		case Double_Positive:
			validateDoublePositive(pref.getValue());
			break;

		case Double_Negative:
			validateDoubleNegative(pref.getValue());
			break;

		case MultiInput_String:
			validateMultiInputString(pref.getValue(), pref.getOptions());
			break;
		case MultiInput_Long:
			validateMultiInputLong(pref.getValue(), pref.getOptions());
			break;
		case MultiInput_Double:
			validateMultiInputDouble(pref.getValue(), pref.getOptions());
			break;
		case MultiInput_Boolean:
			validateMultiInputBoolean(pref.getValue(), pref.getOptions());
			break;
			
		case HH_MM:
			validateHHMM(pref.getValue());
			break;

		case MultiInput_HH_MM:
			validateMultiHHMM(pref.getValue(), pref.getOptions());
			break;
			
		case YYYY_MM_DD:
			validateYYYY_MM_DD(pref.getValue());
			break;
			
		default:
			break;
		}
	}

	private static void validateYYYY_MM_DD(String value) throws ValidationException {
		if(value == null ||
				!Pattern.compile("^((?:(?:1[6-9]|2[0-9])\\d{2})(-)(?:(?:(?:0[13578]|1[02])(-)31)|((0[1,3-9]|1[0-2])(-)(29|30))))$|^(?:(?:(?:(?:1[6-9]|[2-9]\\d)?(?:0[48]|[2468][048]|[13579][26])|(?:(?:16|[2468][048]|[3579][26])00)))(-)02(-)29)$|^(?:(?:1[6-9]|2[0-9])\\d{2})(-)(?:(?:0[1-9])|(?:1[0-2]))(-)(?:0[1-9]|1\\d|2[0-8])$").matcher(value).matches()) {
			throw new ValidationException("Enter a valid date in format yyyy-MM-dd 2001-12-21", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static void validateMultiHHMM(String value, Map<String, String> options) throws ValidationException {
		validateMultiInput(value, options, (JSONObject, key) -> {
			validateHHMM(JSONObject.getString(key));
		}, "  a valid time in 24 hour format HH:MM like 09:00, 23:30 ..");
	}

	private static void validateHHMM(String value) throws ValidationException {
		if(value == null ||
				!Pattern.compile("([01]?[0-9]|2[0-3]):[0-5][0-9]").matcher(value).matches()) {
			throw new ValidationException("Enter a valid time in 24 hour format HH:MM like 09:00, 23:30 ..", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static void validateMultiInputBoolean(String value, Map<String, String> options)
			throws ValidationException {
		validateMultiInput(value, options, (JSONObject, key) -> {
			JSONObject.getBoolean(key);
		}, " either 'true' or 'false'");
	}

	private static void validateMultiInputDouble(String value, Map<String, String> options) throws ValidationException {
		validateMultiInput(value, options, (JSONObject, key) -> {
			JSONObject.getDouble(key);
		}, " a decimal number like 1.0, 10.50, 100.20 ..");
	}

	private static void validateMultiInputLong(String value, Map<String, String> options) throws ValidationException {
		validateMultiInput(value, options, (JSONObject, key) -> {
			JSONObject.getLong(key);
		}, " a whole number like 1, 2, 3...");

	}

	private static void validateMultiInputString(String value, Map<String, String> options) throws ValidationException {
		validateMultiInput(value, options, (JSONObject, key) -> {
			JSONObject.getString(key);
		}, " a valid string of characters");
	}

	private static void validateDoubleNegative(String value) throws ValidationException {
		if (validateDouble(value) > 0) {
			throw new ValidationException("Enter a value smaller than 0.", HttpStatus.NOT_ACCEPTABLE);
		}

	}

	private static void validateDoublePositive(String value) throws ValidationException {
		if (validateDouble(value) < 0) {
			throw new ValidationException("Enter a value greater than 0.", HttpStatus.NOT_ACCEPTABLE);
		}

	}

	private static void validateLongNegative(String value) throws ValidationException {
		if (validateLong(value) > 0) {
			throw new ValidationException("Enter a value smaller than 0.", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static void validateLongPositive(String value) throws ValidationException {
		if (validateLong(value) < 0) {
			throw new ValidationException("Enter a value greater than 0.", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static void validateMultiSelect(String value, Map<String, String> options) throws ValidationException {
		// TODO write validation after atleast one of multi select option is made
		throw new ValidationException("No validation found.", HttpStatus.NOT_FOUND);
	}

	private static void validateSelect(String value, Map<String, String> options) throws ValidationException {
		if (!options.containsKey(value)) {
			throw new ValidationException(String.format("Select a value from the provided %d options.", options.size()),
					HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static boolean validateBoolean(String value) throws ValidationException {
		try {
			validateEmpty(value);
			return Boolean.parseBoolean(value);
		} catch (NumberFormatException e) {
			throw new ValidationException("Enter a valid value 'true' or 'false'", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static double validateDouble(String value) throws ValidationException {
		try {
			validateEmpty(value);
			return Double.parseDouble(value);
		} catch (NumberFormatException e) {
			throw new ValidationException("Enter a valid decimal number like 0, 1.1, 2.5 ..",
					HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static long validateLong(String value) throws ValidationException {
		try {
			validateEmpty(value);
			return Long.parseLong(value);
		} catch (NumberFormatException e) {
			throw new ValidationException("Enter a valid whole number like ..-2, -1, 0, 1, 2..",
					HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private static void validateEmpty(String value) {
		if(value == null || value.equals("")) {
			throw new NumberFormatException();
		}
		
	}

	private static void validateString(String value) throws ValidationException {
		if (value == null) {
			throw new ValidationException("Value cannot be null or empty", HttpStatus.NOT_ACCEPTABLE);
		}
	}

	private interface MultiInputValidationInt {
		void validateInididualOption(JSONObject jsonObject, String key) throws JSONException, ValidationException;
	}

	private static void validateMultiInput(String value, Map<String, String> options,
			MultiInputValidationInt multiInputValidationInt, String message) throws ValidationException {
		try {
			final JSONObject ob = new JSONObject(value);
			for (String k : options.keySet()) {
				// value should have all keys
				if (!ob.has(k)) {
					throw new ValidationException("Enter value of " + options.get(k) + " key " + k,
							HttpStatus.NOT_ACCEPTABLE);
				}
				// all values should be of that type
				try {
					multiInputValidationInt.validateInididualOption(ob, k);
				} catch (JSONException | ValidationException e) {
					throw new ValidationException(String.format("Value of '%s' should be %s",  options.get(k), message),
							HttpStatus.NOT_ACCEPTABLE);
				}
			}
		} catch (JSONException e) {
			throw new ValidationException("Not a valid json. Contact developer.", HttpStatus.NOT_ACCEPTABLE);
		}
	}

}
