package com.vrs.android.fyreon.utils.net;

import android.content.Context;

import com.squareup.okhttp.Cache;
import com.squareup.okhttp.CacheControl;
import com.squareup.okhttp.Callback;
import com.squareup.okhttp.Interceptor;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;
import com.vrs.android.fyreon.utils.AppConstants;
import com.vrs.android.fyreon.utils.logger.Logger;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HttpsURLConnection;

/**
 * Package Name : com.vrs.android.fyreon.utils.net
 * Created by vrsuser on 7/5/15.
 * Time : 12:39 PM
 * Project : FyreonTest
 * Modified : 07 12 39
 */
public class OkHttpClientUtil {
    /* Constants */
    private static final String LOG_TAG = OkHttpClientUtil.class.getSimpleName();
    private static final Logger logger = new Logger(AppConstants.isDebug, LOG_TAG);
    private static final MediaType JSON
            = MediaType.parse("application/json; charset=utf-8");
    private static long HTTP_CACHE_SIZE = 10 * 1024 * 1024; // 10 MiB
    /* Properties */
    private static OkHttpClient mHttpClient;
    private static Cache cache;
    private static OkHttpClientUtil instance = null;
    private Context context;

    public static OkHttpClientUtil getInstance() {
        if (instance == null) {
            instance = new OkHttpClientUtil();
        }
        return instance;
    }

    public static void doPostRequest(String url, String params, Callback callback) throws IOException {
        RequestBody body = RequestBody.create(JSON, params);
        Request cachedRequest = new Request.Builder()
                .cacheControl(new CacheControl.Builder().onlyIfCached().build())
                .url(url)
                .post(body)
                .build();
        Response forceCacheResponse = mHttpClient.newCall(cachedRequest).execute();
        if (forceCacheResponse.code() != 504) {
            // The resource was cached! Show it.
            callback.onResponse(forceCacheResponse);

        } else {
            Request networkRequest = new Request.Builder().url(url).post(body).build();
            mHttpClient.newCall(networkRequest).enqueue(callback);
        }
    }

    public static void doPostRequest(String url, Callback callback) throws IOException {
        Request cachedRequest = new Request.Builder()
                .cacheControl(new CacheControl.Builder().onlyIfCached().build())
                .url(url)
                .build();
        Response forceCacheResponse = mHttpClient.newCall(cachedRequest).execute();
        if (forceCacheResponse.code() != 504) {
            // The resource was cached! Show it.
            callback.onResponse(forceCacheResponse);

        } else {
            Request networkRequest = new Request.Builder().url(url).build();
            mHttpClient.newCall(networkRequest).enqueue(callback);
        }
    }

    public static void doPostRequest(String url, String[] headerKey, String[] headerVal, String
            params, Callback callback) throws IOException {
        if (headerKey.length > 0) {
            if (headerKey.length == headerVal.length) {
                RequestBody body = RequestBody.create(JSON, params);
                Request cachedRequest = new Request.Builder()
                        .cacheControl(new CacheControl.Builder().onlyIfCached().build())
                        .url(url)
                        .post(body)
                        .build();
                for (int i = 0; i < headerKey.length; i++) {
                    cachedRequest.newBuilder().addHeader(headerKey[i], headerVal[i]);
                }
                Response forceCacheResponse = mHttpClient.newCall(cachedRequest).execute();
                if (forceCacheResponse.code() != 504) {
                    // The resource was cached! Show it.
                    callback.onResponse(forceCacheResponse);

                } else {
                    Request networkRequest = new Request.Builder().url(url).post(body).build();
                    for (int i = 0; i < headerKey.length; i++) {
                        networkRequest.newBuilder().addHeader(headerKey[i], headerVal[i]);
                    }
                    mHttpClient.newCall(networkRequest).enqueue(callback);
                }

            } else {
                throw new ArrayIndexOutOfBoundsException("Header key and Header Value Array do " +
                        "not match");
            }
        } else {
            throw new IllegalArgumentException("You need to provide Headers for Call");
        }
    }

    private static Cache createHttpClientCache(Context context) {
        File cacheDir = context.getDir("cache_api", Context.MODE_PRIVATE);
        return new Cache(cacheDir, HTTP_CACHE_SIZE);

    }

    /**
     * Initialize HttpClient
     */
    public void initialize(Context context) {
        setContext(context);

        mHttpClient = new OkHttpClient();
        mHttpClient.networkInterceptors().add(new LoggingInterceptor());
        mHttpClient.setFollowRedirects(true);
        mHttpClient.setFollowSslRedirects(true);
        configureSslSocketFactory();

        configureCache();

        configureTimeouts();
    }

    private void setContext(Context context) {
        this.context = context;
    }

    private void configureSslSocketFactory() {
        mHttpClient.setSslSocketFactory(HttpsURLConnection.getDefaultSSLSocketFactory());
    }

    private void configureCache() {
        if (cache == null)
            cache = createHttpClientCache(context);

        mHttpClient.setCache(cache);
    }

    private void configureTimeouts() {
        mHttpClient.setConnectTimeout(10, TimeUnit.SECONDS);
        mHttpClient.setWriteTimeout(35, TimeUnit.SECONDS);
        mHttpClient.setReadTimeout(35, TimeUnit.SECONDS);
    }

    class LoggingInterceptor implements Interceptor {
        @Override
        public Response intercept(Chain chain) throws IOException {
            Request request = chain.request();

            long t1 = System.nanoTime();
            logger.info(LOG_TAG, String.format("Sending request %s on %s%n%s",
                    request.url(), chain.connection(), request.headers()));

            Response response = chain.proceed(request);

            long t2 = System.nanoTime();
            logger.info(LOG_TAG, String.format("Received response for %s in %.1fms%n%s",
                    response.request().url(), (t2 - t1) / 1e6d, response.headers()));
            return response;
        }
    }
}
