/*
 * Decompiled with CFR 0.152.
 */
package io.openshift.booster.catalog;

import io.openshift.booster.CopyFileVisitor;
import io.openshift.booster.Files;
import io.openshift.booster.catalog.Booster;
import io.openshift.booster.catalog.Mission;
import io.openshift.booster.catalog.Runtime;
import io.openshift.booster.catalog.YamlConstructor;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.json.Json;
import javax.json.JsonObject;
import javax.json.JsonReader;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.BaseConstructor;

public class BoosterCatalogService {
    private static final String GITHUB_URL = "https://github.com/";
    private static final String CLONED_BOOSTERS_DIR = ".boosters";
    private static final String METADATA_FILE = "metadata.json";
    private static final Yaml yaml = new Yaml((BaseConstructor)new YamlConstructor());
    private static final List<String> EXCLUDED_PROJECT_FILES = Arrays.asList(".git", ".travis", ".travis.yml", ".ds_store", ".obsidian", ".gitmodules");
    private static final Logger logger = Logger.getLogger(BoosterCatalogService.class.getName());
    private final ReentrantReadWriteLock reentrantLock = new ReentrantReadWriteLock();
    private volatile List<Booster> boosters = Collections.emptyList();
    private String catalogRepositoryURI;
    private String catalogRef;

    private BoosterCatalogService(String catalogRepositoryURI, String catalogRef) {
        this.catalogRepositoryURI = catalogRepositoryURI;
        this.catalogRef = catalogRef;
    }

    public void index() {
        ReentrantReadWriteLock.WriteLock lock = this.reentrantLock.writeLock();
        try {
            lock.lock();
            logger.log(Level.INFO, "Indexing contents from {0} using {1} ref", new Object[]{this.catalogRepositoryURI, this.catalogRef});
            Path catalogPath = java.nio.file.Files.createTempDirectory("booster-catalog", new FileAttribute[0]);
            catalogPath.toFile().deleteOnExit();
            logger.info(() -> "Created " + catalogPath);
            Git.cloneRepository().setURI(this.catalogRepositoryURI).setBranch(this.catalogRef).setCloneSubmodules(true).setDirectory(catalogPath.toFile()).call().close();
            this.boosters = this.indexBoosters(catalogPath);
        }
        catch (GitAPIException e) {
            logger.log(Level.SEVERE, "Error while performing Git operation", e);
        }
        catch (Exception e) {
            logger.log(Level.SEVERE, "Error while indexing", e);
        }
        finally {
            logger.info(() -> "Finished content indexing");
            lock.unlock();
        }
    }

    private List<Booster> indexBoosters(Path catalogPath) throws IOException {
        final Path moduleRoot = catalogPath.resolve(CLONED_BOOSTERS_DIR);
        Path metadataFile = catalogPath.resolve(METADATA_FILE);
        final ArrayList<Booster> boosters = new ArrayList<Booster>();
        final HashMap<String, Mission> missions = new HashMap<String, Mission>();
        final HashMap<String, Runtime> runtimes = new HashMap<String, Runtime>();
        if (java.nio.file.Files.exists(metadataFile, new LinkOption[0])) {
            this.processMetadata(metadataFile, missions, runtimes);
        }
        java.nio.file.Files.walkFileTree(catalogPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                File ioFile = file.toFile();
                String fileName = ioFile.getName().toLowerCase();
                if (!fileName.startsWith(".") && (fileName.endsWith(".yaml") || fileName.endsWith(".yml"))) {
                    String id = Files.removeFileExtension(fileName);
                    Path modulePath = moduleRoot.resolve(id);
                    BoosterCatalogService.this.indexBooster(id, file, modulePath, missions, runtimes).ifPresent(boosters::add);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return dir.startsWith(moduleRoot) ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
            }
        });
        boosters.sort(Comparator.comparing(Booster::getName));
        return Collections.unmodifiableList(boosters);
    }

    void processMetadata(Path metadataFile, Map<String, Mission> missions, Map<String, Runtime> runtimes) {
        logger.info(() -> "Reading metadata at " + metadataFile + " ...");
        try (BufferedReader reader = java.nio.file.Files.newBufferedReader(metadataFile);
             JsonReader jsonReader = Json.createReader((Reader)reader);){
            JsonObject index = jsonReader.readObject();
            index.getJsonArray("missions").stream().map(JsonObject.class::cast).map(e -> new Mission(e.getString("id"), e.getString("name"))).forEach(m -> missions.put(m.getId(), (Mission)m));
            index.getJsonArray("runtimes").stream().map(JsonObject.class::cast).map(e -> new Runtime(e.getString("id"), e.getString("name"))).forEach(r -> runtimes.put(r.getId(), (Runtime)r));
        }
        catch (IOException e2) {
            logger.log(Level.SEVERE, "Error while processing metadata " + metadataFile, e2);
        }
    }

    private Optional<Booster> indexBooster(String id, Path file, Path moduleDir, Map<String, Mission> missions, Map<String, Runtime> runtimes) {
        logger.info(() -> "Indexing " + file + " ...");
        Booster booster = null;
        try (BufferedReader reader = java.nio.file.Files.newBufferedReader(file);){
            booster = (Booster)yaml.loadAs((Reader)reader, Booster.class);
        }
        catch (IOException e) {
            logger.log(Level.SEVERE, "Error while reading " + file, e);
        }
        if (booster != null) {
            try {
                booster.setId(id);
                String runtimeId = file.getParent().toFile().getName();
                String missionId = file.getParent().getParent().toFile().getName();
                booster.setMission(missions.computeIfAbsent(missionId, Mission::new));
                booster.setRuntime(runtimes.computeIfAbsent(runtimeId, Runtime::new));
                booster.setContentPath(moduleDir);
                if (java.nio.file.Files.notExists(moduleDir, new LinkOption[0])) {
                    try (Git git = Git.cloneRepository().setDirectory(moduleDir.toFile()).setURI(GITHUB_URL + booster.getGithubRepo()).setCloneSubmodules(true).setBranch(booster.getGitRef()).call();){
                        git.checkout().setName(booster.getGitRef()).setStartPoint(booster.getGitRef()).call();
                    }
                }
                Path metadataPath = moduleDir.resolve(booster.getBoosterDescriptorPath());
                try (BufferedReader metadataReader = java.nio.file.Files.newBufferedReader(metadataPath);){
                    Map metadata = (Map)yaml.loadAs((Reader)metadataReader, Map.class);
                    booster.setMetadata(metadata);
                }
                Path descriptionPath = moduleDir.resolve(booster.getBoosterDescriptionPath());
                if (java.nio.file.Files.exists(descriptionPath, new LinkOption[0])) {
                    byte[] descriptionContent = java.nio.file.Files.readAllBytes(descriptionPath);
                    booster.setDescription(new String(descriptionContent));
                }
            }
            catch (GitAPIException gitException) {
                logger.log(Level.SEVERE, "Error while reading git repository", gitException);
            }
            catch (Exception e) {
                logger.log(Level.SEVERE, "Error while reading metadata from " + file, e);
            }
        }
        return Optional.ofNullable(booster);
    }

    public Path copy(Booster booster, Path projectRoot) throws IOException {
        Path modulePath = booster.getContentPath();
        return java.nio.file.Files.walkFileTree(modulePath, new CopyFileVisitor(projectRoot, p -> !EXCLUDED_PROJECT_FILES.contains(p.toFile().getName().toLowerCase())));
    }

    public Set<Mission> getMissions() {
        return this.boosters.stream().map(Booster::getMission).collect(Collectors.toCollection(TreeSet::new));
    }

    public Set<Runtime> getRuntimes(Mission mission) {
        if (mission == null) {
            return Collections.emptySet();
        }
        return this.boosters.stream().filter(b -> mission.equals(b.getMission())).map(Booster::getRuntime).collect(Collectors.toCollection(TreeSet::new));
    }

    public Optional<Booster> getBooster(Mission mission, Runtime runtime) {
        Objects.requireNonNull(mission, "Mission should not be null");
        Objects.requireNonNull(runtime, "Runtime should not be null");
        return this.boosters.stream().filter(b -> mission.equals(b.getMission())).filter(b -> runtime.equals(b.getRuntime())).findFirst();
    }

    public List<Booster> getBoosters() {
        ReentrantReadWriteLock.ReadLock readLock = this.reentrantLock.readLock();
        try {
            readLock.lock();
            List<Booster> list = this.boosters;
            return list;
        }
        finally {
            readLock.unlock();
        }
    }

    public static class Builder {
        private String catalogRepositoryURI = "https://github.com/openshiftio/booster-catalog.git";
        private String catalogRef = "master";

        public Builder catalogRef(String catalogRef) {
            this.catalogRef = catalogRef;
            return this;
        }

        public Builder catalogRepository(String catalogRepositoryURI) {
            this.catalogRepositoryURI = catalogRepositoryURI;
            return this;
        }

        public BoosterCatalogService build() {
            return new BoosterCatalogService(this.catalogRepositoryURI, this.catalogRef);
        }
    }
}

