/*
 * Decompiled with CFR 0.152.
 */
package overflowdb;

import com.sun.management.GarbageCollectionNotificationInfo;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import javax.management.ListenerNotFoundException;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class HeapUsageMonitor
implements AutoCloseable {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<NotificationEmitter, NotificationListener> gcNotificationListeners = new HashMap<NotificationEmitter, NotificationListener>(2);

    public HeapUsageMonitor(int heapPercentageThreshold, HeapNotificationListener notificationListener) {
        if (heapPercentageThreshold < 0 || heapPercentageThreshold > 100) {
            throw new IllegalArgumentException("heapPercentageThreshold must be between 0 and 100, but is " + heapPercentageThreshold);
        }
        float heapUsageThreshold = (float)heapPercentageThreshold / 100.0f;
        this.installGCMonitoring(heapUsageThreshold, notificationListener);
    }

    protected void installGCMonitoring(float heapUsageThreshold, HeapNotificationListener notificationListener) {
        List<GarbageCollectorMXBean> gcbeans = ManagementFactory.getGarbageCollectorMXBeans();
        for (GarbageCollectorMXBean gcbean : gcbeans) {
            NotificationListener listener = this.createNotificationListener(heapUsageThreshold, notificationListener);
            NotificationEmitter emitter = (NotificationEmitter)((Object)gcbean);
            emitter.addNotificationListener(listener, null, null);
            this.gcNotificationListeners.put(emitter, listener);
        }
        int heapUsageThresholdPercent = (int)Math.floor(heapUsageThreshold * 100.0f);
        this.logger.info("installed GC monitors. will clear references if heap (after GC) is larger than " + heapUsageThresholdPercent + "%");
    }

    private NotificationListener createNotificationListener(float heapUsageThreshold, HeapNotificationListener notificationListener) {
        Map capturedMDC = MDC.getCopyOfContextMap();
        HashSet<String> ignoredMemoryAreas = new HashSet<String>(Arrays.asList("Code Cache", "Compressed Class Space", "Metaspace"));
        return (notification, handback) -> {
            if (notification.getType().equals("com.sun.management.gc.notification")) {
                GarbageCollectionNotificationInfo info = GarbageCollectionNotificationInfo.from((CompositeData)notification.getUserData());
                long totalMemUsed = 0L;
                long totalMemMax = 0L;
                for (Map.Entry<String, MemoryUsage> entry : info.getGcInfo().getMemoryUsageAfterGc().entrySet()) {
                    String name = entry.getKey();
                    if (ignoredMemoryAreas.contains(name)) continue;
                    MemoryUsage detail = entry.getValue();
                    totalMemUsed += detail.getUsed();
                    totalMemMax += detail.getMax();
                }
                float heapUsage = (float)totalMemUsed / (float)totalMemMax;
                int heapUsagePercent = (int)Math.floor(heapUsage * 100.0f);
                if (heapUsage > heapUsageThreshold) {
                    Map oldMDC = MDC.getCopyOfContextMap();
                    try {
                        MDC.setContextMap((Map)capturedMDC);
                        String msg = "heap usage after GC: " + heapUsagePercent + "% -> will clear some references (if possible)";
                        if (heapUsagePercent > 95) {
                            this.logger.warn(msg);
                        } else {
                            this.logger.info(msg);
                        }
                        notificationListener.notifyHeapAboveThreshold();
                    }
                    finally {
                        MDC.setContextMap((Map)oldMDC);
                    }
                } else {
                    this.logger.trace("heap usage after GC: " + heapUsagePercent + "%");
                }
            }
        };
    }

    @Override
    public void close() {
        while (!this.gcNotificationListeners.isEmpty()) {
            Map.Entry<NotificationEmitter, NotificationListener> entry = this.gcNotificationListeners.entrySet().iterator().next();
            try {
                entry.getKey().removeNotificationListener(entry.getValue());
                this.gcNotificationListeners.remove(entry.getKey());
            }
            catch (ListenerNotFoundException e) {
                throw new RuntimeException("unable to remove GC monitor", e);
            }
        }
        this.logger.debug("uninstalled GC monitors.");
    }

    static interface HeapNotificationListener {
        public void notifyHeapAboveThreshold();
    }
}

