package ir.kavoshgaran.bitrah.services;

import com.fasterxml.jackson.databind.ObjectMapper;
import ir.kavoshgaran.bitrah.entities.*;
import ir.kavoshgaran.bitrah.exception.BitrahErrorResponse;
import ir.kavoshgaran.bitrah.exception.BitrahException;
import ir.kavoshgaran.bitrah.exception.BitrahUnAuthorizedException;
import ir.kavoshgaran.bitrah.responses.BitrahStatusResponse;
import ir.kavoshgaran.bitrah.responses.BitrahSubmitOrderResponse;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import reactor.core.publisher.Mono;

import java.io.IOException;
import java.util.Map;

public class BitrahClient {

    private static final String BASE_URL = "http://bitrah.ir/api/v1";
    private String merchantId;
    private String callBackUrl;
    private String webHookUrl;
    private String username;
    private String password;
    private String token;
    private AcceptedLanguage acceptedLanguage;
    private final String AUTHORIZATION_KEY = "Authentication";


    public BitrahClient(String merchantId
            , String callBackUrl
            , String webHookUrl
            , AcceptedLanguage acceptedLanguage
            , String username
            , String password) {
        this.merchantId = merchantId;
        this.callBackUrl = callBackUrl;
        this.webHookUrl = webHookUrl;
        this.acceptedLanguage = acceptedLanguage;
        this.username = username;
        this.password = password;
    }


    public void register() {
        if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
            throw new BitrahException("UserName or password is empty.", HttpStatus.UNAUTHORIZED.value());
        }
        String loginUrl = BASE_URL + "/authentication/login";

        AuthenticateModel authenticateModel = new AuthenticateModel(username, password);
        try {
            Map<Object, Object> loginResponse = WebClient.create().post()
                    .uri(loginUrl).header("Accepted-Language", this.acceptedLanguage.getLanguage())
                    .contentType(MediaType.APPLICATION_JSON)
                    .body(BodyInserters.fromObject(authenticateModel))
                    .retrieve().bodyToMono(Map.class).block();

            Object data = loginResponse.get("data");
            if (data != null && data instanceof Map) {
                Map<String, String> tokenMap = (Map<String, String>) data;
                String token = tokenMap.get("token");
                if (token != null) {
                    this.token = token;
                }
            }

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        }
    }


    public BitrahSubmitOrderResponse submitOrder(String orderId, Long rialValue, Coin coin) {

        if (StringUtils.isEmpty(token)) {
            register();
        }
        String submitOrderUrl = BASE_URL + "/order/submit/wr";
        Order order = new Order();
        order.setMerchantId(this.merchantId);
        order.setOrderId(orderId);
        order.setRialValue(rialValue);
        order.setCallbackUrl(this.callBackUrl);
        order.setWebhookUrl(this.webHookUrl);
        if (coin != null) {
            order.setCoin(coin.getIso());
        }

        BitrahSubmitOrderResponse response = null;
        try {
            Mono<BitrahSubmitOrderResponse> bodyMono = WebClient.create().post()
                    .uri(submitOrderUrl).header("Accepted-Language", this.acceptedLanguage.getLanguage())
                    .contentType(MediaType.APPLICATION_JSON)
                    .header(AUTHORIZATION_KEY, "bearer " + token)
                    .body(BodyInserters.fromObject(order))
                    .retrieve().bodyToMono(BitrahSubmitOrderResponse.class);
            response = bodyMono.block();

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        } catch (BitrahUnAuthorizedException authorizedException) {
            register();
        }

        return response;
    }

    public BitrahStatusResponse orderStatus(Long refId) {

        if (StringUtils.isEmpty(token)) {
            register();
        }

        String submitOrderUrl = BASE_URL + "/order/status/rd";
        OrdersRequest ordersRequest = new OrdersRequest();
        ordersRequest.setMerchantId(this.merchantId);
        ordersRequest.setRefId(refId);

        BitrahStatusResponse response = null;
        try {
            Mono<BitrahStatusResponse> bodyMono = WebClient.create().post()
                    .uri(submitOrderUrl).header("Accept-Language", this.acceptedLanguage.getLanguage())
                    .header(AUTHORIZATION_KEY, "bearer " + token)
                    .contentType(MediaType.APPLICATION_JSON)
                    .body(BodyInserters.fromObject(ordersRequest))
                    .retrieve().bodyToMono(BitrahStatusResponse.class);
            response = bodyMono.block();

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        } catch (BitrahUnAuthorizedException authorizedException) {
            register();
        }

        return response;
    }

    private void handleBitrahResponseException(WebClientResponseException e) {
        ObjectMapper mapper = new ObjectMapper();
        try {
            BitrahErrorResponse bitrahError = mapper.readValue(e.getResponseBodyAsByteArray(), BitrahErrorResponse.class);
            throw new BitrahException(bitrahError.getData().getMessage(), bitrahError.getData().getCode());
        } catch (IOException ioException) {
            e.printStackTrace();
        }
    }
}
