package ir.kavoshgaran.bitrah.services;

import com.fasterxml.jackson.databind.ObjectMapper;
import ir.kavoshgaran.bitrah.entities.*;
import ir.kavoshgaran.bitrah.exception.BitrahErrorResponse;
import ir.kavoshgaran.bitrah.exception.BitrahException;
import ir.kavoshgaran.bitrah.exception.BitrahUnAuthorizedException;
import ir.kavoshgaran.bitrah.responses.BitrahLoginResponse;
import ir.kavoshgaran.bitrah.responses.BitrahStatusResponse;
import ir.kavoshgaran.bitrah.responses.BitrahSubmitOrderResponse;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import reactor.core.publisher.Mono;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;

public class BitrahClient {

    private static final String BASE_URL = "https://api.bitrah.ir/api/v1";
    private String merchantId;
    private String callBackUrl;
    private String webHookUrl;
    private String username;
    private String password;
    private String token;
    private AcceptedLanguage acceptedLanguage;
    private final String AUTHORIZATION_KEY = "Authentication";
    private String expiredTime;


    public BitrahClient(String merchantId
            , String callBackUrl
            , String webHookUrl
            , AcceptedLanguage acceptedLanguage
            , String username
            , String password) {
        this.merchantId = merchantId;
        this.callBackUrl = callBackUrl;
        this.webHookUrl = webHookUrl;
        this.acceptedLanguage = acceptedLanguage;
        this.username = username;
        this.password = password;
    }


    public BitrahSubmitOrderResponse submitOrder(String orderId, Long rialValue, Coin coin) {

        tokenValidation();

        String submitOrderUrl = BASE_URL + "/order/submit/wr";
        Order order = new Order();
        order.setMerchantId(this.merchantId);
        order.setOrderId(orderId);
        order.setRialValue(rialValue);
        order.setCallbackUrl(this.callBackUrl);
        order.setWebhookUrl(this.webHookUrl);
        if (coin != null) {
            order.setCoin(coin.getIso());
        }

        BitrahSubmitOrderResponse response = null;
        try {
            Mono<BitrahSubmitOrderResponse> bodyMono = WebClient.create().post()
                    .uri(submitOrderUrl).header("Accepted-Language", this.acceptedLanguage.getLanguage())
                    .contentType(MediaType.APPLICATION_JSON)
                    .header(AUTHORIZATION_KEY, "bearer " + token)
                    .body(BodyInserters.fromObject(order))
                    .retrieve().bodyToMono(BitrahSubmitOrderResponse.class);
            response = bodyMono.block();

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        }

        return response;
    }

    public BitrahStatusResponse orderStatus(Long refId) {

        tokenValidation();

        String submitOrderUrl = BASE_URL + "/order/status/rd";
        OrdersRequest ordersRequest = new OrdersRequest();
        ordersRequest.setMerchantId(this.merchantId);
        ordersRequest.setRefId(refId);

        BitrahStatusResponse response = null;
        try {
            Mono<BitrahStatusResponse> bodyMono = WebClient.create().post()
                    .uri(submitOrderUrl).header("Accept-Language", this.acceptedLanguage.getLanguage())
                    .header(AUTHORIZATION_KEY, "bearer " + token)
                    .contentType(MediaType.APPLICATION_JSON)
                    .body(BodyInserters.fromObject(ordersRequest))
                    .retrieve().bodyToMono(BitrahStatusResponse.class);
            response = bodyMono.block();

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        }

        return response;
    }

    private void tokenValidation() {
        if (StringUtils.isEmpty(token)) {
            loginMerchant();

        } else if (expiredTime != null) {
            LocalDateTime expireDateTime = LocalDateTime.parse(expiredTime, DateTimeFormatter.ofPattern("yyyy/M/dd HH:mm:ss"));
            if (expireDateTime.until(LocalDateTime.now(), ChronoUnit.MINUTES) >= 0) {
                loginMerchant();
            }
        }
        if (token == null){
            throw new BitrahUnAuthorizedException("Jwt token is null");
        }
    }

    public void loginMerchant() {
        if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
            throw new BitrahException("UserName or password is empty.", HttpStatus.UNAUTHORIZED.value());
        }
        String loginUrl = BASE_URL + "/authentication/login";

        AuthenticateModel authenticateModel = new AuthenticateModel(username, password);
        try {
            BitrahLoginResponse loginResponse = WebClient.create().post()
                    .uri(loginUrl).header("Accepted-Language", this.acceptedLanguage.getLanguage())
                    .contentType(MediaType.APPLICATION_JSON)
                    .body(BodyInserters.fromObject(authenticateModel))
                    .retrieve().bodyToMono(BitrahLoginResponse.class).block();

            if (loginResponse != null && Boolean.TRUE.equals(loginResponse.getSuccess())) {
                this.token = loginResponse.getData().getToken();
                this.expiredTime = loginResponse.getData().getExpireDate();
            }

        } catch (WebClientResponseException e) {
            handleBitrahResponseException(e);
        }
    }

    private void handleBitrahResponseException(WebClientResponseException e) {
        ObjectMapper mapper = new ObjectMapper();
        try {
            BitrahErrorResponse bitrahError = mapper.readValue(e.getResponseBodyAsByteArray(), BitrahErrorResponse.class);
            throw new BitrahException(bitrahError.getData().getMessage(), bitrahError.getData().getCode());
        } catch (IOException ioException) {
            e.printStackTrace();
        }
    }
}
