package ir.dorantech.stepcounter

import android.content.Context
import android.content.Intent
import androidx.annotation.DrawableRes
import ir.dorantech.stepcounter.countermodule.R
import ir.dorantech.stepcounter.domain.usecase.GetTotalStepsUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveFirstForegroundNotificationIconUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveFirstForegroundNotificationMessageUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveFirstForegroundNotificationTitleUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveSecondForegroundNotificationIconUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveSecondForegroundNotificationMessageUseCase
import ir.dorantech.stepcounter.domain.usecase.SaveSecondForegroundNotificationTitleUseCase
import ir.dorantech.stepcounter.domain.usecase.SetTotalStepsUseCase
import ir.dorantech.stepcounter.foundation.tools.LogManager
import ir.dorantech.stepcounter.service.ActivityMonitorService
import ir.dorantech.stepcounter.worker.ServiceMonitorScheduler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.launch
import org.kodein.di.DI
import org.kodein.di.instance

class StepCounterServiceApi internal constructor(private val context: Context, appDi: DI) {
    private val getTotalStepsUseCase: GetTotalStepsUseCase by appDi.instance()
    private val setTotalStepsUseCase: SetTotalStepsUseCase by appDi.instance()
    private val logManager: LogManager by appDi.instance()
    private val saveFirstForegroundNotificationIconUseCase:
            SaveFirstForegroundNotificationIconUseCase by appDi.instance()
    private val saveFirstForegroundNotificationTitleUseCase:
            SaveFirstForegroundNotificationTitleUseCase by appDi.instance()
    private val saveFirstForegroundNotificationMessageUseCase:
            SaveFirstForegroundNotificationMessageUseCase by appDi.instance()
    private val saveSecondForegroundNotificationIconUseCase:
            SaveSecondForegroundNotificationIconUseCase by appDi.instance()
    private val saveSecondForegroundNotificationTitleUseCase:
            SaveSecondForegroundNotificationTitleUseCase by appDi.instance()
    private val saveSecondForegroundNotificationMessageUseCase:
            SaveSecondForegroundNotificationMessageUseCase by appDi.instance()


    /**
     * StartService by the application
     *
     * @param serviceStartNotificationTitle Notification title for first foreground service
     * @param serviceStartNotificationMessage Notification message for first foreground service
     * @param serviceCounterNotificationTitle Notification title for foreground service when counting steps
     * @param serviceCounterNotificationMessage Notification message for foreground service when counting steps
     * @param startNotificationIcon Notification icon for first foreground service
     * @param countingNotificationIcon Notification icon for foreground service when counting steps
     */

    fun startService(
        serviceStartNotificationTitle: String? = null,
        serviceStartNotificationMessage: String? = null,
        serviceCounterNotificationTitle: String? = null,
        serviceCounterNotificationMessage: String? = null,
        @DrawableRes startNotificationIcon: Int? = null,
        @DrawableRes countingNotificationIcon: Int? = null
    ) {
        logManager.addLog("startService is called")
        saveInputs(
            serviceStartNotificationTitle = serviceStartNotificationTitle,
            serviceStartNotificationMessage = serviceStartNotificationMessage,
            serviceCounterNotificationTitle = serviceCounterNotificationTitle,
            serviceCounterNotificationMessage = serviceCounterNotificationMessage,
            startNotificationIcon = startNotificationIcon,
            countingNotificationIcon = countingNotificationIcon
        )
        ServiceMonitorScheduler.scheduleWorker(context.applicationContext)
    }

    fun stopService() {
        val serviceIntent = Intent(context, ActivityMonitorService::class.java)
        context.stopService(serviceIntent)
        ServiceMonitorScheduler.cancelScheduledWork(context)
    }

    fun readStepsCount(): Flow<Long> {
        return getTotalStepsUseCase()
    }

    suspend fun resetCounter() {
        setTotalStepsUseCase(0)
    }

    internal fun saveInputs(
        serviceStartNotificationTitle: String?,
        serviceStartNotificationMessage: String?,
        serviceCounterNotificationTitle: String?,
        serviceCounterNotificationMessage: String?,
        @DrawableRes startNotificationIcon: Int? = null,
        @DrawableRes countingNotificationIcon: Int? = null
    ) {
        CoroutineScope(Dispatchers.IO).launch {
            saveFirstForegroundNotificationTitleUseCase(
                serviceStartNotificationTitle ?: "App Is Run"
            )
            saveFirstForegroundNotificationMessageUseCase(
                serviceStartNotificationMessage ?: "Application Service Started"
            )
            saveSecondForegroundNotificationTitleUseCase(
                serviceCounterNotificationTitle ?: "Counting Steps"
            )
            saveSecondForegroundNotificationMessageUseCase(
                serviceCounterNotificationMessage ?: "Steps Are Counting"
            )
            saveFirstForegroundNotificationIconUseCase(
                startNotificationIcon ?: R.drawable.notifications
            )
            saveSecondForegroundNotificationIconUseCase(
                countingNotificationIcon ?: R.drawable.directions_run
            )
        }

    }
}